/**
 * Copyright (c) 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 *
 */

/**
 * @classdesc
 * A TimeUnit represents time durations at a given unit of granularity.  TimeUnit is mainly used to
 * inform time-based functions how a given timing parameter should be interpreted.
 *
 * @class
 * @memberof iotcs
 * @alias iotcs.TimeUnit
 */
class TimeUnit {
   // Instance "variables" & properties...see constructor.

   /**
    * @ignore
    */
   constructor(name, displayName = name) {
      if (TimeUnit.INSTANCES.has(name)) {
         throw new Error('duplicate TimeUnit name: ' + name);
      }

      if (!TimeUnit.canCreateMoreInstances) {
         throw new Error('TimeUnit (' + name + ', ' + displayName + ') already created.');
      }

      this.name = name;
      this.displayName = displayName;
      Object.freeze(this);
      TimeUnit.INSTANCES.set(this.code, this);
   }

   toDays(duration) {
      if (this.name == TimeUnit.NANOSECONDS) {
         return Math.floor(duration / (60 * 60 * 24 * 1000 * 1000 * 1000));
      } else if (this.name == TimeUnit.MICROSECONDS) {
         return Math.floor(duration / (60 * 60 * 24 * 1000 * 1000));
      } else if (this.name == TimeUnit.MILLISECONDS) {
         return Math.floor(duration / (60 * 60 * 24 * 1000));
      } else if (this.name == TimeUnit.SECONDS) {
         return Math.floor(duration / (60 * 60 * 24));
      } else if (this.name == TimeUnit.MINUTES) {
         return Math.floor(duration / (60 * 24));
      } else if (this.name == TimeUnit.HOURS) {
         return duration * 24;
      } else if (this.name == TimeUnit.DAYS) {
         return duration;
      }
   }

   toHours(duration) {
      if (this.name == TimeUnit.NANOSECONDS) {
         return Math.floor(duration / (60 * 60 * 1000 * 1000 * 1000));
      } else if (this.name == TimeUnit.MICROSECONDS) {
         return Math.floor(duration / (60 * 60 * 1000 * 1000));
      } else if (this.name == TimeUnit.MILLISECONDS) {
         return Math.floor(duration / (60 * 60 * 1000));
      } else if (this.name == TimeUnit.SECONDS) {
         return Math.floor(duration / (60 * 60));
      } else if (this.name == TimeUnit.MINUTES) {
         return Math.floor(duration / 60);
      } else if (this.name == TimeUnit.HOURS) {
         return duration;
      } else if (this.name == TimeUnit.DAYS) {
         return duration * 24;
      }
   }

   toMicros(duration) {
      if (this.name == TimeUnit.NANOSECONDS) {
         return Math.floor(duration / 1000);
      } else if (this.name == TimeUnit.MICROSECONDS) {
         return duration;
      } else if (this.name == TimeUnit.MILLISECONDS) {
         return duration * 1000;
      } else if (this.name == TimeUnit.SECONDS) {
         return duration * 60 * 1000;
      } else if (this.name == TimeUnit.MINUTES) {
         return duration * 60 * 60 * 1000;
      } else if (this.name == TimeUnit.HOURS) {
         return duration * 60 * 60 * 60 * 1000;
      } else if (this.name == TimeUnit.DAYS) {
         return duration * 60 * 60 * 24 * 1000 * 1000;
      }
   }

   toMillis(duration) {
      if (this.name == TimeUnit.NANOSECONDS) {
         return Math.floor(duration / (1000 * 1000));
      } else if (this.name == TimeUnit.MICROSECONDS) {
         return Math.floor(duration / 1000);
      } else if (this.name == TimeUnit.MILLISECONDS) {
         return duration;
      } else if (this.name == TimeUnit.SECONDS) {
         return duration * 1000;
      } else if (this.name == TimeUnit.MINUTES) {
         return duration * 60 * 1000;
      } else if (this.name == TimeUnit.HOURS) {
         return duration * 60 * 60 * 1000;
      } else if (this.name == TimeUnit.DAYS) {
         return duration * 24;
      }
   }

   toMinutes(duration) {
      if (this.name == TimeUnit.NANOSECONDS) {
         return Math.floor(duration / (60 * 1000 * 1000 * 1000));
      } else if (this.name == TimeUnit.MICROSECONDS) {
         return Math.floor(duration / (60 * 1000 * 1000));
      } else if (this.name == TimeUnit.MILLISECONDS) {
         return Math.floor(duration / (60 * 1000));
      } else if (this.name == TimeUnit.SECONDS) {
         return Math.floor(duration / 60);
      } else if (this.name == TimeUnit.MINUTES) {
         return duration;
      } else if (this.name == TimeUnit.HOURS) {
         return duration * 60;
      } else if (this.name == TimeUnit.DAYS) {
         return duration * 60 * 24;
      }
   }

   toNanos(duration) {
      if (this.name == TimeUnit.NANOSECONDS) {
         return duration;
      } else if (this.name == TimeUnit.MICROSECONDS) {
         return duration * 1000;
      } else if (this.name == TimeUnit.MILLISECONDS) {
         return duration * 1000 * 1000;
      } else if (this.name == TimeUnit.SECONDS) {
         return duration * 60 * 1000 * 1000;
      } else if (this.name == TimeUnit.MINUTES) {
         return duration * 60 * 60 * 1000 * 1000;
      } else if (this.name == TimeUnit.HOURS) {
         return duration * 60 * 60 * 60 * 1000 * 1000;
      } else if (this.name == TimeUnit.DAYS) {
         return duration * 60 * 60 * 24 * 1000 * 1000;
      }
   }

   toSeconds(duration) {
      if (this.name == TimeUnit.NANOSECONDS) {
         return Math.floor(duration / (1000 * 1000 * 1000));
      } else if (this.name == TimeUnit.MICROSECONDS) {
         return Math.floor(duration / (1000 * 1000));
      } else if (this.name == TimeUnit.MILLISECONDS) {
         return Math.floor(duration / 1000);
      } else if (this.name == TimeUnit.SECONDS) {
         return duration;
      } else if (this.name == TimeUnit.MINUTES) {
         return duration * 60;
      } else if (this.name == TimeUnit.HOURS) {
         return duration * 60 * 60;
      } else if (this.name == TimeUnit.DAYS) {
         return duration * 60 * 60 * 24;
      }
   }

   toString() {
      return this.name;
   }

   static values() {
      return Array.from(TimeUnit.INSTANCES.values());
   }

   static fromName(name) {
      if (!TimeUnit.INSTANCES.has(name))
         throw new Error('Unknown TimeUnit: ' + name);
      else
         return TimeUnit.INSTANCES.get(name);
   }
}

lib.TimeUnit = TimeUnit;
TimeUnit.canCreateMoreInstances = true;
TimeUnit.INSTANCES = new Map();

/**
 * DAYS
 *
 * @type {TimeUnit}
 */
TimeUnit.DAYS = new TimeUnit('DAYS', 'DAYS');
/**
 * HOURS
 *
 * @type {TimeUnit}
 */
TimeUnit.HOURS = new TimeUnit('HOURS', 'HOURS');
/**
 * MICROSECONDS
 *
 * @type {TimeUnit}
 */
TimeUnit.MICROSECONDS = new TimeUnit('MICROSECONDS');
/**
 * MILLISECONDS
 *
 * @type {TimeUnit}
 */
TimeUnit.MILLISECONDS = new TimeUnit('MILLISECONDS', 'MILLISECONDS');
/**
 * MINUTES
 *
 * @type {TimeUnit}
 */
TimeUnit.MINUTES = new TimeUnit('MINUTES');
/**
 * NANOSECONDS
 *
 * @type {TimeUnit}
 */
TimeUnit.NANOSECONDS = new TimeUnit('NANOSECONDS');
/**
 * SECONDS
 *
 * @type {TimeUnit}
 */
TimeUnit.SECONDS = new TimeUnit('SECONDS');

TimeUnit.canCreateMoreInstances = false;
Object.freeze(TimeUnit);
