/**
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 *
 */

/**
 * This class contains functions for the browser port for the device.
 */

/** @ignore */
var $port = lib.$port || {};

if (lib.debug) {
    lib.$port = $port;
}

if (typeof window === 'undefined') {
    lib.error('invalid target platform');
}

let _b2h = (function () {
    let r = [];

    for (let i=0; i<256; i++) {
        r[i] = (i + 0x100).toString(16).substr(1);
    }

    return r;
})();

$port.HttpStatusCode = {
    OK: 200,
    NOT_FOUND: 404
};

let _authWindow = null;


////////////////////////////////////////////////////////////////////////////////////////////////////
// miscellaneous

$port.userAuthNeeded = function () {
    return true;
};
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
// file
$port.file = {};

$port.file.append = function (path, data) {
    var original_data = localStorage.getItem(path);

    if (!original_data) {
        lib.error('could not load file "'+path+'"');
        return;
    }

    localStorage.setItem(path, original_data + data);
};

/**
 * Returns {@code true} if the specified file exists in local storage.
 *
 * @param {string} pathFileName the path and name of the file to check if exists.
 * @returns {boolean} {@code true} if the specified file exists in local storage.
 */
$port.file.exists = function (pathFileName) {
    return (localStorage.getItem(pathFileName) !== null);
};

/**
 * Opens the file with the given path/filename and returns it's contents.
 *
 * @param {string} pathFileName the path and name of the file to load.
 * @returns {string} the contents of the file.
 */
$port.file.load = function (pathFileName) {
    let fileContents = localStorage.getItem(pathFileName);

    if (!fileContents) {
        lib.error('Could not load file: "' + pathFileName + '"');
        return;
    }

    return fileContents;
};

/**
 * Removes the file with the given path/filename from local storage.
 *
 * @param {string} pathFileName the path and name of the file to remove.
 */
$port.file.remove = function (pathFileName) {
    localStorage.removeItem(pathFileName);
};

/**
 * Stores the file with the given path/filename in local storage with the data.
 *
 * @param {string} pathFileName the path and name of the file to remove.
 * @param {string} data the data to store in the file.
 */
$port.file.store = function (pathFileName, data) {
    localStorage.setItem(pathFileName, data);
};
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
// https
$port.https = {};

$port.https.csrf = {};
$port.https.csrf.token = null;
$port.https.csrf.inProgress = false;
$port.https.csrf.tokenName = 'X-CSRF-TOKEN';

$port.https.authRequest = {};
$port.https.authRequest.path = '/iot/webapi/v2/private/server';

$port.https.req = function (options, payload, callback) {
    // If the device has been activated we send the ActivationID as the ClientID.  If the device
    // has not been activated, we send the DeviceID as the ClientID.
    if (options.tam &&
        (typeof options.tam.getTrustAnchorCertificates === 'function') &&
        Array.isArray(options.tam.getTrustAnchorCertificates()) &&
        (options.tam.getTrustAnchorCertificates().length > 0))
    {
        options.ca = options.tam.getTrustAnchorCertificates();
    }

    options.rejectUnauthorized = true;
    options.agent = false;

    if ((options.method === 'GET') && (payload)) {
        lib.log('There should be no payload when using GET method; use "path" for passing query.');
    }

    // // If this is the first attempt to access IoT-CS...
    // if ((oracleIoT) && (!$port.https.csrf.token) && (!$port.https.csrf.inProgress)) {
    //     $port.https.csrf.inProgress = true;
    //     $port.https.getTokenAndRequest(options, payload, 1, callback);
    // } else {
    //     $port.https.request(options, payload, callback, oracleIoT);
    // }

    $port.https.request(options, payload, callback, true);
};

/**
 * This function performs the HTTP request to the IoT CS.
 *
 * @param options The HTTPS options.
 * @param payload The payload to send if any is to be sent.
 * @param callback The callback with the results or error.
 * @param oracleIoT
 */
$port.https.request = function (options, payload, callback, oracleIoT) {
    let baseUrl = (options.protocol || 'https') +
        '://' +
        (options.hostname || options.host || 'localhost') +
        (((options.port) && ((options.protocol === 'https' && options.port !== 443)) ||
        (options.protocol === 'http' && options.port !== 80)) ? (':' + options.port) : '');

    let url = baseUrl + (options.path || '/');
    let authUrl = baseUrl + $port.https.authRequest.path;

    if (options.tam
        && (typeof options.tam.getTrustAnchorCertificates === 'function')
        && Array.isArray(options.tam.getTrustAnchorCertificates())
        && (options.tam.getTrustAnchorCertificates().length > 0))
    {
        options.ca = options.tam.getTrustAnchorCertificates();
    }

    options.rejectUnauthorized = true;
    options.protocol = options.protocol + ':';
    options.agent = false;

    let _onNotAuth = function () {
        callback(null, new Error('{"statusCode": 401, "statusMessage": "Unauthorized", "body": ""}'));
    };

    let xhr = new XMLHttpRequest();

    /**
     * Function which is called when xhr events occur.
     *
     * @param req the request.
     */
    let _onready = function (req) {
        // req.readyState === 4 indicates the request is done.
        if (req.readyState === 4) {
            if ((req.status === 302) || (req.status === 0) ||
               (req.responseUrl && req.responseUrl.length &&
                (decodeURI(req.responseURL) !== url)))
            {
                _onNotAuth();
                return;
            }

            if ((req.status === 401) || (req.status === 403)) {
                _onNotAuth();
                return;
            }

            if ((req.status === 200) || (req.status === 202)) {
                if (xhr.getResponseHeader($port.https.csrf.tokenName) &&
                xhr.getResponseHeader($port.https.csrf.tokenName).length)
               {
                    $port.https.csrf.token = xhr.getResponseHeader($port.https.csrf.tokenName);
                }

                callback(req.responseText);
            } else {
                callback(null, lib.createError(req.responseText));
            }
        }
    };

    xhr.open(options.method, url, true);

    if (oracleIoT) {
        xhr.withCredentials = true;

        if ($port.https.csrf.token) {
            xhr.setRequestHeader($port.https.csrf.tokenName, $port.https.csrf.token);
        }
    }

    xhr.onreadystatechange = function () {
        _onready(xhr);
    };

    if (options.headers) {
        Object.keys(options.headers).forEach(function (key, index) {
            if ((!oracleIoT) && (key === 'Authorization') && (options.auth)) {
                xhr.setRequestHeader(key, options.auth);
            } else {
                xhr.setRequestHeader(key, options.headers[key]);
            }
        });
    }

    xhr.send(payload || null);
};
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
// os

$port.os = {};

$port.os.type = function () {
    return window.navigator.platform;
};

$port.os.release = function () {
    return '0';
};
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
// util

$port.util = {};

$port.util.rng = function (count) {
    var a = new Array(count);
    for (var i=0; i<count; i++) {
        a[i] = Math.floor(Math.random()*256);
    }
    return a;
};

$port.util.uuidv4 = function () {
    var r16 = $port.util.rng(16);
    r16[6]  &= 0x0f;  // clear version
    r16[6]  |= 0x40;  // set to version 4
    r16[8]  &= 0x3f;  // clear variant
    r16[8]  |= 0x80;  // set to IETF variant
    var i = 0;
    return _b2h[r16[i++]] + _b2h[r16[i++]] + _b2h[r16[i++]] + _b2h[r16[i++]] + '-' +
        _b2h[r16[i++]] + _b2h[r16[i++]] + '-' +
        _b2h[r16[i++]] + _b2h[r16[i++]] + '-' +
        _b2h[r16[i++]] + _b2h[r16[i++]] + '-' +
        _b2h[r16[i++]] + _b2h[r16[i++]] + _b2h[r16[i++]] +
        _b2h[r16[i++]] + _b2h[r16[i++]] + _b2h[r16[i]];
};

$port.util.btoa = function (str) {
    return btoa(str);
};

$port.util.atob = function (str) {
    return atob(str);
};

$port.util.query = {};

$port.util.query.escape = function (str) {
    return escape(str);
};

$port.util.query.unescape = function (str) {
    return unescape(str);
};

$port.util.query.parse = function (str, sep, eq, options) {
    var _sep = sep || '&';
    var _eq  = eq  || '=';
    var decodeURIComponent = $port.util.query.unescape;
    var obj = {};
    var args = str.split(_sep);
    for (var i=0; i < args.length; i++) {
        var pair = args[i].split(_eq);
        var field = decodeURIComponent(pair[0]);
        var value = decodeURIComponent(pair[1]);
        if (obj[field]) {
            if (!Array.isArray(obj[field])) {
                var current = obj[field];
                obj[field] = new Array(current);
            }
            obj[field].push(value);
        } else {
            obj[field] = value;
        }
    }
    return obj;
};

$port.util.query.stringify = function (obj, sep, eq, options) {
    var _sep = sep || '&';
    var _eq  = eq  || '=';
    var encodeURIComponent = $port.util.query.escape;
    var str = '';
    Object.keys(obj).forEach(function (key) {
        if (typeof obj[key] === 'object') {
            obj[key].forEach(function (e) {
                str += _sep + key + _eq + encodeURIComponent(e);
            });
        } else {
            str += _sep + key + _eq + encodeURIComponent(obj[key]);
        }
    });
    return str.substring(1);
};

/*@TODO: check that Promise are actually supported! either try/catch or if (!Promise) else lib.error ...
*/
$port.util.promise = function(executor){
    return new Promise(executor);
};

////////////////////////////////////////////////////////////////////////////////////////////////////
