/*
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl.device;

import com.oracle.iot.client.device.DirectlyConnectedDevice;
import com.oracle.iot.client.device.util.MessageDispatcher;
import com.oracle.iot.client.device.persistence.MessagePersistence;
import com.oracle.iot.client.device.util.RequestDispatcher;
import com.oracle.iot.client.device.util.RequestHandler;
import com.oracle.iot.client.impl.DiagnosticsImpl;
import com.oracle.iot.client.impl.TestConnectivity;
import com.oracle.iot.client.message.Message;
import com.oracle.iot.client.message.RequestMessage;
import com.oracle.iot.client.message.Resource;
import com.oracle.iot.client.message.ResourceMessage;
import com.oracle.iot.client.message.ResponseMessage;
import com.oracle.iot.client.message.StatusCode;
import oracle.iot.client.StorageObject.SyncStatus;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * MessageDispatcherImpl
 */
public final class MessageDispatcherImpl extends MessageDispatcher {

    private static final int DEFAULT_MAXIMUM_MESSAGES_TO_QUEUE = 10000;
    private static final int DEFAULT_MAXIMUM_MESSAGES_PER_CONNECTION = 100;
    private static final long DEFAULT_POLLING_INTERVAL = 3000; // milliseconds
    private static final int DEFAULT_BASIC_NUMBER_OF_RETRIES = 3;
    private static final long DEFAULT_SETTLE_TIME = 10000; // milliseconds
    private static final int REQUEST_DISPATCHER_THREAD_POOL_SIZE =
                 Math.max( Integer.getInteger("oracle.iot.client.device.request_dispatcher_thread_pool_size", 1),1);
    static final String MAXIMUM_MESSAGES_TO_QUEUE_PROPERTY =
        "oracle.iot.client.device.dispatcher_max_queue_size";
    private static final String MAXIMUM_MESSAGES_PER_CONNECTION_PROPERTY = 
        "oracle.iot.client.device.dispatcher_max_messages_per_connection";
    private static final String POLLING_INTERVAL_PROPERTY =
        "oracle.iot.client.device.dispatcher_polling_interval";
    private static final String BASIC_NUMBER_OF_RETRIES_PROPERTY = 
        "oracle.iot.client.device.dispatcher_basic_number_of_retries";
    private static final String DISABLE_LONG_POLLING_PROPERTY =
        "com.oracle.iot.client.disable_long_polling";
    private static final String SETTLE_TIME_PROPERTY =
            "oracle.iot.client.device.dispatcher_settle_time";

    // This flag is used by receiver thread to wait for reconnection
    // when SocketException exception happens due to network disruptions.
    private AtomicBoolean waitOnReconnect = new AtomicBoolean(false);

    static final String COUNTERS_URL;
    static final String RESET_URL;
    static final String POLLING_INTERVAL_URL;
    static final String DIAGNOSTICS_URL;
    static final String TEST_CONNECTIVITY_URL;

    static {
        COUNTERS_URL = "deviceModels/" + ActivationManager.MESSAGE_DISPATCHER_URN + "/counters";
        RESET_URL = "deviceModels/" + ActivationManager.MESSAGE_DISPATCHER_URN + "/reset";
        POLLING_INTERVAL_URL = "deviceModels/" + ActivationManager.MESSAGE_DISPATCHER_URN + "/pollingInterval";
        DIAGNOSTICS_URL = "deviceModels/" + ActivationManager.DIAGNOSTICS_URN + "/info";
        TEST_CONNECTIVITY_URL = "deviceModels/" + ActivationManager.DIAGNOSTICS_URN + "/testConnectivity";
    }

    private RequestHandler counterHandler;
    private final RequestHandler resetHandler;

    private final RequestHandler pollingIntervalHandler;
    private RequestHandler diagnosticsHandler;
    private RequestHandler testConnectivityHandler;

    final HashMap<StorageObjectImpl, HashSet<String>> contentMap;
    final HashSet<String> failedContentIds;

    /*
     * Queue of outgoing messages to be dispatched to server
     * (with thread for servicing the queue).
     * Note: it has package access for unit test access.
     */
    final PriorityQueue<Message> outgoingMessageQueue;

    /*
     * maximum number of messages allowed in the outgoing message queue
     */
    private final int maximumQueueSize;

    /*
     * maximum number of messages to send in one connection
     */
    private final int maximumMessagesPerConnection;


    // Counter indicating total number of messages that have been delivered
    private int totalMessagesSent;

    // Counter indicating total number of messages that have been delivered
    private int totalMessagesReceived;

    // Counter indicating total number of messages that were retried for delivery
    private int totalMessagesRetried;

    // Counter indicating total number of bytes that have been delivered
    private long totalBytesSent;

    // Counter indicating total number of bytes that have been delivered
    private long totalBytesReceived;

    // Counter indicating total number of protocol errors.
    private long totalProtocolErrors;

    /*
     * Maximum time in milliseconds to wait for a message to be queued
     * before the send thread will poll the server
     */
    private long pollingInterval;

    private final Lock sendLock = new ReentrantLock();
    private final Condition messageQueued = sendLock.newCondition();

    private final Lock contentLock = new ReentrantLock();

    private final Lock receiveLock = new ReentrantLock();
    private final Condition messageSent = receiveLock.newCondition();

    /*
     * Thread for sending messages.
     */
    private final Thread transmitThread;

    /*
     * Thread for receiving requests when long polling.
     */
    private final Thread receiveThread;

    private final DirectlyConnectedDevice deviceClient;

    /*
     * Set to true if close method has been called.
     */
    private boolean closed;
    private volatile boolean requestClose;

    private MessageDispatcher.DeliveryCallback deliveryCallback;
    private MessageDispatcher.ErrorCallback errorCallback;
    private final boolean useLongPolling;

    @Override
    public void setOnDelivery(MessageDispatcher.DeliveryCallback deliveryCallback) {
        this.deliveryCallback = deliveryCallback;
    }

    @Override
    public void setOnError(MessageDispatcher.ErrorCallback errorCallback) {
        this.errorCallback = errorCallback;
    }

    @Override
    public RequestDispatcher getRequestDispatcher() {
        return RequestDispatcher.getInstance();
    }

    /**
     * Implmentation of MessageDispatcher.
     * @param deviceClient The DirectlyConnectedDevice that uses this message dispatcher.
     *
     */
    public MessageDispatcherImpl(DirectlyConnectedDevice deviceClient) {

        this.requestClose = false;
        this.closed = false;

        this.deviceClient = deviceClient;
        this.pollingInterval = getPollingInterval();
        this.maximumQueueSize = getQueueSize();
        this.maximumMessagesPerConnection = getMaximumMessagesPerConnection();

        final String endpointId = MessageDispatcherImpl.this.deviceClient.getEndpointId();

        this.useLongPolling = !Boolean.getBoolean(DISABLE_LONG_POLLING_PROPERTY);
        
        this.contentMap = new HashMap<StorageObjectImpl, HashSet<String>>();
        this.failedContentIds = new HashSet<String>();
        
        this.outgoingMessageQueue = new PriorityQueue<Message>(
                getQueueSize(),
                new Comparator<Message>() {
                    @Override
                    public int compare(Message o1, Message o2) {

                        // Note this implementation is not consistent with equals. It is possible
                        // that a.compareTo(b) == 0 is not that same boolean value as a.equals(b)

                        // The natural order of enum is the enum's ordinal, i.e.,
                        // x.getPriority().compareTo(y.getPriority() will give {x,y}
                        // if x is a lower priority. What we want is to sort by the
                        // higher priority.
                        int c = o2.getPriority().compareTo(o1.getPriority());

                        // If they are the same priority, take the one that was created first
                        if (c == 0) {
                            c = o1.getEventTime().compareTo(o2.getEventTime());
                        }

                        // If they are still the same, take the one with higher reliability.
                        if (c == 0) {
                            c = o1.getReliability().compareTo(o2.getReliability());
                        }

                        // If they are still the same, take the one that was created first.
                        if (c == 0) {
                            long lc = o1.getOrdinal() - o2.getOrdinal();
                            if (lc > 0) c = 1;
                            else if (lc < 0) c = -1;
                            else c = 0; // this would mean o1 == o2! This shouldn't happen.
                        }

                        return c;
                    }
                }
        );

        this.totalMessagesSent = this.totalMessagesReceived = 0;
        this.totalMessagesRetried = 0;
        this.totalBytesSent = this.totalBytesReceived = this.totalProtocolErrors = 0L;

        // Start the receive thread first in order to receive any pending
        // requests messages when messages are first transmitted.
        receiveThread = new Thread(new Receiver(), "MessageDispatcher-receive");
        // Allow the VM to exit if this thread is still running
        receiveThread.setDaemon(true);
        receiveThread.start();

        transmitThread = new Thread(new Transmitter(), "MessageDispatcher-transmit");
        // Allow the VM to exit if this thread is still running
        transmitThread.setDaemon(true);
        transmitThread.start();

        counterHandler = new RequestHandler() {

            public ResponseMessage handleRequest(RequestMessage request) throws Exception {
                if (request == null) {
                    throw new NullPointerException("Request is null");
                }

                StatusCode status;
                if (request.getMethod() != null) {
                    if ("GET".equals(request.getMethod().toUpperCase(Locale.ROOT))) {
                        try {
                            JSONObject job = new JSONObject();
                            float size = outgoingMessageQueue.size();
                            float max = maximumQueueSize;
                            // we only want 1 decimal point percision
                            int load = (int)(size / max * 1000f);
                            job.put("load", (float)load / 10f);
                            job.put("totalBytesSent", totalBytesSent);
                            job.put("totalBytesReceived", totalBytesReceived);
                            job.put("totalMessagesReceived", totalMessagesReceived);
                            job.put("totalMessagesRetried", totalMessagesRetried);
                            job.put("totalMessagesSent", totalMessagesSent);
                            job.put("totalProtocolErrors", totalProtocolErrors);
                            String jsonBody = job.toString();
                            return new ResponseMessage.Builder(request)
                                .statusCode(StatusCode.OK)
                                .body(jsonBody)
                                .build();
                        } catch (Exception exception) {
                            status = StatusCode.BAD_REQUEST;
                        }
                    } // GET
                    else  {
                        //Unsupported request method
                        status = StatusCode.METHOD_NOT_ALLOWED;
                    }
                }
                else {
                    //Unsupported request method
                    status = StatusCode.METHOD_NOT_ALLOWED;
                }
                return new ResponseMessage.Builder(request)
                    .statusCode(status)
                    .build();
            }
        };        

        resetHandler = new RequestHandler() {

            public ResponseMessage handleRequest(RequestMessage request) throws Exception {
                if (request == null) {
                    throw new NullPointerException("Request is null");
                }

                StatusCode status;     
                if (request.getMethod() != null) {
                    if ("PUT".equals(request.getMethod().toUpperCase(Locale.ROOT))) {
                        try {
                            totalBytesSent = totalBytesReceived = totalProtocolErrors = 0L;
                            totalMessagesSent = totalMessagesReceived = totalMessagesRetried = 0;
                            status = StatusCode.OK;
                        } catch (Exception exception) {
                            status = StatusCode.BAD_REQUEST;
                        }
                    }
                    else  {
                        //Unsupported request method
                        status = StatusCode.METHOD_NOT_ALLOWED;
                    } 
                }
                else {
                    status = StatusCode.METHOD_NOT_ALLOWED;
                }
                return new ResponseMessage.Builder(request)
                    .statusCode(status)
                    .build();
            }
        };        

        /*
         * resource definition for pollingInterval: {
         *   "get": {
         *     "schema": {
         *       "properties": {
         *         "value": {
         *           "type": "number",
         *           "description": "The incoming message polling interval
         *              in seconds on the directly connected device."
         *         }
         *       }
         *     },
         *   "put" : {
         *     "parameters": [
         *       {"name": "value",
         *        "type": "integer",
         *        "in": "body",
         *        "description": "The incoming message polling interval in
         *          seconds.",
         *        "required": "true"},
         *       ]
         *   }
         */
        this.pollingIntervalHandler = new RequestHandler() {
            public ResponseMessage handleRequest(RequestMessage request) throws Exception {
                if (request == null) {
                    throw new NullPointerException("Request is null");
                }

                StatusCode status;
                if (request.getMethod() != null) {
                    if ("GET".equals(request.getMethod().toUpperCase(Locale.ROOT))) {
                        try {
                            JSONObject job = new JSONObject();
                            job.put("value", pollingInterval);
                            return new ResponseMessage.Builder(request)
                                .statusCode(StatusCode.OK)
                                .body(job.toString())
                                .build();
                        } catch (Exception exception) {
                            status = StatusCode.BAD_REQUEST;
                        }
                    }
                    else if ("PUT".equals(request.getMethod().toUpperCase(Locale.ROOT))) {
                        try {
                            String jsonRequestBody =
                                new String(request.getBody(), "UTF-8");
                            JSONObject jsonObject =
                                new JSONObject(jsonRequestBody);
                            StringBuilder errors = new StringBuilder();
                            // Use Long objects here in case there was a
                            // problem getting the parameters.
                            long newPollingInterval = getParam(jsonObject, "value",
                                                       errors);
                            if (errors.toString().length() != 0) {
                                return getBadRequestResponse(request,
                                    endpointId, errors.toString());
                            }
                            if (newPollingInterval < 0) {
                                return getBadRequestResponse(request, endpointId, 
                                                             "Polling interval must be a numeric value greater than or equal to 0.");
							}else {
								pollingInterval = newPollingInterval;
							}
                            status = StatusCode.OK;
                        } catch (JSONException exception) {
                            status = StatusCode.BAD_REQUEST;
                        }                            
                    }
                    else  {
                        //Unsupported request method
                        status = StatusCode.METHOD_NOT_ALLOWED;
                    } 
                }
                else {
                    status = StatusCode.METHOD_NOT_ALLOWED;
                }
                return new ResponseMessage.Builder(request)
                    .statusCode(status)
                    .build();
            }
        };        

        try {
            diagnosticsHandler = new DiagnosticsImpl();
            TestConnectivity testConnectivity = new TestConnectivity(endpointId, this);
            testConnectivityHandler = testConnectivity.getTestConnectivityHandler();
            ResourceMessage resourceMessage = new ResourceMessage.Builder()
                .endpointName(endpointId)
                .source(endpointId)
                .register(getResource(endpointId, COUNTERS_URL, counterHandler, Resource.Method.GET))
                .register(getResource(endpointId, RESET_URL, resetHandler,Resource.Method.PUT))     
                .register(getResource(endpointId, POLLING_INTERVAL_URL, pollingIntervalHandler, Resource.Method.GET, Resource.Method.PUT))
                .register(getResource(endpointId, DIAGNOSTICS_URL, diagnosticsHandler, Resource.Method.GET))
                .register(getResource(endpointId, TEST_CONNECTIVITY_URL, testConnectivityHandler, Resource.Method.GET, Resource.Method.PUT))
                .build();
            // TODO: handle error
            this.queue(resourceMessage);
        } catch (Exception e) {
            getLogger().log(Level.SEVERE, e.getMessage());
            e.printStackTrace();
        } catch(Throwable t) {
            t.printStackTrace();
        }

        // Do this last after everything else is established.
        // Populate outgoing message queue from persisted messages, but leave the
        // messages in persistence. The messages are removed from persistence when
        // they are delivered successfully.
        // TODO: IOT-49043
        MessagePersistence messagePersistence = MessagePersistence.getInstance();
        final List<Message> messages = messagePersistence != null
                ? messagePersistence.load(deviceClient.getEndpointId())
                : null;
        if (messages != null && !messages.isEmpty()) {
            sendLock.lock();
            try {
                this.outgoingMessageQueue.addAll(messages);
                messageQueued.signal();
            } finally {
                sendLock.unlock();
            }
        }

    }

    private static int getQueueSize() {
        int size = Integer.getInteger(MAXIMUM_MESSAGES_TO_QUEUE_PROPERTY, DEFAULT_MAXIMUM_MESSAGES_TO_QUEUE);
        // size must be at least 1 or PriorityQueue constructor will throw exception
        return (size > 0 ? size : DEFAULT_MAXIMUM_MESSAGES_TO_QUEUE);
    }

    private static int getMaximumMessagesPerConnection() {
        int max = Integer.getInteger(MAXIMUM_MESSAGES_PER_CONNECTION_PROPERTY, DEFAULT_MAXIMUM_MESSAGES_PER_CONNECTION);
        // size must be at least 1
        return (max > 0 ? max : DEFAULT_MAXIMUM_MESSAGES_PER_CONNECTION);
    }

    private static long getPollingInterval() {
        long interval = Long.getLong(POLLING_INTERVAL_PROPERTY, DEFAULT_POLLING_INTERVAL);
        // polling interval may be zero, which means wait forever
        return (interval >= 0 ? interval : DEFAULT_POLLING_INTERVAL);
    }

    private static int getBasicNumberOfRetries() {
        int max = Integer.getInteger(BASIC_NUMBER_OF_RETRIES_PROPERTY, DEFAULT_BASIC_NUMBER_OF_RETRIES);
        // number of retries must be at least 3
        return (max > 0 ? max : DEFAULT_BASIC_NUMBER_OF_RETRIES);
    }

    /**
     * Returns the Long parameter specified by 'paramName' in the request if
     * it's available.
     *
     * @param jsonObject a {@link JSONObject} containing the JSON request.
     * @param paramName   the name of the parameter to get.
     * @param errors  a {@link StringBuilder} of errors.  Any errors produced
     * from retrieving the parameter will be
     *                appended to this.
     * @return the parameter if it can be retrieved, otherwise {@code null}.
     */
    private Long getParam(JSONObject jsonObject, String paramName,
            StringBuilder errors) {
        Long value = null;

        try {
            Number jsonNumber = (Number)jsonObject.opt(paramName);

            if (jsonNumber != null) {
                try {
                    value = jsonNumber.longValue();

                } catch (NumberFormatException nfe) {
                    errors.append(paramName).append(
                        " must be a numeric value.");
                }
            } else {
                appendSpacesToErrors(errors);
                errors.append("The ").append(paramName).append(
                    " value must be supplied.");
            }
        } catch(ClassCastException cce) {
            appendSpacesToErrors(errors);
            errors.append("The ").append(paramName).append(
                " value must be a number.");
        }

        return value;
    }
    /**
     * Appends spaces (error separators) to the errors list if it's not empty.
     *
     * @param errors a list of errors in a StringBuilder.
     */
    private void appendSpacesToErrors(StringBuilder errors) {
        if (errors.toString().length() != 0) {
            errors.append("  ");
        }
    }

    /**
     * Returns an appropriate response if the request is bad.
     *
     * @param requestMessage the request for this capability.
     * @return an appropriate response if the request is bad.
     */
    private ResponseMessage getBadRequestResponse(RequestMessage requestMessage, String message, String src) {
        return getResponseMessage(requestMessage, message, StatusCode.BAD_REQUEST);
    }

    /**
     * Returns an {@link RequestMessage} with the supplied parameters.
     *
     * @param requestMessage the request for this capability.
     * @param body           the body for the response.
     * @param statusCode     the status code of the response.
     * @return an appropriate response if the request is {@code null}.
     */
    private ResponseMessage getResponseMessage(RequestMessage requestMessage,
            String body, StatusCode statusCode) {
        return new ResponseMessage.Builder(requestMessage)
            .body(body)
            .statusCode(statusCode)
            .build();
    }

    // Get current software version
    private String getSoftwareVersion() {
        // TODO: Need better mechanism for obtaining the version
        return System.getProperty("oracle.iot.client.version", "Unknown");
    }
    
    public void addStorageObjectDependency(StorageObjectImpl storageObject, String clientMsgId) {
        contentLock.lock();
        try {
            if(!contentMap.containsKey(storageObject)) {
                contentMap.put(storageObject, new HashSet<String>());
            }
            contentMap.get(storageObject).add(clientMsgId);
        } finally {
            contentLock.unlock();
        }
    }
    
    public void removeStorageObjectDependency(StorageObjectImpl storageObject) {
        boolean completed = storageObject.getSyncStatus() == SyncStatus.IN_SYNC;
        HashSet<String> ids;
        contentLock.lock();
        try {
            ids = contentMap.remove(storageObject);
            if (!completed && ids != null) {
                failedContentIds.addAll(ids);
            }
        } finally {
            contentLock.unlock();
        }
    }
    
    boolean isContentDependent(String clientId) {
        contentLock.lock();
        try {
            Collection<HashSet<String>> sets = contentMap.values();
            for(HashSet<String> set:sets) {
                if (set.contains(clientId)) {
                    return true;
                }
            }
            return false;
        } finally {
            contentLock.unlock();
        }
    }

    @Override
    public void queue(Message... messages) {
        if (messages == null || messages.length == 0) {
            throw new IllegalArgumentException("message is null");
        }

        final MessagePersistence messagePersistence = MessagePersistence.getInstance();
        if (messagePersistence != null) {
            final Collection<Message> collection = new ArrayList<Message>();
            for (Message message : messages) {
                if (message.getReliability() == Message.Reliability.GUARANTEED_DELIVERY) {
                    collection.add(message);
                }
            }
            // TODO: call save from separate thread?
             if (collection.size() > 0) {
                 messagePersistence.save(collection, deviceClient.getEndpointId());
             }
        }

        sendLock.lock();
        try {
            if ((outgoingMessageQueue.size() + messages.length) >= maximumQueueSize) {
                throw new ArrayStoreException("queue is full");
            }
            for (Message message : messages) {
                outgoingMessageQueue.offer(message);
            }
            messageQueued.signal();
        } finally {
            sendLock.unlock();
        }
    }

    @Override
    public void offer(Message... messages) {
        if (messages == null || messages.length == 0) {
            throw new IllegalArgumentException("message is null");
        }

        final MessagingPolicyImpl messagingPolicy;

        final PersistenceStore persistenceStore =
                PersistenceStoreManager.getPersistenceStore(deviceClient.getEndpointId());
        final Object mpiObj = persistenceStore.getOpaque(MessagingPolicyImpl.class.getName(), null);
        if (mpiObj == null) {
            messagingPolicy = new MessagingPolicyImpl(deviceClient);
            persistenceStore
                    .openTransaction()
                    .putOpaque(MessagingPolicyImpl.class.getName(), messagingPolicy)
                    .commit();

            final DevicePolicyManager devicePolicyManager
                    = DevicePolicyManager.getDevicePolicyManager(deviceClient);
            devicePolicyManager.addChangeListener(messagingPolicy);

        } else {
            messagingPolicy = MessagingPolicyImpl.class.cast(mpiObj);
        }

        final List<Message> messagesToQueue = new ArrayList<Message>();
        try {
            for (Message message : messages) {
                Message[] messagesFromPolicy = messagingPolicy.applyPolicies(message);
                if (messagesToQueue != null) {
                    Collections.addAll(messagesToQueue, messagesFromPolicy);
                }
            }
        } catch (IOException e) {
            // TODO: retry?
            getLogger().log(Level.WARNING, e.getMessage());
            if (errorCallback != null) {
                final List<Message> messageList = new ArrayList<Message>(messages.length);
                Collections.addAll(messageList, messages);
                errorCallback.failed(messageList, e);
            }
        } catch (GeneralSecurityException e) {
            // TODO: retry?
            getLogger().log(Level.WARNING, e.getMessage());
            if (errorCallback != null) {
                final List<Message> messageList = new ArrayList<>(messages.length);
                Collections.addAll(messageList, messages);
                errorCallback.failed(messageList, e);
            }
        }

        if (messagesToQueue.size() > 0) {
            queue(messagesToQueue.toArray(new Message[messagesToQueue.size()]));
        }
    }

    /** {@inheritDoc} */
    @Override
    public synchronized void close() throws IOException {
        if (!closed) {

            requestClose = true;

            if (receiveThread != null) {
                receiveThread.interrupt();
            }

            try {
                transmitThread.interrupt();
                transmitThread.join();
            } catch (InterruptedException e) {
                // Restore the interrupted status
                Thread.currentThread().interrupt();
            }

            closed = true;
        }
    }

    public boolean isClosed() {
        return requestClose;
    }

    /*************************************************************************
     *
     * Handling of the blocking queue, servicing the queue,
     * and dispatching dequeued entries.
     *
     *************************************************************************/
    private class Transmitter implements Runnable {

        @Override
        public void run() {
            while (true) {

                if (requestClose && outgoingMessageQueue.isEmpty()) {
                    break;
                }

                List<Message> messageList = new ArrayList<Message>();
                LinkedList<Message> errorList = new LinkedList<Message>();
                sendLock.lock();
                try {
                    // Transmit thread blocks waiting for an outgoing message
                    while (!requestClose && outgoingMessageQueue.isEmpty()) {
                        messageQueued.await();
                    }

                    final int size = outgoingMessageQueue.size();
                    if (size > 0) {

                        HashSet<String> inProgressSources = new HashSet<String>(size);
                        LinkedList<Message> waitList = new LinkedList<Message>();
                        Message m;
                        while ((m = outgoingMessageQueue.poll()) != null) {
                            String clientId = m.getClientId();
                            String source = m.getSource();

                            if (failedContentIds.contains(clientId)) {
                                errorList.add(m);
                                continue;
                            }

                            if (m.getType() == Message.Type.RESPONSE
                                    || !(inProgressSources.contains(source) || isContentDependent(clientId))) {
                                m.setRemainingRetries(getBasicNumberOfRetries()); // TODO - why?
                                messageList.add(m);
                                if (messageList.size() == size) {
                                    break;
                                }
                            } else {
                                inProgressSources.add(source);
                                waitList.add(m);
                            }
                        }
                        outgoingMessageQueue.addAll(waitList);
                    }

                } catch (InterruptedException e) {
                    // restore interrupt state
                    Thread.currentThread().interrupt();

                } finally {
                    sendLock.unlock();
                }

                final int nMessages = messageList.size();
                if (nMessages == 0) {
                    continue;
                }

                send(messageList);

                if (!errorList.isEmpty() && MessageDispatcherImpl.this.errorCallback != null) {
                    MessageDispatcherImpl.this.errorCallback.
                            failed(errorList, new IOException("Content sync failed"));
                }
            }
        }

        private void send(List<Message> messages) {

            if ((messages == null) || (messages.isEmpty())) {
                return;
            }

            try {
                deviceClient.send(messages.toArray(new Message[messages.size()]));

                // Send is successful, so if the receiver thread was waiting
                // for reconnection, send a wakeup signal.
                if (waitOnReconnect.get()) {
                    synchronized (waitOnReconnect) {
                        waitOnReconnect.set(false);
                        waitOnReconnect.notify();
                    }
                }

                if (MessageDispatcherImpl.this.deliveryCallback !=
                        null) {
                    MessageDispatcherImpl.this.deliveryCallback.
                            delivered(messages);
                }
                totalMessagesSent += messages.size();

                JSONArray jsonArray = new JSONArray();
                for (Message message : messages) {
                    jsonArray.put(message.toJson());
                }

                final Iterator<Message> iterator = messages.iterator();
                while (iterator.hasNext()) {
                    final Message message = iterator.next();

                    // TODO: There must be a better way to handle totalBytesSent
                    totalBytesSent += message.toJson().toString()
                            .getBytes(Charset.forName("UTF-8")).length;

                    //
                    // Guaranteed delivery messages need to be deleted from
                    // message persistence. If the message is not guaranteed
                    // delivery, remove it from the list of messages. At
                    // the end of the loop, only GUARANTEED_DELIVERY messages
                    // will be in the original list and we can call delete on that.
                    //
                    final MessagePersistence messagePersistence = MessagePersistence.getInstance();
                    if (messagePersistence != null &&
                            message.getReliability() != Message.Reliability.GUARANTEED_DELIVERY) {
                        iterator.remove();
                    }
                }

                // wake up receive thread
                receiveLock.lock();
                try {
                    messageSent.signal();
                } finally {
                    receiveLock.unlock();
                }

                // TODO: call delete from separate thread?
                final MessagePersistence messagePersistence = MessagePersistence.getInstance();
                if (messagePersistence != null && !messages.isEmpty()) {
                    messagePersistence.delete(messages);
                }

            } catch (IOException e) {

                totalProtocolErrors++;

                for (Message message : messages) {

                    if (message.getRemainingRetries() > 0) {

                        if (outgoingMessageQueue.size() < maximumQueueSize
                                || message.getReliability() == Message.Reliability.GUARANTEED_DELIVERY) {
                            // allow the queue to overflow for guaranteed delivery
                            totalMessagesRetried++;
                            message.setRemainingRetries(message.getRemainingRetries() - 1);
                            outgoingMessageQueue.add(message);

                        } else {
                            getLogger().log(Level.INFO,
                                    "Cannot queue message for retry. Message discarded: "
                                            + message.getClientId());
                            continue;
                        }


                    } else if (message.getReliability() != Message.Reliability.GUARANTEED_DELIVERY) {
                        getLogger().log(Level.INFO,
                                "Cannot queue message for retry. Message discarded: "
                                        + message.getClientId());
                        continue;
                    }

                }

                if (MessageDispatcherImpl.this.errorCallback != null) {
                    MessageDispatcherImpl.this.errorCallback.
                            failed(messages, e);
                }

            } catch (GeneralSecurityException e) {

                // Do not retry messages that failed because of GeneralSecurityException
                if (MessageDispatcherImpl.this.errorCallback != null) {
                    MessageDispatcherImpl.this.errorCallback.
                            failed(messages, e);
                }

            }
        }
    }

    private final Lock pendingQueueLock = new ReentrantLock();
    private final Condition pendingTrigger = pendingQueueLock.newCondition();

    // RequestMessages that were not handled are stored here by the Dispatcher thread.
    // See Dispatcher#run() method.
    private final Queue<RequestMessage> pendingRequestMessages = new LinkedList<RequestMessage>();

    private class PendingRequestProcessor implements Runnable {

        private final int MAX_WAIT_TIME = 1000;
        private final long settleTime;
        private final long timeZero;
        private final long averageWaitTime; // in nano-seconds!
        private boolean settled;

        private PendingRequestProcessor(long settleTime) {

            this.settleTime = settleTime;
            this.timeZero = System.currentTimeMillis();
            if (settleTime <= MAX_WAIT_TIME) {
                this.averageWaitTime = TimeUnit.MILLISECONDS.toNanos(this.settleTime);
            } else {
                final long quotient = this.settleTime / MAX_WAIT_TIME;
                final long waitTimeMillis = MAX_WAIT_TIME + (this.settleTime - quotient * MAX_WAIT_TIME) / quotient;
                averageWaitTime = TimeUnit.MILLISECONDS.toNanos(waitTimeMillis);
            }

            settled = false;
        }

        @Override
        public void run() {

            while(!requestClose && !settled) {

                List<RequestMessage> messageList = null;

                pendingQueueLock.lock();
                try {

                    long waitTime = TimeUnit.MILLISECONDS.toNanos(
                            // wait at most the amount of time left before settleTime expires
                            settleTime - (System.currentTimeMillis() - timeZero)
                    );

                    if (waitTime > 0 && pendingRequestMessages.isEmpty()) {
                        waitTime = pendingTrigger.awaitNanos(waitTime);
                    }

                    //
                    // waitTime represents how much time is left before settleTime expires.
                    // If we waited in the pendingTrigger.awaitNanos call, then waitTime
                    // has been adjusted by how long we actually waited. If waitTime is
                    // greater than zero and we blocked in awaitNanos, then a request
                    // was added to the queue. If waitTime is less than or equal to zero,
                    // then we waited the entire time, or waitTime was less than or equal
                    // to zero to begin with because settleTime has expired.
                    //

                    // If we waited the entire time and the queue is still empty, bail.
                    // Otherwise, a request was queued, or the queue wasn't empty to begin with.
                    if (waitTime <= 0 && pendingRequestMessages.isEmpty()) {
                        break;
                    }

                    // In the case where waitTime is greater than zero, we want to make
                    // sure we wait at least averageWaitTime before dispatching the request.
                    // In other words, if we were blocked in the awaitNanos and a request
                    // was added to the queue, we don't want to process it immediately;
                    // rather, wait at least one average wait time so we don't thrash.
                    if (waitTime > 0) {
                        waitTime = averageWaitTime;
                        while (waitTime > 0) {
                            waitTime = pendingTrigger.awaitNanos(waitTime);
                        }
                    }

                    // Operate on a copy of pendingRequestMessages so that we don't hold
                    // pendingQueueLock while dispatching. At the end of this loop,
                    // pendingRequestMessages will be empty.
                    if (!pendingRequestMessages.isEmpty()) {
                        messageList = new ArrayList<RequestMessage>(pendingRequestMessages.size());
                        RequestMessage requestMessage = null;
                        while ((requestMessage = pendingRequestMessages.poll()) != null) {
                            messageList.add(requestMessage);
                        }
                    }

                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    continue;
                } finally {
                    pendingQueueLock.unlock();
                }

                settled |= settleTime <= (System.currentTimeMillis() - timeZero);

                if (messageList != null) {
                    Iterator<RequestMessage> iterator = messageList.iterator();
                    while (iterator.hasNext()) {
                        RequestMessage requestMessage = iterator.next();
                        dispatcher.execute(new Dispatcher(requestMessage, settled));
                    }
                }
            }
        }
    }

    private class Dispatcher implements Runnable {

        final RequestMessage requestMessage;
        private boolean settled;

        private Dispatcher(RequestMessage requestMessage, boolean settled) {

            this.requestMessage = requestMessage;
            this.settled = settled;
        }


        @Override
        public void run() {

                ResponseMessage responseMessage =
                    RequestDispatcher.getInstance().dispatch(requestMessage);


                if (settled || responseMessage.getStatusCode() != StatusCode.NOT_FOUND) {
                    try {
                        MessageDispatcherImpl.this.queue(responseMessage);
                    } catch (Throwable t) {
                        getLogger().log(Level.SEVERE, t.toString());
                    }

                } else { // not settled && status code == not found

                    // try this again later.
                    pendingQueueLock.lock();
                    try {
                        if (pendingRequestMessages.offer(requestMessage)) {
                            pendingTrigger.signal();
                        } else {
                            getLogger().log(Level.SEVERE, "Cannot queue request for dispatch");
                            responseMessage =
                                    new ResponseMessage.Builder(requestMessage)
                                            .statusCode(StatusCode.INTERNAL_SERVER_ERROR)
                                            .build();
                            MessageDispatcherImpl.this.queue(responseMessage);
                        }
                    } finally {
                        pendingQueueLock.unlock();
                    }
                    return;
                }
            }
    }

    private  static ThreadFactory threadFactory = new ThreadFactory() {
        @Override
        public Thread newThread(Runnable r) {
            final SecurityManager s = System.getSecurityManager();
            final ThreadGroup group = (s != null) ? s.getThreadGroup()
                    : Thread.currentThread().getThreadGroup();

            final Thread t = new Thread(group, r, "dispatcher-thread", 0);

            // this is opposite of what the Executors.DefaultThreadFactory does
            if (!t.isDaemon())
                t.setDaemon(true);
            if (t.getPriority() != Thread.NORM_PRIORITY)
                t.setPriority(Thread.NORM_PRIORITY);
            return t;
        }
    };

    private static final Executor dispatcher;
    static {
        if (REQUEST_DISPATCHER_THREAD_POOL_SIZE > 1) {
            dispatcher = Executors.newFixedThreadPool(REQUEST_DISPATCHER_THREAD_POOL_SIZE, threadFactory);
        } else {
            dispatcher =  Executors.newSingleThreadExecutor(threadFactory);
        }
    }

    private class Receiver implements Runnable {

        private final long settleTime;
        private final long timeZero;
        private boolean settled;

        private Receiver() {

            long value = Long.getLong(SETTLE_TIME_PROPERTY, DEFAULT_SETTLE_TIME);
            settleTime = (value >= 0 ? value : DEFAULT_SETTLE_TIME);
            timeZero = System.currentTimeMillis();
            settled = settleTime == 0;

            if (settleTime > 0) {
                final PendingRequestProcessor pendingRequestProcessor =
                        new PendingRequestProcessor(settleTime);
                final Thread thread = new Thread(pendingRequestProcessor);
                thread.setDaemon(true);
                thread.start();
            }

        }
        @Override
        public void run() {

            mainLoop:
            while (!requestClose) {

                // If there was a SocketException observed before,
                // wait for this flag "waitOnReconnect" to be set to false
                // by the transmitter thread after the next message was sent successfully
                // or close() was called.
                while(waitOnReconnect.get() && !requestClose) {
                    synchronized (waitOnReconnect) {
                        try {
                            waitOnReconnect.wait();
                        } catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                        // If close() was called, exit the main loop.
                        if (requestClose) {
                            break mainLoop;
                        }
                    }
                }

                // If not long polling, block until the pollingInterval has
                // timed out or a message has been sent.
                if (!MessageDispatcherImpl.this.useLongPolling){
                    receiveLock.lock();
                    try {
                        // Wait here until the messageSent is signaled, or
                        // until pollingInterval is expired.
                        if (pollingInterval > 0) {
                            messageSent.await(pollingInterval, TimeUnit.MILLISECONDS);
                        } else {
                            messageSent.await();
                        }
                    } catch (InterruptedException e) {
                        // Restore the interrupted status
                        Thread.currentThread().interrupt();
                        continue;
                    } finally {
                        receiveLock.unlock();
                    }
                }

                try {
                    RequestMessage requestMessage = null;
                    // Receive will ignore -1 timeout when not long polling
                    while ((requestMessage = deviceClient.receive(-1)) != null) {
                        totalMessagesReceived++;
                        totalBytesReceived += requestMessage.toJson().toString()
                                .getBytes(Charset.forName("UTF-8")).length;
                        if (!requestClose) {
                            settled |= settleTime <= (System.currentTimeMillis() - timeZero);
                            dispatcher.execute(new Dispatcher(requestMessage, settled));
                        }
                    }
                } catch (IOException ie) {
                    getLogger().log(Level.FINEST,
                            "MessageDispatcher.receiver.run: " + ie.toString());
                    // Network connection issues detected.
                    // So wait for reconnection signal from transmitter thread.
                    waitOnReconnect.set(true);
                } catch (GeneralSecurityException ge) {
                    getLogger().log(Level.FINEST,
                            "MessageDispatcher.receiver.run: " + ge.toString());
                    // Network connection issues detected.
                    // So wait for reconnection signal from transmitter thread.
                    waitOnReconnect.set(true);
                }
            }
        }
    }

    /* package */
    private Resource getResource(String endpointId, String name, RequestHandler requestHandler, Resource.Method... methods) {

        final MessageDispatcher messageDispatcher = this;
                final RequestDispatcher requestDispatcher =
                    messageDispatcher.getRequestDispatcher();
                requestDispatcher.registerRequestHandler(endpointId, name, requestHandler);

        Resource.Builder builder = new Resource.Builder().endpointName(endpointId);
        for (Resource.Method m: methods) {
            builder = builder.method(m);
        }
        return builder.path(name).name(name).build();
    }

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }   
}
