/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.iot.client.impl;

import com.oracle.iot.client.HttpResponse;
import com.oracle.iot.client.RestApi;
import com.oracle.iot.client.SecureConnection;
import com.oracle.iot.client.StorageObject;
import com.oracle.iot.client.impl.StorageAuthenticationResponse;
import com.oracle.iot.client.impl.StorageConnection;
import com.oracle.iot.client.impl.StorageObjectDelegate;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.channels.FileChannel;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONException;
import org.json.JSONObject;

public abstract class StorageConnectionBase
implements StorageConnection {
    private static final String STORAGE_CLOUD_HOST;
    private static final String DEFAULT_STORAGE_CLOUD_HOST = "storage.oraclecloud.com";
    private static final boolean virtualStorageDirectories;
    private static final String REST_STORAGE_AUTHENTICATION;
    private static final String AUTH_TOKEN_HEADER = "X-Auth-Token";
    private static final int DEFAULT_RETRY_LIMIT = 2;
    private static final int BAD_CHECKSUM_STATUS = -1;
    private static final int RETRY_LIMIT;
    private final SecureConnection secureConnection;
    private String storageContainerUrl = null;
    private String authToken = null;
    private boolean closed;
    private static final Logger LOGGER;

    protected StorageConnectionBase(SecureConnection secureConnection) {
        this.secureConnection = secureConnection;
        this.closed = false;
        this.authenticate();
    }

    @Override
    public final void sync(StorageObject storageObject) throws IOException, GeneralSecurityException {
        if (storageObject.getInputStream() != null) {
            this.upload(storageObject);
        } else if (storageObject.getOutputStream() != null) {
            this.download(storageObject);
        } else {
            throw new IllegalArgumentException("InputStream and OutputStream are not set.");
        }
    }

    private void upload(StorageObject storageObject) throws IOException, GeneralSecurityException {
        int responseCode = this.transfer(storageObject);
        if (responseCode != 201) {
            throw new IOException("Upload failed " + responseCode);
        }
    }

    private void download(StorageObject storageObject) throws IOException, GeneralSecurityException {
        int responseCode = this.transfer(storageObject);
        if (responseCode != 200 && responseCode != 206) {
            throw new IOException("Download failed " + responseCode);
        }
    }

    public final StorageObject createStorageObject(String clientId, String name, String contentType) throws IOException, GeneralSecurityException {
        String path = virtualStorageDirectories && clientId != null ? clientId + "/" + name : name;
        String contentStorageLocation = this.getStorageContainerUrl() + "/" + path;
        StorageObject storageObject = this.createStorageObject(contentStorageLocation, name, contentType, null, null, -1);
        return storageObject;
    }

    public final StorageObject createStorageObject(String storageUrl) throws IOException, GeneralSecurityException {
        String name;
        URL url;
        try {
            url = new URL(storageUrl);
            String fullContainerUrl = this.getStorageContainerUrl() + "/";
            if (!storageUrl.startsWith(fullContainerUrl)) {
                throw new GeneralSecurityException("Storage container URL does not match.");
            }
            name = virtualStorageDirectories ? storageUrl.substring(storageUrl.lastIndexOf(47) + 1) : storageUrl.substring(fullContainerUrl.length());
        }
        catch (MalformedURLException ex) {
            throw new IllegalArgumentException("Storage Cloud URL is invalid", ex);
        }
        for (int i = 0; i < RETRY_LIMIT; ++i) {
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            connection.setRequestMethod("HEAD");
            connection.setRequestProperty(AUTH_TOKEN_HEADER, this.authToken);
            connection.connect();
            int length = connection.getContentLength();
            String type = connection.getContentType();
            String encoding = connection.getContentEncoding();
            String date = connection.getHeaderField("Last-Modified");
            int responseCode = connection.getResponseCode();
            if (responseCode == 200) {
                return this.createStorageObject(storageUrl, name, type, encoding, date, length);
            }
            if (responseCode != 401) {
                throw new IOException("HTTP " + responseCode);
            }
            this.authenticate();
        }
        throw new IOException("HTTP 401");
    }

    protected abstract StorageObject createStorageObject(String var1, String var2, String var3, String var4, String var5, int var6);

    public static boolean isStorageCloudURI(String uri) {
        try {
            if (new URI(uri).getHost().contains(STORAGE_CLOUD_HOST)) {
                return true;
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return false;
    }

    @Override
    public final synchronized void close() throws IOException {
        if (!this.closed) {
            this.closed = true;
        }
    }

    private String getStorageContainerUrl() throws IOException, GeneralSecurityException {
        if (this.storageContainerUrl == null) {
            StorageAuthenticationResponse storageAuthenticationResponse = this.getStorageAuthentication();
            this.storageContainerUrl = storageAuthenticationResponse.getStorageContainerUrl();
            this.authToken = storageAuthenticationResponse.getAuthToken();
        }
        return this.storageContainerUrl;
    }

    private int transfer(StorageObject storageObject) throws IOException, GeneralSecurityException {
        InputStream inputStream;
        boolean upload;
        int responseCode = 0;
        if (this.authToken == null && !this.authenticate()) {
            responseCode = 401;
        }
        boolean bl = upload = (inputStream = storageObject.getInputStream()) != null;
        if (upload && inputStream.markSupported()) {
            inputStream.mark(Integer.MAX_VALUE);
        }
        String scPath = storageObject.getURI() == null ? this.storageContainerUrl + "/" + storageObject.getName() : storageObject.getURI();
        for (int retries = 0; retries < RETRY_LIMIT; ++retries) {
            if (this.authToken != null) {
                responseCode = this.transferContent(scPath, storageObject);
            }
            if (upload && responseCode == 201 || !upload && (responseCode == 200 || responseCode == 206)) break;
            if (responseCode == 401) {
                this.authenticate();
            } else if (responseCode != -1) break;
            if (!upload) continue;
            if (inputStream.markSupported()) {
                inputStream.reset();
                continue;
            }
            if (!(inputStream instanceof FileInputStream)) {
                return responseCode;
            }
            FileInputStream fs = (FileInputStream)inputStream;
            FileChannel fc = fs.getChannel();
            fc.position(0L);
        }
        return responseCode;
    }

    private int transferContent(String scPath, StorageObject storageObject) throws IOException {
        String digestOut;
        int len;
        MessageDigest digest = null;
        try {
            digest = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException ex) {
            StorageConnectionBase.getLogger().log(Level.WARNING, "Storage Cloud Service: checksum could not be verified.", ex);
        }
        String contentType = storageObject.getType();
        String encoding = storageObject.getEncoding();
        InputStream inputStream = storageObject.getInputStream();
        OutputStream outputStream = storageObject.getOutputStream();
        long transferredBytes = 0L;
        boolean upload = inputStream != null;
        OutputStream os = null;
        InputStream is = null;
        URL url = new URL(scPath);
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        if (upload) {
            is = inputStream;
            connection.setRequestMethod("PUT");
            connection.setChunkedStreamingMode(1024);
            connection.setRequestProperty("Content-Type", contentType != null ? contentType : "application/octet-stream");
            if (encoding != null) {
                connection.setRequestProperty("Content-Encoding", encoding);
            }
            connection.setDoOutput(true);
        } else {
            os = outputStream;
            connection.setRequestMethod("GET");
            connection.setDoOutput(false);
        }
        connection.setRequestProperty(AUTH_TOKEN_HEADER, this.authToken);
        connection.connect();
        if (upload) {
            os = connection.getOutputStream();
        } else {
            is = connection.getInputStream();
        }
        byte[] b = new byte[1024];
        while ((len = is.read(b)) != -1) {
            if (((StorageObjectDelegate)storageObject).isCancelled()) {
                if (upload) {
                    os.close();
                } else {
                    is.close();
                }
                return -2;
            }
            os.write(b, 0, len);
            if (digest != null) {
                digest.update(b, 0, len);
            }
            ((StorageObjectDelegate)storageObject).setTransferredBytes(transferredBytes += (long)len);
        }
        if (upload) {
            os.close();
        } else {
            is.close();
        }
        int responseCode = connection.getResponseCode();
        String checksum = connection.getHeaderField("ETag");
        String date = connection.getHeaderField("Last-Modified");
        if (checksum != null && digest != null && !checksum.equals(digestOut = StorageConnectionBase.bytesToHexString(digest.digest()))) {
            StorageConnectionBase.getLogger().log(Level.INFO, "Storage Cloud Service: checksum mismatch");
            return -1;
        }
        if (upload && responseCode == 201) {
            ((StorageObjectDelegate)storageObject).setMetadata(date, transferredBytes);
            is.close();
        } else if (!(upload || responseCode != 200 && responseCode != 206)) {
            os.close();
        }
        return responseCode;
    }

    private StorageAuthenticationResponse getStorageAuthentication() throws IOException, GeneralSecurityException {
        JSONObject json;
        HttpResponse response = this.secureConnection.get(REST_STORAGE_AUTHENTICATION);
        int status = response.getStatus();
        if (status != 200) {
            throw new IOException(response.getVerboseStatus("GET", REST_STORAGE_AUTHENTICATION));
        }
        String jsonResponse = new String(response.getData(), "UTF-8");
        try {
            json = new JSONObject(jsonResponse);
        }
        catch (JSONException e) {
            throw new RuntimeException(e);
        }
        StorageAuthenticationResponse storageAuthenticationResponse = StorageAuthenticationResponse.fromJson(json);
        if (StorageConnectionBase.getLogger().isLoggable(Level.FINE)) {
            StorageConnectionBase.getLogger().fine(storageAuthenticationResponse.toString());
        }
        return storageAuthenticationResponse;
    }

    private boolean authenticate() {
        try {
            StorageAuthenticationResponse storageAuthenticationResponse = this.getStorageAuthentication();
            this.storageContainerUrl = storageAuthenticationResponse.getStorageContainerUrl();
            this.authToken = storageAuthenticationResponse.getAuthToken();
            return true;
        }
        catch (Exception e) {
            StorageConnectionBase.getLogger().log(Level.INFO, "IoT storage API cannot be accessed: " + e.getMessage());
            return false;
        }
    }

    private static String bytesToHexString(byte[] in) {
        StringBuilder sb = new StringBuilder();
        for (byte b : in) {
            sb.append(String.format("%02x", b));
        }
        return sb.toString();
    }

    private static Logger getLogger() {
        return LOGGER;
    }

    static {
        REST_STORAGE_AUTHENTICATION = RestApi.V2.getReqRoot() + "/provisioner/storage";
        Integer val = Integer.getInteger("com.oracle.iot.client.storage_connection_retry_limit");
        RETRY_LIMIT = val != null && val > 0 ? val : 2;
        STORAGE_CLOUD_HOST = System.getProperty("com.oracle.iot.client.storage_connection_host_name", DEFAULT_STORAGE_CLOUD_HOST);
        String value = System.getProperty("oracle.iot.client.disable_storage_object_prefix");
        virtualStorageDirectories = value == null || "".equals(value) || !Boolean.parseBoolean(value);
        LOGGER = Logger.getLogger("oracle.iot.client");
    }
}

