/*
 *  Copyright © 2016, 2017, Oracle and/or its affiliates. All rights reserved.
 *  This software is dual-licensed to you under the MIT License (MIT) and
 *  the Universal Permissive License (UPL). See the LICENSE file in the root
 *  directory for license terms. You may choose either license, or both.
 */

import Foundation
import UIKit

/**
 * Represents a file.
 */
class FileRep {
    
    let displayName: String
    let isDirectory: Bool
    let fileExtension: String?
    let fileAttributes: Dictionary<FileAttributeKey, Any>?
    let filePath: URL
    let type: FileRepType
    
    /**
     * Initialize a FileRep object with a filePath
     *
     * - parameter filePath: URL filePath
     *
     */
    init(filePath: URL) {
        self.filePath = filePath
        let isDirectory = checkDirectory(filePath: filePath)
        self.isDirectory = isDirectory
        if self.isDirectory {
            self.fileAttributes = nil
            self.fileExtension = nil
            self.type = .Directory
        }
        else {
            self.fileAttributes = getFileAttributes(filePath: self.filePath)
            self.fileExtension = filePath.pathExtension
            if let fileExtension = fileExtension {
                self.type = FileRepType(rawValue: fileExtension) ?? .Default
            }
            else {
                self.type = .Default
            }
        }
        self.displayName = filePath.lastPathComponent
    }
    
    func getDisplayName() -> String {
        return self.displayName
    }
    
    var description: String {
        var desc:String = String()
        desc += "--------------------------------------------------------------------" + "\r\n"
        desc += "displayName " + self.displayName + "\r\n"
        desc += "isDirectory " + String(self.isDirectory) + "\r\n"
        if (fileExtension != nil) {
            desc += "fileExtension " + self.fileExtension! + "\r\n"
        }
        desc += "filePath " + self.filePath.path + "\r\n"
        desc += "type " + self.type.rawValue + "\r\n"
        return desc
    }
}

/**
 * FileRep type
 */
public enum FileRepType: String {
    case Directory = "directory"
    case GIF = "gif"
    case JPG = "jpg"
    case JSON = "json"
    case PDF = "pdf"
    case PLIST = "plist"
    case PNG = "png"
    case ZIP = "zip"
    case CONF = "conf"
    case UPF = "upf"
    case Default = "file"
    
    /**
     * Get representative image for file type
     *
     * - returns: UIImage for file type
     */
    public func image() -> UIImage? {
        let bundle =  Bundle(for: FileParser.self)
        var fileName = String()
        switch self {
        case .Directory: fileName = "folder@2x.png"
        case .JPG, .PNG, .GIF: fileName = "image@2x.png"
        case .PDF: fileName = "pdf@2x.png"
        case .ZIP: fileName = "zip@2x.png"
        default: fileName = "file@2x.png"
        }
        let file = UIImage(named: fileName, in: bundle, compatibleWith: nil)
        return file
    }
}

/**
 * Check if file path URL is directory or file.
 *
 * - parameter filePath: URL file path.
 *
 * - returns: true if directory, false otherwise
 */
fileprivate func checkDirectory(filePath: URL) -> Bool {
    var isDir: ObjCBool = ObjCBool(false)
    let exists = FileManager.default.fileExists(atPath: filePath.path,
                isDirectory: UnsafeMutablePointer(&isDir))
    return exists && isDir.boolValue
}

fileprivate func getFileAttributes(filePath: URL) -> Dictionary<FileAttributeKey, Any>? {
    let path = filePath.path
    let fileManager = FileParser.sharedInstance.filemgr
    do {
        let attributes = try fileManager.attributesOfItem(atPath: path)
        return attributes
    } catch {}
    return nil
}
