/*
 *  Copyright © 2016, 2017, Oracle and/or its affiliates. All rights reserved.
 *  This software is dual-licensed to you under the MIT License (MIT) and
 *  the Universal Permissive License (UPL). See the LICENSE file in the root
 *  directory for license terms. You may choose either license, or both.
 */

import UIKit

class ProvisioningView: PasswordView, UIPickerViewDataSource,
                                          UIPickerViewDelegate {
    
    @objc var pickerDataSource: Array<String> = []
    @objc var pathList: Array<String> = []
    
    @IBOutlet weak var filePickerOutlet: UIPickerView!
    @IBOutlet weak var headLineLabelOutlet: UILabel!
    @IBOutlet weak var provFileLabelOutlet: UILabel!

    @IBOutlet var viewControllerOutlet: UIView!
    
    // Requires the application to create Main.storyboard
    // and an intial view controller
    @objc var MainStoryboard: UIStoryboard? = nil

    override func viewDidLoad() {
        self.pickerDataSource = makePickerList()
        self.filePickerOutlet.dataSource = self
        self.filePickerOutlet.delegate = self
        MainStoryboard = UIStoryboard(name: "Main", bundle: nil)
        super.viewDidLoad()
    }
    
    func numberOfComponents(in pickerView: UIPickerView) -> Int {
        return 1
    }
    
    func pickerView(_ pickerView: UIPickerView, didSelectRow row: Int,
                    inComponent component: Int) {
        passwordOutlet.text = ""
    }
    
    func pickerView(_ pickerView: UIPickerView,
                    numberOfRowsInComponent component: Int) -> Int {
        return pickerDataSource.count
    }
    
    func pickerView(_ pickerView: UIPickerView, titleForRow row: Int,
                    forComponent component: Int) -> String? {
        return pickerDataSource[row]
    }
    
    @nonobjc func pickerView(_ pickerView: UIPickerView, viewForRow row: Int,
                             ForComponent component: Int,
                             reusingView view: UIView?) -> UIView {
        var label = view as! UILabel?
        if label == nil {
            label = UILabel()
        }
        let data = pickerDataSource[row]
        let title = NSAttributedString(string: data,
                attributes: [NSAttributedStringKey.font: UIFont.systemFont(ofSize: 6.0,
                                                weight: UIFont.Weight.regular)])
        label?.attributedText = title
        label?.textAlignment = .left
        return label!
    }
    

    // Create the file list.
    @objc func makePickerList() -> [String] {
        let parser = FileParser.sharedInstance
        let path = parser.documentsURL()
        let files: [FileRep] = parser.filesForDirectory(directoryURL: path)
        var list = Array(repeating: "", count: files.count)
        pathList = Array(repeating: "", count: files.count)
        var relevantCount: Int = 0
        for file in files {
            list[relevantCount] = file.getDisplayName()
            pathList[relevantCount] = file.filePath.path
            relevantCount += 1
        }
        if files.count > 0 {
            passwordOutlet.isEnabled = true
        }
        return list
    }
    
    override func verifyPassword(password: String) -> Bool {
        return validateTam(path: pathList[filePickerOutlet.selectedRow(inComponent: 0)],
                           password: password)
    }

    // Handle the ok button.
    override func handleOk(button: UIButton) {
        let path: String = pathList[filePickerOutlet.selectedRow(inComponent: 0)]
        let file: String = URL(string: path)!.lastPathComponent
        // Save the current provisioning information for the next app run.
        saveSettingsFile(path: file, password: getPassword())
        Provisioning.setProvisioningFilePath(pf: path)
        Provisioning.setProvisioningPassword(pp: getPassword())
        // Every App that wants to use generic File and Network bootstrapping
        // must have an "Main" storyboard with at one view controller
        // tagged as the "initial view controller"
        self.navigationController?.pushViewController(
            (self.MainStoryboard?.instantiateInitialViewController())!,
            animated: true)
        
    }
}

