/*
 *  Copyright © 2016, 2017, Oracle and/or its affiliates. All rights reserved.
 *  This software is dual-licensed to you under the MIT License (MIT) and
 *  the Universal Permissive License (UPL). See the LICENSE file in the root
 *  directory for license terms. You may choose either license, or both.
 */

import UIKit

var dcdSample:DirectlyConnectedDeviceSample?

@objc class DirectlyConnectedDeviceSampleView: UIViewController, UITextViewDelegate {
    
    // The virtual device update interval in seconds
    let UPDATE_INTERVAL:Int = 3
    
    //MARK: Properties
    @IBOutlet weak var humidityIdOutlet: UILabel!
    @IBOutlet weak var humidityOutlet: UILabel!
    @IBOutlet weak var humidityThresholdOutlet: UILabel!
    
    @IBOutlet weak var stopButtonOutlet: UIButton!
    
    @IBOutlet weak var resetButtonOutlet: UIButton! {
        didSet {
            if Provisioning.isProvisioningStatic() {
                resetButtonOutlet.isHidden = true
            }
        }
    }
    
    @IBOutlet weak var outputLabelOutlet: UILabel!
    @IBOutlet weak var outputTextViewOutlet: UITextView!
    @IBOutlet var ViewControllerOutlet: UIView!
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        outputTextViewOutlet.delegate = self
        UIGraphicsBeginImageContext(CGSize(width: self.view.frame.size.width,
                                           height: self.view.frame.size.width - 18))
        UIImage(named: "cloudes-sw.png")?.drawAsPattern(in: self.view.bounds)
        let image : UIImage = UIGraphicsGetImageFromCurrentImageContext()!
        UIGraphicsEndImageContext()
        ViewControllerOutlet.backgroundColor = UIColor(patternImage: image)
        
        self.navigationItem.title = "Directly Connected Device Sample"
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        
        do {
            try dcdSample = DirectlyConnectedDeviceSample(
                path:Provisioning.getProvisioningFilePath(),
                password:Provisioning.getProvisioningPassword(),
                updateInterval:UPDATE_INTERVAL,
                usePolicy:Provisioning.getWithPolicies(),
                updateCallback: updateGui,
                errorCallback: errorCallback,
                displayCallback: display)
            
            dcdSample?.start()
        
        } catch {
            showError(controller: self, alertTitle: "DirectlyConnectedDeviceSample startup failure",
                      message: "\(error)", actionTitle: "OK",
                      handler: { (action: UIAlertAction?) -> () in
                // Ensure that user can re-provision the device
                removeLastSettingsFile()
                self.navigationController?.popToRootViewController(animated: false)
            })
        }
    }
    
    override func viewDidAppear(_ animated: Bool) {
        if dcdSample != nil {
            self.navigationItem.setHidesBackButton(true, animated: false)
        }
        super.viewDidAppear(animated)
    }

    // Close the data connector and all used resources,
    // directly connected device, tam and communication channel
    static func terminate() {
        
    }

    private func display(text:String) {
        DispatchQueue.main.async(execute: {
            self.outputTextViewOutlet.text =
                self.outputTextViewOutlet.text + "\n" + text
            // Force text field to (invisibly) scroll until latest line is at bottom
            let bottom: NSRange = NSMakeRange(self.outputTextViewOutlet.text.count - 10, 1)
            self.outputTextViewOutlet.scrollRangeToVisible((bottom))
        })
    }
    
    private func errorCallback(str:String) {
        showError(controller: self, alertTitle: "Library Error",
                  message: str, actionTitle: "OK",
                  handler: { (action: UIAlertAction?) -> () in })
    }

    /*
     * We know there is only one HumiditySensor and one TemperatureSensor
     * The humidity data is obtained with key "humidity" as (String, Int, Int)
     * (endpointId,humidity,maxThreshold)
     * and temperature data is obtained with key "temperature"
     * as (String, Int64, Double, Double, Double, Int, Int)
     * (endpointId,startTime,temperature,minTemp,maxTemp,minThreshold,maxThreshold)
     */
    func updateGui(sensorData:[String:Any]) {
        
        // TODO: This should be done with discrete labels, not as a single
        // string in one label. This will allow updating the humidity or
        // temperature sensor independently

        if let anyData = sensorData["humidity"] {
            
            let hData:(endpointId:String,humidity:Int,maxThreshold:Int) =
                anyData as! (String,Int,Int)
            
            humidityIdOutlet.text = hData.endpointId
            humidityOutlet.text = hData.humidity.description + " %"
            humidityThresholdOutlet.text = hData.maxThreshold.description + " %"
        }
    }

    // MARK: Actions
    
    // Action for the button "Reset"
    @IBAction func onResetButton(_ sender: UIButton) {
        dcdSample?.stop()
        // stop the loops
        outputTextViewOutlet.text = ""
        // Allow reprovisioning
        removeLastSettingsFile()
        self.navigationController?.popToRootViewController(animated: false)
        
    }
    
    // Action for the button "Stop"
    @IBAction func onStopButton(_ sender: UIButton) {
        dcdSample?.stop()
        outputTextViewOutlet.text = ""
        saveSettingsFile(path: URL(fileURLWithPath: Provisioning.getProvisioningFilePath()).lastPathComponent,
            password: String(Provisioning.getProvisioningPassword()))
        exit(0)
    }
    
    override func didReceiveMemoryWarning() {
        super.didReceiveMemoryWarning()
        // Dispose of any resources that can be recreated.
    }
}
