/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

/**
 * @file iotcs_port_thread.h
 * @brief The file contains porting layer for threads.
 */

#ifndef IOTCS_PORT_THREAD_H
#define	IOTCS_PORT_THREAD_H


#ifdef	__cplusplus
extern "C" {
#endif

#include <stdint.h>
#include "iotcs_config.h"

    /**
     * @typedef iotcs_port_thread_func
     * @brief pointer to thread function. 
     */
    typedef void *(*iotcs_port_thread_func) (void *);

    /**
     * @typedef iotcs_port_thread
     * @brief thread handle. 
     */
    typedef void* iotcs_port_thread;

    /**
     * @brief Creates and starts iotcs_port_thread instance
     * @note Optional API. Called by the Library if IOTCS_MESSAGING_THREAD_SAFETY option is defined.
     * @param func_thread function
     * @return iotcs_port_thread on success
     * @return NULL if failed
     */
    iotcs_port_thread iotcs_port_thread_create(iotcs_port_thread_func func_ptr);
 
    /**
     * @brief Creates and starts iotcs_port_thread instance small stack size
     * @note Optional API. Called by the Library if IOTCS_MESSAGING_THREAD_SAFETY option is defined.
     * @param func_thread function
     * @return iotcs_port_thread on success
     * @return NULL if failed
     */
    iotcs_port_thread iotcs_port_thread_create_small_stack(iotcs_port_thread_func func_ptr);

    /**
     * @brief Joins iotcs_port_thread instance
     * If a \a thread is NULL, no operation is performed.
     * @note Optional API. Called by the Library if IOTCS_MESSAGING_THREAD_SAFETY option is defined.
     * @param thread a thread handle
     */
    void iotcs_port_thread_join(iotcs_port_thread thread);

    /**
     * @brief Puts current thread into sleep for a given time
     * @note Optional API. Called by the Library if IOTCS_MESSAGING_THREAD_SAFETY option is defined.
     * @param timeout_ms an interval in milliseconds
     */
    int iotcs_port_sleep_millis(int32_t timeout_ms);

    /**
     * @brief Releases memory that was allocated automatically for new threads
     * Called in thread context before thread finish execution.
     * This could be used to release automatically allocated thread local
     * resources, like openSSL does. In most cases this method could be NO-OP.
     * @note Optional API. Called by the Library if IOTCS_MESSAGING_THREAD_SAFETY option is defined.
     */
    void iotcs_port_thread_cleanup(void);
#ifdef	__cplusplus
}
#endif

#endif	/* IOTCS_PORT_THREAD_H */

