/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

#include <AbstractVirtualDevice.hpp>
#include <Alert.hpp>
#include <string>

/* include iot cs device model APIs */
#include "iotcs_virtual_device.h"
#include "advanced/iotcs_message.h"
#include "device_model/device_model_private_ext.h"
#include <messaging/AlertImpl.hpp>
#include <util/CPPUtils.hpp>
#include <scs/StorageObjectImpl.hpp>
#include <device_model/ErrorEventImpl.hpp>
#include <iotcs/VirtualDeviceImpl.hpp>

#ifdef __MBED__
#include "mbed.h"
#define dynamic_cast static_cast
#endif

using namespace std;

namespace iotdcl {
    MUTEX_TYPE Alert::mapMutex;
    map<void*, std::pair<VirtualDevice*, ErrorCallback*> > Alert::onErrorCallbacks;

    Alert::Alert(VirtualDevice* device, iotcs_virtual_device_handle handler, const string &alertName) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_virtual_device_create_alert_handle(handler, alertName.c_str(), &this->handle);
        this->device = device;
        checkResult(rv);
    }

    Alert::~Alert() {
        mapMutex.lock();
        {
            map<void*, std::pair<VirtualDevice*, ErrorCallback*> >::iterator it = onErrorCallbacks.find(handle);
            if (it != onErrorCallbacks.end()) {
                onErrorCallbacks.erase(it);
            }
        }
        mapMutex.unlock();
        iotcs_virtual_device_free_alert_handle(handle);
    }

    template <>
    Alert& Alert::set(const string &attributeName, const int &value) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_set_integer(handle, attributeName.c_str(), value);
        checkResult(rv);
        return *this;
    }

    template <>
    Alert& Alert::set(const string &attributeName, const float &value) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_set_float(handle, attributeName.c_str(), value);
        checkResult(rv);
        return *this;
    }

    template <>
    Alert& Alert::set(const string &attributeName, const double &value) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_set_float(handle, attributeName.c_str(), (float)value);
        checkResult(rv);
        return *this;
    }

    template <>
    Alert& Alert::set(const string &attributeName, const string &value) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_set_string(handle, attributeName.c_str(), value.c_str());
        checkResult(rv);
        return *this;
    }

    template <>
    Alert& Alert::set(const string &attributeName, const char * const &value) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_set_string(handle, attributeName.c_str(), value);
        checkResult(rv);
        return *this;
    }

    template <>
    Alert& Alert::set(const string &attributeName, const bool &value) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_set_boolean(handle, attributeName.c_str(), value ? IOTCS_TRUE : IOTCS_FALSE);
        checkResult(rv);
        return *this;
    }

    template <>
    Alert& Alert::set(const string &attributeName, const time_t &value) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_set_date_time(handle, attributeName.c_str(), (iotcs_date_time) value);
        checkResult(rv);
        return *this;
    }

    template <>
    Alert& Alert::set(const string &attributeName, const StorageObject& value)  throw(std::invalid_argument) {
        const StorageObjectImpl *ptr(dynamic_cast<const StorageObjectImpl*>(&value));
        iotcs_result rv = IOTCS_RESULT_INVALID_ARGUMENT;
        if (ptr) {
            rv = iotcs_alert_set_storage_object(handle, attributeName.c_str(), ptr->c_handler());
        }
        checkResult(rv);
        return *this;
    }

    void Alert::raise(void) throw(std::invalid_argument) {
        iotcs_result rv = iotcs_alert_raise(handle);
        checkResult(rv);
    }

    void Alert::setOnError(ErrorCallback *callback) {
        mapMutex.lock();
        if (callback) {
            pair<VirtualDevice*, ErrorCallback*> rv = onErrorCallbacks[handle];
            onErrorCallbacks[handle] = make_pair(device, callback);
            iotcs_alert_set_on_error_internal(handle, Alert::onErrorCallback);
        } else {
            iotcs_alert_set_on_error_internal(handle, NULL);
            {
                map<void*, std::pair<VirtualDevice*, ErrorCallback*> >::iterator it = onErrorCallbacks.find(handle);
                if (it != onErrorCallbacks.end()) {
                    onErrorCallbacks.erase(it);
                }
            }
        }
        mapMutex.unlock();
    }

    void Alert::onErrorCallback(void* handler, iotcs_virtual_device_error_event *event) {
        mapMutex.lock();
        map<void*, std::pair<VirtualDevice*, ErrorCallback*> >::const_iterator it = onErrorCallbacks.find(handler);
        if (it != onErrorCallbacks.end()) {
            string name(event->named_value.name);
            NamedValue value(event->named_value);
            try {
                string message = string(event->message);
                ErrorEventImpl evt(it->second.first, &value, &message);
                if (it->second.second) {
                    it->second.second->onError(evt);
                }
            } catch (...) {
                // ignored
            }
        }
        
        mapMutex.unlock();
    }

    Alert* AlertImpl::createAlert(VirtualDevice* device, iotcs_virtual_device_handle handler, const string &alertName) {
        return new AlertImpl(device, handler, alertName);
    }

    AlertImpl::AlertImpl(VirtualDevice* device, iotcs_virtual_device_handle handler, const string &alertName) : Alert(device, handler, alertName) {
    }
};
