/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
#include <iostream>
#include <memory>
#include <exception>
#include <map>
#include <vector>
#include <signal.h>
#include <stdio.h>
#include <string>
#include <thread>
 
#include <assert.h>
#include <math.h>
 
#include <VirtualDevice.hpp>
#include <DeviceModel.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <NamedValue.hpp>
#include <Exception.hpp>
#include <StorageObject.hpp>
#include "HumiditySensor.hpp"
#include "Callbacks.hpp"
 
#include <unistd.h>
 
void sleep_sec(unsigned seconds)
{
    sleep(seconds);
}
 
using namespace iotdcl;
using namespace std;
 
#define MOTION_ACTIVATED_CAMERA_MODEL_URN     "urn:com:oracle:iot:device:motion_activated_camera"
#define MOTION_ACTIVATED_CAMERA_MODEL_REC_URN "urn:com:oracle:iot:device:motion_activated_camera:recording"
#define IMAGE_ATTRIBUTE                       "image"
#define IMAGE_TIME_ATTRIBUTE                  "imageTime"
#define SENSOR_POLLING_INTERVAL           5000
#define SLEEP_TIME                        100
static long number_of_loops = (SLEEP_TIME > SENSOR_POLLING_INTERVAL ? 1 :
                                           SENSOR_POLLING_INTERVAL / SLEEP_TIME); 
static long sleepTime = 
        (SLEEP_TIME > SENSOR_POLLING_INTERVAL ?
         SENSOR_POLLING_INTERVAL :
         SLEEP_TIME + (SENSOR_POLLING_INTERVAL - number_of_loops * SLEEP_TIME) / number_of_loops);          
 
static bool exiting = false;
 
// This callback will be set on a StorageObject when the client library synchronizes a
// StorageObject. The callback simply prints out the
// status of synchronizing the content with the Oracle Storage Cloud Service.
// The same callback is used for all of the StorageObject instances.
 
class ObjectSyncCallback : public SyncCallback {
    public:
        virtual void onSync(const SyncEvent<VirtualDevice>& event) const {
            StorageObject *storageObject = event.getSource();
            cout << "onSync : " << storageObject->getName() << "=\""
                 << storageObject->getSyncStatus() << "\"";
            cout << endl;
        }
};
static ObjectSyncCallback syncCallback;
 
static string const imagePaths[] = {
    "images/Frame-00.jpeg",
    "images/Frame-01.jpeg",
    "images/Frame-02.jpeg",
    "images/Frame-03.jpeg",
    "images/Frame-04.jpeg",
    "images/Frame-05.jpeg",
    "images/Frame-06.jpeg",
    "images/Frame-07.jpeg",
    "images/Frame-08.jpeg",
    "images/Frame-09.jpeg",
    "images/Frame-10.jpeg",
    "images/Frame-11.jpeg"
};
 
static string const videoPaths[] = {
    "videos/stopwatch_015s.mp4",
    "videos/stopwatch_030s.mp4",
    "videos/stopwatch_045s.mp4",
    "videos/stopwatch_060s.mp4",
    "videos/stopwatch_075s.mp4",
    "videos/stopwatch_090s.mp4",
    "videos/stopwatch_105s.mp4",
    "videos/stopwatch_120s.mp4",
    "videos/stopwatch_135s.mp4",
    "videos/stopwatch_150s.mp4",
    "videos/stopwatch_165s.mp4",
    "videos/stopwatch_180s.mp4",
};
 
class DataErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "Data error occurred: " << event.getMessage();
        }
};
DataErrorHandler dataErrorHandler;
 
static DirectlyConnectedDevice *directlyConnectedDevice;

typedef struct {
    int64_t start_time;
    int duration;
    VirtualDevice* virtual_device;
} upload_args;
 
void video_uploader(upload_args *event) {
    int d = event->duration;
    time_t startTime =(time_t)event->start_time;
    if (d <= 0 || startTime == 0) return;
 
    // round to nearest increment of 15
    int duration = (d + 14) / 15 * 15;
    // assumes videos.length > 0 and videos are 15 second increments.
    string videoPath = videoPaths[(duration / 15 - 1) < 11 ? duration / 15 - 1 : 11];
    cout << event->virtual_device->getEndpointId() << " : Call : action : duration=" << d
                                                     << ", startTime=" << startTime << endl;
    /*
     * The name of video will be automatically
     * prefixed with the device's client ID and "/"
     */
    StorageObject& storageObject =
        directlyConnectedDevice->createStorageObject(
            videoPath, "video/mp4");
    storageObject.setOnSync(&syncCallback);
    storageObject.setInputPath(videoPath);
    storageObject.sync();
 
    cout << event->virtual_device->getEndpointId() << " : Set : \""
         << "recording" << "\"=" << storageObject.getURI() << endl;
 
    Data &recording = event->virtual_device->createData(MOTION_ACTIVATED_CAMERA_MODEL_REC_URN);
    recording.set("video", storageObject)
            .set("startTime", (time_t)startTime)
            .set("duration", duration)
            .submit();
    delete event;
}

class CameraOnRecord : public Callable {
    public:
        virtual void call(VirtualDevice *virtualDevice, const NamedValue &data) const {
            int d = data.getValue<int>();
            if (d <= 0) return;
 
            // round to nearest increment of 15
            int duration = (d + 14) / 15 * 15;
            // assumes videos.length > 0 and videos are 15 second increments.

            upload_args *args = new upload_args;
            args->start_time = (int64_t)((int64_t) time(NULL)) * 1000;
            args->duration = data.getValue<int>();
            args->virtual_device = virtualDevice;
            std::thread thr(video_uploader, args);
            thr.join();
        }
};
static CameraOnRecord cameraOnRecord;
 
class CameraOnScheduleRecording : public ActionCallback {
    public:
        virtual void onAction(ActionEvent<VirtualDevice>& event) const {
            upload_args *args = new upload_args;
            if (event.getNamedValue()->getName() == "duration") {
                args->start_time = (int64_t)event.getNamedValue()->next()->getValue<time_t>();
                args->duration = event.getNamedValue()->getValue<int>();
            } else {
                args->start_time = (int64_t)event.getNamedValue()->getValue<time_t>();
                args->duration = event.getNamedValue()->next()->getValue<int>();
            }
            args->virtual_device = event.getVirtualDevice();
            std::thread thr(video_uploader, args);
            thr.join();
        }
};
static CameraOnScheduleRecording cameraOnScheduleRecording;
 
class ErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "ErrorHandler: " << event.getMessage() << endl;
        }
};
static ErrorHandler errorHandler;
 
/* handler for interrupt signal */
extern "C" void int_handler(int ignored) {
    (void)ignored;
    exiting = true;
}
 
int main(int argc, char** argv) {
    string urn(MOTION_ACTIVATED_CAMERA_MODEL_URN);
    if (argc < 3) {
        cerr << "Too few parameters.\n"
                "\nUsage:"
                "\n\tdirectly_connected_device.out <PATH> <PASSWORD>"
                "\n\t<PATH> is a path to trusted assets store."
                "\n\t<PASSWORD> is a password for trusted assets store."
                "\n\tNote: IOTCS_OS_NAME and IOTCS_OS_VERSION must be setted before using" << endl;
        return EXIT_FAILURE;
    }
 
    srand(time(NULL));
 
    try {
        DirectlyConnectedDevice dcd(argv[1], argv[2]);
        directlyConnectedDevice = &dcd;
        if (!dcd.isActivated()) {
            dcd.activate({urn});
        }
 
        DeviceModel& deviceModel = dcd.getDeviceModel(urn);
 
        VirtualDevice& virtualMotionActivatedCamera = dcd.createVirtualDevice(dcd.getEndpointId(), deviceModel);
        virtualMotionActivatedCamera.setCallable("record", &cameraOnRecord);
        virtualMotionActivatedCamera.setOnAction("scheduleRecording", &cameraOnScheduleRecording);
        virtualMotionActivatedCamera.setOnError(&errorHandler);
 
        cout << "\nCreated virtual motion activated camera "  << virtualMotionActivatedCamera.getEndpointId() << endl;
 
        int n = 0;
        sleep_sec(5);
        for (;;) {
 
            string imagePath = imagePaths[n++ % 12];
 
            cout << virtualMotionActivatedCamera.getEndpointId() << " : Set : \"" << IMAGE_ATTRIBUTE << "\"=" << imagePath << endl;
 
            try {
                /*
                 * The name of video will be automatically
                 * prefixed with the device's client ID and "/"
                 *
                 * If this device implements many models or
                 * this was a gateway device, adding the another
                 * level of uniqueness would be warrented.
                 */
                StorageObject& storageObject =
                    dcd.createStorageObject(
                        imagePath, "image/jpeg");
                storageObject.setOnSync(&syncCallback);
                storageObject.setInputPath(imagePath);
 
                virtualMotionActivatedCamera.update()
                                            .set(IMAGE_ATTRIBUTE, &storageObject)
                                            .set(IMAGE_TIME_ATTRIBUTE, time(NULL))
                                            .finish();
                cout << "Sleep 30 seconds!\nPress Ctrl + C to exit\n";
                sleep_sec(30);
            } catch (std::exception &e) {
                cout << "EXCEPTION CAUGHT: " << e.what() << endl;
                continue;
            }
        }
    } catch (std::exception &e) {
        cout << "EXCEPTION CAUGHT: " << e.what() << endl;
        return EXIT_FAILURE;
    }
}
