/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
#include <iostream>
#include <memory>
#include <exception>
#include <map>
#include <vector>
#include <signal.h>
#include <stdio.h>
#include <string>

#include <assert.h>
#include <math.h>

#include <VirtualDevice.hpp>
#include <GatewayDevice.hpp>
#include <DeviceModel.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <NamedValue.hpp>
#include <Exception.hpp>
#include "HumiditySensor.hpp"
#include "TemperatureSensor.hpp"
#include "Callbacks.hpp"

#include <unistd.h>

void sleep_sec(unsigned seconds)
{
    sleep(seconds); // takes microseconds
}

using namespace iotdcl;
using namespace std;

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))
#define ARRAY_LEN(X) (sizeof(X)/sizeof(X[0]))

/*
 *                           Constants & Variables
 */
#ifndef USE_POLICY
#define USE_POLICY false
#endif

/* used as sending loop condition */
static volatile int keep_running = 1;

static const string temperatureSensorUrn("urn:com:oracle:iot:device:temperature_sensor");
static const string humiditySensorUrn("urn:com:oracle:iot:device:humidity_sensor");

/* handler for interrupt signal */
extern "C" void int_handler(int ignored) {
    (void)ignored;
    keep_running = 0;
}

/* print all humidity device attributes */
static void printHSAttributes(VirtualDevice &vd) {
    cout << "humidity attribute value = " << vd.get<int>("humidity") << endl;
    cout << "maxThreshold attribute value = " << vd.get<int>("maxThreshold") << endl;
}

/* print all temperature device attributes */
static void printTSAttributes(VirtualDevice &vd) {
    cout << "Device start time = " << vd.get<time_t>("startTime") << endl;
    cout << "Device temp value = " << vd.get<float>("temp") << endl;
    cout << "Device unit = " << vd.get<string>("unit") << endl;
    cout << "Device min temp = " << vd.get<float>("minTemp") << endl;
    cout << "Device max temp = " << vd.get<float>("maxTemp") << endl;
    cout << "Device min temp threshold = " << vd.get<int>("minThreshold") << endl;
    cout << "Device max temp threshold = " << vd.get<int>("maxThreshold") << endl;
}

class ErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "Error occurred: " << event.getMessage();
            exit(EXIT_FAILURE);
        }
};
ErrorHandler errorHandler;

class AlertErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "Error on alert occurred: " << event.getMessage();
        }
};
AlertErrorHandler alertErrorHandler;

int main(int argc, char** argv) {
    if (argc < 3) {
        cerr << "Too few parameters.\n"
                "\nUsage:"
                "\n\tdirectly_connected_device.out <PATH> <PASSWORD>"
                "\n\t<PATH> is a path to trusted assets store."
                "\n\t<PASSWORD> is a password for trusted assets store."
                "\n\tNote: IOTCS_OS_NAME and IOTCS_OS_VERSION must be setted before using";
        return EXIT_FAILURE;
    }
    
    srand(time(NULL));

    try {
        GatewayDevice gwd(argv[1], argv[2]);
        if (!gwd.isActivated()) {
            gwd.activate({});
        }
        
        DeviceModel& temperatureModel = gwd.getDeviceModel(temperatureSensorUrn);
        DeviceModel& humidityModel = gwd.getDeviceModel(humiditySensorUrn);

        TemperatureSensor temperatureSensor(gwd.getEndpointId() + "_Sample_TS");
        HumiditySensor humiditySensor(gwd.getEndpointId() + "_Sample_HS");
        const map<string, string> temperatureMeta = {
            {"manufacturer", temperatureSensor.getManufacturer()},
            {"modelNumber", temperatureSensor.getModelNumber()},
            {"serialNumber", temperatureSensor.getSerialNumber()},
            {"deviceClass", "temp sensor"},
            {"sampleAttr", "test text temp sensor"},
            {"year", "2015"}};
        const map<string, string> humidityMeta = {
            {"manufacturer", humiditySensor.getManufacturer()},
            {"modelNumber", humiditySensor.getModelNumber()},
            {"serialNumber", humiditySensor.getSerialNumber()},
            {"deviceClass", "humidity sensor"},
            {"sampleStrAttr", "test text humidity sensor"},
            {"year", "2015"}};

        const string temperatureSensorEndpointId = gwd.registerDevice(false, temperatureSensor.getHardwareId(),
                temperatureMeta, {temperatureSensorUrn});

        const string humiditySensorEndpointId = gwd.registerDevice(false, humiditySensor.getHardwareId(), humidityMeta, {humiditySensorUrn});

        /* get temperature device handle */
        VirtualDevice& temperatureDevice = gwd.createVirtualDevice(temperatureSensorEndpointId, temperatureModel);

        /* get humidity device handle */
        VirtualDevice& humidityDevice = gwd.createVirtualDevice(humiditySensorEndpointId, humidityModel);

        /* Callbacks */
        HumidityOnChange humidityOnChange(&humiditySensor);
        HumidityOnMaxThreshold humidityOnMaxThreshold(&humiditySensor);
        HumidityOnMaxThresholdError humidityOnMaxThresholdError;
        TemperatureOnChange temperatureOnChange(&temperatureSensor);
        TemperatureOnMaxThreshold temperatureOnMaxThreshold(&temperatureSensor);
        TemperatureOnMaxThresholdError temperatureOnMaxThresholdError;
        TemperatureOnReset temperatureOnReset(&temperatureSensor);
        TemperatureOnPower temperatureOnPower(&temperatureSensor);

        temperatureDevice.setOnChange(&temperatureOnChange);
        temperatureDevice.setOnError(&errorHandler);
        temperatureDevice.setOnAction("reset", &temperatureOnReset);
        temperatureDevice.setOnAction("power", &temperatureOnPower);
        humidityDevice.setOnChange(&humidityOnChange);
        humidityDevice.setOnChange("maxThreshold", &humidityOnMaxThreshold);
        humidityDevice.setOnError(&errorHandler);
        humidityDevice.setOnError("maxThreshold", &humidityOnMaxThresholdError);
        
        Alert& humidityAlert(humidityDevice.createAlert("urn:com:oracle:iot:device:humidity_sensor:too_humid"));
        Alert& temperatureHotAlert(temperatureDevice.createAlert("urn:com:oracle:iot:device:temperature_sensor:too_hot"));
        Alert& temperatureColdAlert(temperatureDevice.createAlert("urn:com:oracle:iot:device:temperature_sensor:too_cold"));

        cout << "Batch update update of temparature device attributes" << endl;

        time_t startTime = temperatureSensor.getStartTime();
        float temp = temperatureSensor.getTemp();
        float minTemp = temperatureSensor.getMinTemp();
        float maxTemp = temperatureSensor.getMaxTemp();
        int minThreshold = temperatureSensor.getMinThreshold();
        int maxThreshold = temperatureSensor.getMaxThreshold();

        temperatureDevice.update()
                            .set("startTime", startTime)
                            .set("temp", temp)
                            .set("unit", temperatureSensor.getUnit())
                            .set("minTemp", minTemp)
                            .set("maxTemp", maxTemp)
                            .set("minThreshold", minThreshold)
                            .set("maxThreshold", maxThreshold)
                            .finish();

        int humidity = humiditySensor.getHumidity();
        int humidityThreshold =
                humiditySensor.getMaxThreshold();
        cout << "Batch update update of humidity device attributes" << endl;
        humidityDevice.update()
                         .set("humidity", humidity)
                         .set("maxThreshold", humidityThreshold)
                         .finish();

        printTSAttributes(temperatureDevice);
        printHSAttributes(humidityDevice);

        /* Set handler for interrupt signal */
        signal(SIGINT, int_handler);

        bool humidityAlerted = false;
        bool tempAlerted = false;
        bool wasOff = false;
        /* 
         * Emulate device work. 
         * Sensors changed values, then device send data message with new values 
         * to the server. If sensors values is out of threshold then 
         * send alert message.
         * Async message dispatcher send messages to the server and handle 
         * message delivery and errors during send. 
         * Request dispatcher handle incoming requests from the server using 
         * custom request handlers.
         */
        while (keep_running) {
            cout << "Sleep 5 seconds!" << endl;
            sleep_sec(5);

            try {
                humidity =
                        humiditySensor.getHumidity();
                humidityThreshold =
                        humiditySensor.getMaxThreshold();

                cout << humiditySensorEndpointId +
                        " : Set : \"humidity\"=" << humidity << endl;

#if (USE_POLICY)
                /* Emulate humidity sensor work */
                humidityDevice.update();
                /* offer humidity device value */
                humidityDevice.offer("humidity", humidity);
                humidityDevice.offer("maxThreshold", humidityThreshold);
                humidityDevice.finish();
#else
                humidityDevice.set("humidity", humidity);

                if (humidity > humidityThreshold) {
                    cout << "Alert : \"humidity\" = " << humidity << ", \"maxThreshold\" = " << humidityThreshold << endl;
                    humidityAlert.setOnError(&alertErrorHandler);
                    humidityAlert.set("humidity", humidity).raise();
                }
#endif

                if (!temperatureSensor.isPoweredOn()) {
                    wasOff = true;
                    continue;
                }

                if (wasOff) {
                    wasOff = false;

                    cout << "Power back on, updating all attributes." << endl;

                    startTime =
                            temperatureSensor.getStartTime();
                    temp =
                            temperatureSensor.getTemp();
                    minTemp =
                            temperatureSensor.getMinTemp();
                    maxTemp =
                            temperatureSensor.getMaxTemp();
                    minThreshold =
                            temperatureSensor.getMinThreshold();
                    maxThreshold =
                            temperatureSensor.getMaxThreshold();

                    cout << temperatureDevice.
                                getEndpointId() + " : Set : " +
                            "\"power\"=true, " +
                            ", \"temp\"=" << temp <<
                            ", \"unit\"=" << temperatureSensor.getUnit() <<
                            ", \"minTemp\"=" << minTemp <<
                            ", \"maxTemp\"=" << maxTemp <<
                            ", \"minThreshold\"=" <<
                            minThreshold <<
                            ", \"maxThreshold\"=" <<
                            maxThreshold;

                    // After the sensor is powered on,
                    // the attribute values in the
                    // VirtualDevice need to be updated
                    // to reflect the new values.
                    temperatureDevice.update()
                            .set("startTime", startTime)
                            .set("temp", temp)
                            .set("unit", temperatureSensor.getUnit())
                            .set("minTemp", minTemp)
                            .set("maxTemp", maxTemp)
                            .set("minThreshold", minThreshold)
                            .set("maxThreshold", maxThreshold)
                            .finish();
                    continue;
                }

                time_t prevStartTime = temperatureDevice.get<time_t>("startTime");
                float prevMinTemp = temperatureDevice.get<float>("minTemp");
                float prevMaxTemp = temperatureDevice.get<float>("maxTemp");
                float temperature = temperatureSensor.getTemp();
                minTemp = temperatureSensor.getMinTemp();
                maxTemp = temperatureSensor.getMaxTemp();
                startTime = temperatureSensor.getStartTime();
                string id = temperatureSensorEndpointId;

                string sb = id + " : Set : \"temp\"=" + to_string(temperature);

#if (USE_POLICY)
                temperatureDevice.update();
                /* offer humidity device value */
                temperatureDevice.offer("temp", temperature);
                temperatureDevice.offer("minTemp", minTemp);
                temperatureDevice.offer("maxTemp", maxTemp);
                temperatureDevice.finish();
#else
                temperatureDevice.update().set(
                        "temp", temperature);

                if (minTemp != prevMinTemp) {
                    sb += ",minTemp=" + to_string(minTemp);
                    temperatureDevice.set(
                            "minTemp", minTemp);
                }

                if (maxTemp != prevMaxTemp) {
                    sb += ",maxTemp=" + to_string(maxTemp);
                    temperatureDevice.set(
                            "maxTemp", maxTemp);
                }

                if (startTime != prevStartTime) {
                    sb += ",startTime=" + to_string(startTime);
                    temperatureDevice.set(
                            "startTime", startTime);
                }
                cout << (sb) << endl;
                temperatureDevice.finish();

                int minTempThreshold = temperatureSensor.getMinThreshold();
                int maxTempThreshold = temperatureSensor.getMaxThreshold();
                if (temperature > maxTempThreshold) {
                    if (!tempAlerted) {
                        cout << "Send hot alert" << endl;
                        temperatureHotAlert.setOnError(&alertErrorHandler);
                        temperatureHotAlert.set("temp", temperature)
                                              .set("unit", temperatureSensor.getUnit())
                                              .set("maxThreshold", (float)maxTempThreshold)
                                              .raise();
                    }
                } else if (temperature < minTempThreshold) {
                    if (!tempAlerted) {
                        cout << "Send cold alert" << endl;
                        temperatureColdAlert.setOnError(&alertErrorHandler);
                        temperatureColdAlert.set("temp", temperature)
                                               .set("unit", temperatureSensor.getUnit())
                                               .set("minThreshold", (float)minTempThreshold)
                                               .raise();
                    }
                } else {
                    tempAlerted = false;
                }
#endif //USE_POLICY
                printTSAttributes(temperatureDevice);
                printHSAttributes(humidityDevice);
            } catch (std::exception &e) {
                cout << "EXCEPTION CAUGHT: " << e.what() << endl;
                continue;
            }
        } // end while
    } catch (std::exception &e) {
        cout << "EXCEPTION CAUGHT: " << e.what() << endl;
        return EXIT_FAILURE;
    }
    cout << "OK" << endl;
    return EXIT_SUCCESS;
}
