/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

#include <DeviceModel.hpp>
#include <VirtualDevice.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <Exception.hpp>

#include <mutex>
#include <string>
#include <vector>

#pragma once

using namespace iotdcl;
using namespace std;

#ifndef M_PI
#define M_PI 3.14159265358979323846264338327950288
#endif
/* The max range for humidity value */
#define VMAX 100
/* The min range for humidity value */
#define VMIN  0

class HumiditySensor {
    private:
        /* Used to generate fluctuation around the 'set point' */
        int angle = 0;
        /* Current humidity sensor max threshold */
        int maxThreshold = 80;
        /* The HumiditySensor simulates humidity data by generating values that
         * fluctuate around the 'set point', arbitrarily set to 75% humidity. */
        const int setPoint = 75;
        /* Amplitude is the maximum deviation from the 'set point' */
        int amplitude = VMAX - setPoint;
        /* A unique identifier for this sensor */
        std::string hardwareId;

        std::mutex mtx;
    public:
        HumiditySensor(std::string hardwareId) {
            this->hardwareId = hardwareId;
        }

        /* a simple implementation of the humidity sensor work */
        int getHumidity() {
            double delta = amplitude * sin(angle * M_PI / 180);
            angle += 15;
            return round(setPoint + delta);
        }

        /**
         * Set the maximum percent humidity threshold for alerts. The value is
         * clamped to the range [60..100] and represents a whole percentage.
         * @param threshold a value between 60 and 100
         */
        void setMaxThreshold(int threshold) {
            mtx.lock();
            if (threshold < 60) maxThreshold = 60;
            else if (threshold > 100) maxThreshold = 100;
            else maxThreshold = threshold;
            mtx.unlock();
        }

        /**
         * Get the maximum threshold value.
         * @return the maximum threshold
         */
        int getMaxThreshold() {
            return maxThreshold;
        }
        /**
         * Get the manufacturer name, which can be used as part of the
         * device meta-data.
         * @return the manufacturer name
         */
        std::string getManufacturer() {
            return "Sample";
        }

        /**
         * Get the model number, which can be used as part of the
         * device meta-data.
         * @return the model number
         */
        std::string getModelNumber() {
            return "MN-" + hardwareId;
        }

        /**
         * Get the serial number, which can be used as part of the
         * device meta-data.
         * @return the serial number
         */
        std::string getSerialNumber() {
            return "SN-" + hardwareId;
        }

        /**
         * Get the hardware id, which can be used as part of the
         * device meta-data.
         * @return the hardware id
         */
        std::string getHardwareId() {
            return hardwareId;
        }

};
