/*
 * Copyright (c) 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

#include <DeviceModel.hpp>
#include <VirtualDevice.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <Exception.hpp>

#include <mutex>
#include <string>
#include <vector>

#pragma once

using namespace iotdcl;
using namespace std;

#ifndef M_PI
#define M_PI 3.14159265358979323846264338327950288
#endif

class TemperatureSensor {
    private:
        bool powerOn;
        // The time (measured in EPOCH) at which the system was powered ON or reset
        time_t startTime;

        // The minimum value measured by the sensor since power ON or reset
        float minTemp = std::numeric_limits<float>::max();

        // The minimum value measured by the sensor since power ON or reset
        float maxTemp = std::numeric_limits<float>::min();

        // The minimum threshold is in the range [-20,0]
        int minThreshold = 0;

        // The maximum threshold is in the range [65,80]
        int maxThreshold = 70;
        /* Used to generate fluctuation around the 'set point' */
        int angle = 0;
        /* The TemperatureSensor simulates Temperature data by generating values that
         * fluctuate around the 'set point', arbitrarily set to 75% Temperature. */
        const int setPoint = 65;
        /* Amplitude is the maximum deviation from the 'set point' */
        float amplitude = maxThreshold - setPoint + 0.5;
        /* A unique identifier for this sensor */
        string hardwareId;

        mutex mtx;

        const char* unit = "Celsius";

        static float calculateTemp(int setPoint, double amplitude, int angle) {
            double delta = amplitude * sin(angle * M_PI / 180);
            // normal temp sensors don't have readings such as 21.890890309842234
            long l = (long) ((setPoint + delta) * 100.0);
            return (float) l / 100;
        }
    public:
        /**
         * An action to reset the miniumum and maximum measured temperature to
         * the current value.
         */
        void reset() {
            maxTemp = minTemp = calculateTemp(setPoint, amplitude, angle);
            startTime = time(NULL);
        }

        /**
         * Create a TemperatureSensor
         * @param id a unique identifier for this sensor
         */
        TemperatureSensor(std::string hardwareId) {
            this->hardwareId = hardwareId;
            powerOn = true;
            reset();
        }

        /**
         * Set the power on or off.
         * @param on {@code true} to set power on, {@code false} to set power off
         */
        void power(bool on) {
            if (on) {
                if (powerOn) {
                    return;
                }

                powerOn = true;
                reset();
                return;
            }

            powerOn = false;
        }

        /**
         * Get whether or not the sensor is powered on.
         * @return {@code true}, if the sensor is powered on.
         */
        bool isPoweredOn() {
            return powerOn;
        }

        /**
         * Get the current temperature value.
         * <p>
         * The simulated readings will fluctuate +-2 degrees.
         *
         * @return the temperature value
         */
        float getTemp() {
            double temp = calculateTemp(setPoint, amplitude, angle);
            angle += 15;

            if (temp < minTemp) {
                minTemp = temp;
            }

            if (temp > maxTemp) {
                maxTemp = temp;
            }

            return temp;
        }

        /**
         * Get the temperature units (scale)
         * @return the temperature units
         */
        const char* getUnit() {
            return unit;
        }

        /**
         * Get the minimum temperature since the last power on or reset.
         * @return the minimum temperature since the last power on or reset
         */
        float getMinTemp() {
            return minTemp;
        }

        /**
         * Get the maximum temperature since the last power on or reset.
         * @return the maximum temperature since the last power on or reset
         */
        float getMaxTemp() {
            return maxTemp;
        }

        /**
         * Get the minumum threshold value.
         * @return the minimum threshold
         */
        int getMinThreshold() {
            return minThreshold;
        }

        /**
         * Set the minimum degrees Celsius threshold for alerts. The value is
         * clamped to the range [-20..0].
         * @param threshold a value between -20 and 0
         */
        void setMinThreshold(int threshold) {
            mtx.lock();
            if (threshold < -20) minThreshold = -20;
            else if (threshold > 0) minThreshold = 0;
            else minThreshold = threshold;
            mtx.unlock();
        }

        /**
         * Get the maximum threshold value.
         * @return the maximum threshold
         */
        int getMaxThreshold() {
            return maxThreshold;
        }

        /**
         * Set the maximum degrees Celsius threshold for alerts. The value is
         * clamped to the range [65..80].
         * @param threshold a value between 65 and 80
         */
        void setMaxThreshold(int threshold) {
            mtx.lock();
            if (threshold < 65) maxThreshold = 65;
            else if (threshold > 80) maxThreshold = 100;
            else maxThreshold = threshold;
            mtx.unlock();
        }

        /**
         * Get the time at which the device was powered on or reset.
         * @return the device start time
         */
        time_t getStartTime() {
            return startTime;
        }

        /**
         * Get the manufacturer name, which can be used as part of the
         * device meta-data.
         * @return the manufacturer name
         */
        string getManufacturer() {
            return "Sample";
        }

        /**
         * Get the model number, which can be used as part of the
         * device meta-data.
         * @return the model number
         */
        string getModelNumber() {
            return "MN-" + hardwareId;
        }

        /**
         * Get the serial number, which can be used as part of the
         * device meta-data.
         * @return the serial number
         */
        string getSerialNumber() {
            return "SN-" + hardwareId;
        }

        /**
         * Get the hardware id, which can be used as part of the
         * device meta-data.
         * @return the hardware id
         */
        string getHardwareId() {
            return hardwareId;
        }
};
