/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
#include <iostream>
#include <memory>
#include <exception>
#include <map>
#include <vector>
#include <signal.h>
#include <stdio.h>
#include <string>

#include <assert.h>
#include <math.h>

#include <VirtualDevice.hpp>
#include <DeviceModel.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <NamedValue.hpp>
#include <Exception.hpp>
#include "HumiditySensor.hpp"
#include "Callbacks.hpp"

/* used for humidity sensor work simulation */
#define HUMIDITY_SENSOR_MODEL_URN            "urn:com:oracle:iot:device:humidity_sensor"
#define HUMIDITY_ATTRIBUTE                   "humidity"
#define MAX_THRESHOLD_ATTRIBUTE              "maxThreshold"
#define TOO_HUMID_ALERT                      "urn:com:oracle:iot:device:humidity_sensor:too_humid"

#ifdef _WIN32
    #include <windows.h>

    void sleep_sec(unsigned seconds)
    {
        Sleep(seconds * 1000); // takes milliseconds
    }
#else
    #include <unistd.h>

    void sleep_sec(unsigned seconds)
    {
        sleep(seconds);
    }
#endif

using namespace iotdcl;
using namespace std;

/*
 *                           Constants & Variables
 */
/* used as sending loop condition */
static volatile int keep_running = 1;

#ifndef USE_POLICY
#define USE_POLICY false
#endif

/* handler for interrupt signal */
extern "C" void int_handler(int ignored) {
    (void)ignored;
    keep_running = 0;
}

class ErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "ErrorHandler: " << event.getMessage() << endl;
            exit(EXIT_FAILURE);
        }
};
ErrorHandler errorHandler;

int main(int argc, char** argv) {
    string urn(HUMIDITY_SENSOR_MODEL_URN);
    if (argc < 3) {
        cerr << "Too few parameters.\n"
                "\nUsage:"
                "\n\tdirectly_connected_device.out <PATH> <PASSWORD>"
                "\n\t<PATH> is a path to trusted assets store."
                "\n\t<PASSWORD> is a password for trusted assets store."
                "\n\tNote: IOTCS_OS_NAME and IOTCS_OS_VERSION must be setted before using";
        return EXIT_FAILURE;
    }
    
    srand(time(NULL));
    
    try {
        DirectlyConnectedDevice dcd(argv[1], argv[2]);
        if (!dcd.isActivated()) {
            dcd.activate({urn});
        }

        DeviceModel& humidityModel = dcd.getDeviceModel(urn);

        VirtualDevice& humidityDevice = dcd.createVirtualDevice(dcd.getEndpointId(), humidityModel);
        HumiditySensor sensor(dcd.getEndpointId() + "_Sample_HS");

        /* Callbacks */
        HumidityOnChange humidityOnChange(&sensor);
        HumidityOnMaxThreshold humidityOnMaxThreshold(&sensor);
        HumidityOnMaxThresholdError humidityOnMaxThresholdError;

        Alert& humidityAlert = humidityDevice.createAlert("urn:com:oracle:iot:device:humidity_sensor:too_humid");
        humidityDevice.setOnChange(&humidityOnChange);
        humidityDevice.setOnChange("maxThreshold", &humidityOnMaxThreshold);
        humidityDevice.setOnError(&errorHandler);
        humidityDevice.setOnError("maxThreshold", &humidityOnMaxThresholdError);
     
        // Initialize the sensor to the device model's maxThreshold default value
        int defaultThreshold = humidityDevice.get<int>("maxThreshold");
        sensor.setMaxThreshold(defaultThreshold);
        // The previous maximum humidity threshold.
        // If the new maximum humidity threshold is not equal to the previous,
        // update "maxThreshold" in the virtual HumiditySensor
        int prevMaxHumidityThreshold = sensor.getMaxThreshold();    
        cout << "Batch update update of humidity sensor attributes\n";
        humidityDevice.update()
                         .set("humidity", sensor.getHumidity())
                         .set("maxThreshold", sensor.getMaxThreshold())
                         .finish();
        cout << "humidity value = " << humidityDevice.get<int>("humidity") << endl;
        cout << "maxThreshold value = " << humidityDevice.get<int>("maxThreshold") << endl;
        bool humidityAlerted = false;

        /* Set handler for interrupt signal */
        signal(SIGINT, int_handler);

        /* 
         * Emulate device work. 
         * Sensors changed values, then device send data message with new values 
         * to the server. If sensors values is out of threshold then 
         * send alert message.
         * Async message dispatcher send messages to the server and handle 
         * message delivery and errors during send. 
         * Request dispatcher handle incoming requests from the server using 
         * custom request handlers.
         */
        while (keep_running) {
            cout << "Sleep 5 seconds!" << endl;
            sleep_sec(5);
            try {
                int humidity = sensor.getHumidity();
                int humidityThreshold = sensor.getMaxThreshold();

                cout << dcd.getEndpointId() +
                        " : Set : \"humidity\"=" << humidity << endl;
#if (USE_POLICY)
                /* Emulate humidity sensor work */
                humidityDevice.update();
                /* offer humidity device value */
                humidityDevice.offer("humidity", humidity);
                humidityDevice.offer("maxThreshold", humidityThreshold);
                humidityDevice.finish();
#else
                /* Emulate humidity sensor work */
                humidityDevice.update();
                /* set humidity device value */
                humidityDevice.set("humidity", humidity);
                // Send humidityThreshold if it has changed or is not the default
                if (prevMaxHumidityThreshold != humidityThreshold) {
                    prevMaxHumidityThreshold = humidityThreshold;

                    humidityDevice.set("maxThreshold", humidityThreshold);
                }
                humidityDevice.finish();

                if (humidity > humidityThreshold) {
                    cout << "Alert : \"humidity\" = " << humidity << ", \"maxThreshold\" = " << humidityThreshold << endl;
                    humidityAlert.set("humidity", humidity).raise();
                }
#endif
                /* get humidity device value */
                cout << "humidity value = " << humidityDevice.get<int>("humidity") << endl;
                cout << "maxThreshold value = " << humidityDevice.get<int>("maxThreshold") << endl;
            } catch (std::exception &e) {
                cout << "EXCEPTION CAUGHT: " << e.what() << endl;
                continue;
            }
        } // end while
    } catch (std::exception &e) {
        cout << "EXCEPTION CAUGHT: " << e.what() << endl;
        return EXIT_FAILURE;
    }
    cout << "OK\n";
    return EXIT_SUCCESS;
}
