/*
 * Copyright (c) 2015, 2017, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

/**
 * @file iotcs_port_system.h
 * @brief The file contains porting layer for system methods.
 */

#ifndef IOTCS_PORT_SYSTEM_H
#define	IOTCS_PORT_SYSTEM_H


#ifdef	__cplusplus
extern "C" {
#endif

#include <inttypes.h>
#include "iotcs_config.h"

#define IOTCS_SEC_TO_MILLISEC(sec) (((int64_t) (sec)) * 1000)

#define IOTCS_MILLISEC_TO_SEC(msec) ((msec) / 1000)
#define IOTCS_MILLISEC_TO_NANOSEC(msec) ((msec) * 1000000)
    
#define IOTCS_MICROSEC_TO_MILLISEC(microsec) ((microsec) / 1000)
#define IOTCS_MICROSEC_TO_NANOSEC(microsec) ((microsec) * 1000)
    
#define IOTCS_NANOSEC_TO_MILLISEC(nanosec) ((nanosec) / 1000000)
   
    /**
     * @brief Initializes the platform
     * Library calls this method during library's initialization and before
     * it starts using any port API.
     * @note Mandatory API. Called by the Library in any configuration.
     */
    void iotcs_port_platform_init();

    /**
     * @brief Finalizes the platform
     * Library calls this method during library's finalization and after
     * it stopped using port API.
     * @note Mandatory API. Called by the Library in any configuration.
     */
    void iotcs_port_platform_finilize();

    /**
     * @brief Number of milliseconds since Epoch
     * @note Mandatory API. Called by the Library in any configuration.
     * @return Number of milliseconds since Epoch
     */
    int64_t iotcs_port_get_current_time_millis(void);

    /**
     * @brief Sets current server time in seconds since Epoch
     * When IOTCS_USE_SERVER_TIME is enabled and HTTPS protocol is used and
     * request to get access token failed (probably because of time difference
     * between device and server) then, library parses server response, extracts
     * server time from it and passes result to this call.
     * 
     * Implementation doesn't have to adjust hardware RTC, it could just
     * calculate a time difference and adjust return value of
     * iotcs_port_get_current_time_millis call by the difference.
     * @note Optional API. Called by the Library if IOTCS_USE_SERVER_TIME option is defined.
     * @param seconds current server time in seconds since Epoch
     */
    void iotcs_port_set_current_time(int64_t seconds);

    /**
     * @brief Returns null terminated string with operation system name
     * Used in /iot/api/v2/activation/policy?OSName=XXX&OSVersion=YYY REST API
     * call in place of XXX. For more info please read REST API documentation.
     * @note Mandatory API. Called by the Library in any configuration.
     * @return null terminated string with operation system name
     */
    const char *iotcs_port_get_os_name(void);

    /**
     * @brief Returns null terminated string with operation system version
     * Used in /iot/api/v2/activation/policy?OSName=XXX&OSVersion=YYY REST API
     * call in place of YYY. For more info please read REST API documentation.
     * @note Mandatory API. Called by the Library in any configuration.
     * @return null terminated string with operation system version
     */
    const char *iotcs_port_get_os_version(void);

    /**
     * @brief Returns null terminated string like "proxy_host:proxy_port"
     * @note Mandatory API. Called by the Library in any connection.
     * @return null terminated string with operation system version
     */
    const char *iotcs_port_get_http_proxy(void);


#ifdef	__cplusplus
}
#endif

#endif	/* IOTCS_PORT_SYSTEM_H */

