/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
#include "mbed.h"
#include "EthernetInterface.h"
#include "NTPClient.h"
#include "SDFileSystem.h"
#include "cmsis_os.h"
#include "rtos.h"
#include <iostream>
#include <memory>
#include <exception>
#include <map>
#include <vector>
#include <signal.h>
#include <stdio.h>
#include <string>
#include <thread>
 
#include <assert.h>
#include <math.h>
 
#include <VirtualDevice.hpp>
#include <DeviceModel.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <NamedValue.hpp>
#include <Exception.hpp>
#include <StorageObject.hpp>
#include "HumiditySensor.hpp"
#include "Callbacks.hpp"
 
#ifndef IOTCS_MBED_TS_PASSWORD
#define IOTCS_MBED_TS_PASSWORD changeit\0
#endif // #ifndef IOTCS_MBED_TS_PASSWORD
 
#ifndef IOTCS_MBED_TS_PATH
#define IOTCS_MBED_TS_PATH /sd/trusted_assets_store\0
#endif // #ifndef IOTCS_MBED_TS_PATH
 
#define __STRINGIFY(x) #x
#define STRINGIFY(x) __STRINGIFY(x)
 
#define CMD_SET_TIME   (0x73)  // 's'
 
Serial pc(USBTX, USBRX);
EthernetInterface eth;
NTPClient ntp;
 
DigitalOut led_red(LED_RED);
DigitalOut led_green(LED_GREEN);
DigitalOut led_blue(LED_BLUE);
 
char *mbed_ts_path = STRINGIFY(IOTCS_MBED_TS_PATH);
char *mbed_ts_password = STRINGIFY(IOTCS_MBED_TS_PASSWORD);
 
#ifndef IOTCS_SDCARD_MOUNT_NAME
#define IOTCS_SDCARD_MOUNT_NAME sd\0
#endif // #ifndef IOTCS_SDCARD_MOUNT_NAME
// mount sd card with IOTCS_SDCARD_MOUNT_NAME name
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, STRINGIFY(IOTCS_SDCARD_MOUNT_NAME)); // MOSI, MISO, SCK, CS
using namespace iotdcl;
using namespace std;
 
#define MOTION_ACTIVATED_CAMERA_MODEL_URN     "urn:com:oracle:iot:device:motion_activated_camera"
#define MOTION_ACTIVATED_CAMERA_MODEL_REC_URN "urn:com:oracle:iot:device:motion_activated_camera:recording"
#define IMAGE_ATTRIBUTE                       "image"
#define IMAGE_TIME_ATTRIBUTE                  "imageTime"
#define SENSOR_POLLING_INTERVAL           5000
#define SLEEP_TIME                        100
static long number_of_loops = (SLEEP_TIME > SENSOR_POLLING_INTERVAL ? 1 :
                                           SENSOR_POLLING_INTERVAL / SLEEP_TIME); 
static long sleepTime = 
        (SLEEP_TIME > SENSOR_POLLING_INTERVAL ?
         SENSOR_POLLING_INTERVAL :
         SLEEP_TIME + (SENSOR_POLLING_INTERVAL - number_of_loops * SLEEP_TIME) / number_of_loops);          
 
static bool exiting = false;
 
// This callback will be set on a StorageObject when the client library synchronizes a
// StorageObject. The callback simply prints out the
// status of synchronizing the content with the Oracle Storage Cloud Service.
// The same callback is used for all of the StorageObject instances.
 
class ObjectSyncCallback : public SyncCallback {
    public:
        virtual void onSync(const SyncEvent<VirtualDevice>& event) const {
            StorageObject *storageObject = event.getSource();
            cout << "onSync : " << storageObject->getName() << "=\""
                 << storageObject->getSyncStatus() << "\"";
            cout << endl;
        }
};
static ObjectSyncCallback syncCallback;
 
static string const imagePaths[] = {
    "/sd/images/Frame-00.jpeg",
    "/sd/images/Frame-01.jpeg",
    "/sd/images/Frame-02.jpeg",
    "/sd/images/Frame-03.jpeg",
    "/sd/images/Frame-04.jpeg",
    "/sd/images/Frame-05.jpeg",
    "/sd/images/Frame-06.jpeg",
    "/sd/images/Frame-07.jpeg",
    "/sd/images/Frame-08.jpeg",
    "/sd/images/Frame-09.jpeg",
    "/sd/images/Frame-10.jpeg",
    "/sd/images/Frame-11.jpeg"
};
 
static string const videoPaths[] = {
    "/sd/videos/stopwatch_015s.mp4",
    "/sd/videos/stopwatch_030s.mp4",
    "/sd/videos/stopwatch_045s.mp4",
    "/sd/videos/stopwatch_060s.mp4",
    "/sd/videos/stopwatch_075s.mp4",
    "/sd/videos/stopwatch_090s.mp4",
    "/sd/videos/stopwatch_105s.mp4",
    "/sd/videos/stopwatch_120s.mp4",
    "/sd/videos/stopwatch_135s.mp4",
    "/sd/videos/stopwatch_150s.mp4",
    "/sd/videos/stopwatch_165s.mp4",
    "/sd/videos/stopwatch_180s.mp4",
};
 
class DataErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "Data error occurred: " << event.getMessage();
        }
};
DataErrorHandler dataErrorHandler;
 
static DirectlyConnectedDevice *directlyConnectedDevice;

typedef struct {
    int64_t start_time;
    int duration;
    VirtualDevice* virtual_device;
} upload_args;
 
void video_uploader(const void *arg) {
    upload_args *event = (upload_args *)arg;
    int d = event->duration;
    time_t startTime =(time_t)event->start_time;
    if (d <= 0 || startTime == 0) return;
 
    // round to nearest increment of 15
    int duration = (d + 14) / 15 * 15;
    // assumes videos.length > 0 and videos are 15 second increments.
    string videoPath = videoPaths[(duration / 15 - 1) < 11 ? duration / 15 - 1 : 11];
    cout << event->virtual_device->getEndpointId() << " : Call : action : duration=" << d
                                                     << ", startTime=" << startTime << endl;
    /*
     * The name of video will be automatically
     * prefixed with the device's client ID and "/"
     */
    StorageObject& storageObject =
        directlyConnectedDevice->createStorageObject(
            videoPath, "video/mp4");
    storageObject.setOnSync(&syncCallback);
    storageObject.setInputPath(videoPath);
    storageObject.sync();
 
    cout << event->virtual_device->getEndpointId() << " : Set : \""
         << "recording" << "\"=" << storageObject.getURI() << endl;
 
    Data &recording = event->virtual_device->createData(MOTION_ACTIVATED_CAMERA_MODEL_REC_URN);
    recording.set("video", storageObject)
            .set("startTime", (time_t)startTime)
            .set("duration", duration)
            .submit();
    delete event;
}

class CameraOnRecord : public Callable {
    public:
        virtual void call(VirtualDevice *virtualDevice, const NamedValue &data) const {
            int d = data.getValue<int>();
            if (d <= 0) return;
 
            // round to nearest increment of 15
            int duration = (d + 14) / 15 * 15;
            // assumes videos.length > 0 and videos are 15 second increments.

            upload_args *args = new upload_args;
            args->start_time = (int64_t)((int64_t) time(NULL)) * 1000;
            args->duration = data.getValue<int>();
            args->virtual_device = virtualDevice;
            Thread thr;
    //thread->start(mbed::Callback<void()>((const void*) args, reinterpret_cast<void (*)(const void*)> (func_ptr)));
            thr.start(mbed::Callback<void()>((const void*) args, reinterpret_cast<void (*)(const void*)> (video_uploader)));
            //thr.start(callback(video_uploader, (const void*)args));
            thr.join();
        }
};
static CameraOnRecord cameraOnRecord;
 
class CameraOnScheduleRecording : public ActionCallback {
    public:
        virtual void onAction(ActionEvent<VirtualDevice>& event) const {
            upload_args *args = new upload_args;
            if (event.getNamedValue()->getName() == "duration") {
                args->start_time = (int64_t)event.getNamedValue()->next()->getValue<time_t>();
                args->duration = event.getNamedValue()->getValue<int>();
            } else {
                args->start_time = (int64_t)event.getNamedValue()->getValue<time_t>();
                args->duration = event.getNamedValue()->next()->getValue<int>();
            }
            args->virtual_device = event.getVirtualDevice();
            Thread thr;
            thr.start(mbed::Callback<void()>((const void*) args, reinterpret_cast<void (*)(const void*)> (video_uploader)));
            thr.join();
        }
};
static CameraOnScheduleRecording cameraOnScheduleRecording;
 
class ErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "ErrorHandler: " << event.getMessage() << endl;
        }
};
static ErrorHandler errorHandler;
 
int sample_main(char* file, char* password) {
    string urn(MOTION_ACTIVATED_CAMERA_MODEL_URN);
 
    srand(time(NULL));
 
    try {
        DirectlyConnectedDevice dcd(file, password);
        directlyConnectedDevice = &dcd;
        if (!dcd.isActivated()) {
            dcd.activate({urn});
        }
 
        DeviceModel& deviceModel = dcd.getDeviceModel(urn);
 
        VirtualDevice& virtualMotionActivatedCamera = dcd.createVirtualDevice(dcd.getEndpointId(), deviceModel);
        virtualMotionActivatedCamera.setCallable("record", &cameraOnRecord);
        virtualMotionActivatedCamera.setOnAction("scheduleRecording", &cameraOnScheduleRecording);
 
        virtualMotionActivatedCamera.setOnError(&errorHandler);
 
        cout << "\nCreated virtual motion activated camera "  << virtualMotionActivatedCamera.getEndpointId() << endl;
 
        int n = 0;
        osDelay(5000);
        for (;;) {
 
            string imagePath = imagePaths[n++ % 12];
 
            cout << virtualMotionActivatedCamera.getEndpointId() << " : Set : \"" << IMAGE_ATTRIBUTE << "\"=" << imagePath << endl;
 
            try {
                /*
                 * The name of video will be automatically
                 * prefixed with the device's client ID and "/"
                 *
                 * If this device implements many models or
                 * this was a gateway device, adding the another
                 * level of uniqueness would be warrented.
                 */
                StorageObject& storageObject =
                    dcd.createStorageObject(
                        imagePath, "image/jpeg");
                storageObject.setOnSync(&syncCallback);
                storageObject.setInputPath(imagePath);
 
                virtualMotionActivatedCamera.update()
                                            .set(IMAGE_ATTRIBUTE, &storageObject)
                                            .set(IMAGE_TIME_ATTRIBUTE, time(NULL))
                                            .finish();
                cout << "Sleep 30 seconds!\n";
                osDelay(30000);
            } catch (std::exception &e) {
                cout << "EXCEPTION CAUGHT: " << e.what() << endl;
                continue;
            }
        }
    } catch (std::exception &e) {
        cout << "EXCEPTION CAUGHT: " << e.what() << endl;
        return 1;
    }
    cout << "OK\n";
    return 0;
}
 
int main() {
    int status;
    // set serial baudrate    
    pc.baud(115200);
    // switch on blue led only
    led_red = 1;
    led_green = 1;
    led_blue = 0;
 
    // configure ethernet interface
    pc.printf("Init network...\n");
    if (eth.init() == 0 && eth.connect(20000) == 0) { //Use DHCP
        pc.printf("Network initialized\n");
        pc.printf("  IP Address:      %s\n", eth.getIPAddress());
        pc.printf("  Subnet mask:     %s\n", eth.getNetworkMask());
        pc.printf("  Gateway:         %s\n", eth.getGateway());
        pc.printf("  MAC-address:     %s\n", eth.getMACAddress());
        pc.printf("Trying to update time by NTP...\n");
        if (ntp.setTime(eth.getGateway()) == 0 || ntp.setTime("pool.ntp.org") == 0) {
            pc.printf("Set time successfully\n");
            time_t ctTime;
            ctTime = time(NULL);
            pc.printf("Time is set to (UTC): %s\n", ctime(&ctTime));
        } else {
            pc.printf("Can't get time from NTP\n");
        }
        pc.printf("Mount sd card\n");
        pc.printf("  Sd card name: %s\n\n", STRINGIFY(IOTCS_SDCARD_MOUNT_NAME));
 
        pc.printf("check if bootstrap is required...\n");
 
        status = sample_main(mbed_ts_path, mbed_ts_password);
        led_blue = 1;
        if (EXIT_SUCCESS == status) {
            led_green = 0;
        } else {
            led_red = 0;
        }
        while (1);
    } else {
        pc.printf("Can not obtain DHCP settings\n");
        led_red = 0;
    }
    eth.disconnect();
}
