/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

#include <DeviceModel.hpp>
#include <VirtualDevice.hpp>
#include <NamedValue.hpp>
#include <Exception.hpp>
#include "HumiditySensor.hpp"
#include "TemperatureSensor.hpp"

#pragma once

using namespace iotdcl;
using namespace std;

class HumidityOnMaxThresholdError: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "HumidityOnMaxThresholdError: " << event.getMessage() << endl;
            exit(EXIT_FAILURE);
        }
};

class HumidityOnMaxThreshold: public ChangeCallback {
    HumiditySensor* sensor;
    public:
        HumidityOnMaxThreshold(HumiditySensor* sensor) {
            this->sensor = sensor;
        }
        virtual void onChange(const ChangeEvent<VirtualDevice>& event) const {
        cout << "HumidityOnMaxThreshold::onChange is called" << endl;
        assert(event.getNamedValue() != NULL && event.getNamedValue()->getName() == "maxThreshold");
        int current_maxThreshold = event.getNamedValue()->getValue<int>();
        cout << "Device max humidity threshold = " << current_maxThreshold << endl;
        sensor->setMaxThreshold(current_maxThreshold);
    }
};

class HumidityOnChange: public ChangeCallback {
    HumiditySensor* sensor;
    public:
        HumidityOnChange(HumiditySensor* sensor) {
            this->sensor = sensor;
        }
        virtual void onChange(const ChangeEvent<VirtualDevice>& event) const {
            cout << "HumidityOnChange::onChange is called - do nothing" << endl;
        }
};

class TemperatureOnMaxThresholdError: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "TemperatureOnMaxThresholdError: " << event.getMessage() << endl;
            exit(EXIT_FAILURE);
        }
};

class TemperatureOnMaxThreshold: public ChangeCallback {
    TemperatureSensor* sensor;
    public:
        TemperatureOnMaxThreshold(TemperatureSensor* sensor) {
            this->sensor = sensor;
        }
        virtual void onChange(const ChangeEvent<VirtualDevice>& event) const {
            cout << "TemperatureOnMaxThreshold::onChange is called" << endl;
            assert(event.getNamedValue() != NULL && event.getNamedValue()->getName() == "maxThreshold");
            int current_maxThreshold = event.getNamedValue()->getValue<int>();
            cout << "Device max Temperature threshold = " << current_maxThreshold << endl;
            sensor->setMaxThreshold(current_maxThreshold);
        }
};

class TemperatureOnChange : public ChangeCallback {
    TemperatureSensor* sensor;
    public:
        TemperatureOnChange(TemperatureSensor* sensor) {
            this->sensor = sensor;
        }
        virtual void onChange(const ChangeEvent<VirtualDevice>& event) const {
            cout << "TemperatureOnChange::onChange is called" << endl;
            NamedValue *value = event.getNamedValue();
            while (value != NULL) {
                if (value->getName() == "minThreshold") {
                    sensor->setMinThreshold(value->getValue<int>());
                    cout << "Device min temp threshold = " << value->getValue<int>() << endl;
                } else if (value->getName() == "maxThreshold") {
                    sensor->setMaxThreshold(value->getValue<int>());
                    cout << "Device max temp threshold = " << value->getValue<int>() << endl;
                } else {
                    cout << "Unexpected attribute in temp on change callback" << endl;
                }
                value = value->next();
            }
        }
};

class TemperatureOnReset : public Callable {
    TemperatureSensor* sensor;
    public:
        TemperatureOnReset(TemperatureSensor* sensor) {
            this->sensor = sensor;
        }
        virtual void call( VirtualDevice &virtualDevice, const NamedValue &data) const {
            cout << "Call : reset : void" << endl;
            sensor->reset();
        }
};

class TemperatureOnPower : public Callable {
    TemperatureSensor* sensor;
    public:
        TemperatureOnPower(TemperatureSensor* sensor) {
            this->sensor = sensor;
        }
        virtual void call( VirtualDevice &virtualDevice, const NamedValue &data) const {
            if (data.getValue<bool>()) {
                cout << "Call : power : Power on temp sensor!!!" << endl;
            } else {
                cout << "Call : power : Power off temp sensor!!!" << endl;
            }
            sensor->power(data.getValue<bool>());
        }
};
