/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
#include "mbed.h"
#include "EthernetInterface.h"
#include "NTPClient.h"
#include "SDFileSystem.h"
#include "cmsis_os.h"
#include <iostream>
#include <memory>
#include <exception>
#include <map>
#include <vector>
#include <signal.h>
#include <stdio.h>
#include <string>

#include <assert.h>
#include <math.h>

#include <VirtualDevice.hpp>
#include <GatewayDevice.hpp>
#include <DeviceModel.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <NamedValue.hpp>
#include <Exception.hpp>
#include "HumiditySensor.hpp"
#include "TemperatureSensor.hpp"
#include "Callbacks.hpp"

#ifndef IOTCS_MBED_TS_PASSWORD
#define IOTCS_MBED_TS_PASSWORD changeit\0
#endif // #ifndef IOTCS_MBED_TS_PASSWORD

#ifndef IOTCS_MBED_TS_PATH
#define IOTCS_MBED_TS_PATH /sd/trusted_assets_store\0
#endif // #ifndef IOTCS_MBED_TS_PATH

#define __STRINGIFY(x) #x
#define STRINGIFY(x) __STRINGIFY(x)

#define CMD_SET_TIME   (0x73)  // 's'

Serial pc(USBTX, USBRX);
EthernetInterface eth;
NTPClient ntp;

DigitalOut led_red(LED_RED);
DigitalOut led_green(LED_GREEN);
DigitalOut led_blue(LED_BLUE);

char *mbed_ts_path = STRINGIFY(IOTCS_MBED_TS_PATH);
char *mbed_ts_password = STRINGIFY(IOTCS_MBED_TS_PASSWORD);

#ifndef IOTCS_SDCARD_MOUNT_NAME
#define IOTCS_SDCARD_MOUNT_NAME sd\0
#endif // #ifndef IOTCS_SDCARD_MOUNT_NAME
// mount sd card with IOTCS_SDCARD_MOUNT_NAME name
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, STRINGIFY(IOTCS_SDCARD_MOUNT_NAME)); // MOSI, MISO, SCK, CS


using namespace iotdcl;
using namespace std;

/*
 *                           Constants & Variables
 */
/* Policy is switched off by default for CPP MBED */
#ifndef USE_POLICY
#define USE_POLICY false
#endif


/* used as sending loop condition */
static volatile int keep_running = 1;
char result[30];
string to_string(int value) {
    sprintf(result, "%d", value);
    return string(result);
}
string to_string(float value) {
    sprintf(result, "%f", value);
    return string(result);
}
const string temperatureSensorUrn("urn:com:oracle:iot:device:temperature_sensor");
const string humiditySensorUrn("urn:com:oracle:iot:device:humidity_sensor");

/* print all humidity device attributes */
static void printHSAttributes(VirtualDevice &vd) {
    cout << "humidity attribute value = " << vd.get<int>("humidity") << endl;
    cout << "maxThreshold attribute value = " << vd.get<int>("maxThreshold") << endl;
}
/* print all temperature device attributes */
static void printTSAttributes(VirtualDevice &vd) {
    cout << "Device start time = " << vd.get<time_t>("startTime") << endl;
    cout << "Device temp value = " << vd.get<float>("temp") << endl;
    cout << "Device unit = " << vd.get<string>("unit") << endl;
    cout << "Device min temp = " << vd.get<float>("minTemp") << endl;
    cout << "Device max temp = " << vd.get<float>("maxTemp") << endl;
    cout << "Device min temp threshold = " << vd.get<int>("minThreshold") << endl;
    cout << "Device max temp threshold = " << vd.get<int>("maxThreshold") << endl;
}

class ErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "Error occurred: " << event.getMessage();
        }
};
ErrorHandler errorHandler;
/* Callbacks */
HumidityOnChange *humidityOnChange;
HumidityOnMaxThreshold *humidityOnMaxThreshold;
HumidityOnMaxThresholdError *humidityOnMaxThresholdError;
TemperatureOnChange *temperatureOnChange;
TemperatureOnMaxThreshold *temperatureOnMaxThreshold;
TemperatureOnMaxThresholdError *temperatureOnMaxThresholdError;
TemperatureOnReset *temperatureOnReset;
TemperatureOnPower *temperatureOnPower;

int sample_main(char* file, char* password) {
    srand(time(NULL));
    try {
        GatewayDevice gwd(file, password);
        if (!gwd.isActivated()) {
            gwd.activate({temperatureSensorUrn, humiditySensorUrn});
        }
        
        DeviceModel& humidityModel = gwd.getDeviceModel(humiditySensorUrn);
        DeviceModel& temperatureModel = gwd.getDeviceModel(temperatureSensorUrn);

        TemperatureSensor temperatureSensor(gwd.getEndpointId() + "_Sample_TS");
        HumiditySensor humiditySensor(gwd.getEndpointId() + "_Sample_HS");
        const map<string, string> temperatureMeta = {
            {"manufacturer", temperatureSensor.getManufacturer()},
            {"modelNumber", temperatureSensor.getModelNumber()},
            {"serialNumber", temperatureSensor.getSerialNumber()},
            {"deviceClass", "temp sensor"},
            {"sampleAttr", "test text temp sensor"},
            {"year", "2015"}};
        const map<string, string> humidityMeta = {
            {"manufacturer", humiditySensor.getManufacturer()},
            {"modelNumber", humiditySensor.getModelNumber()},
            {"serialNumber", humiditySensor.getSerialNumber()},
            {"deviceClass", "humidity sensor"},
            {"sampleStrAttr", "test text humidity sensor"},
            {"year", "2015"}};

        const string humiditySensorEndpointId = gwd.registerDevice(false, humiditySensor.getHardwareId(), humidityMeta, {humiditySensorUrn});

        const string temperatureSensorEndpointId = gwd.registerDevice(false, temperatureSensor.getHardwareId(),
                temperatureMeta, {temperatureSensorUrn});
        /* get temperature device handle */
        VirtualDevice& temperatureDevice = gwd.createVirtualDevice(temperatureSensorEndpointId, temperatureModel);
        /* get humidity device handle */
        VirtualDevice& humidityDevice = gwd.createVirtualDevice(humiditySensorEndpointId, humidityModel);


        humidityOnChange = new HumidityOnChange(&humiditySensor);
        humidityOnMaxThreshold = new HumidityOnMaxThreshold(&humiditySensor);
        humidityOnMaxThresholdError = new HumidityOnMaxThresholdError();
        temperatureOnChange = new TemperatureOnChange(&temperatureSensor);
        temperatureOnMaxThreshold = new TemperatureOnMaxThreshold(&temperatureSensor);
        temperatureOnMaxThresholdError = new TemperatureOnMaxThresholdError();
        temperatureOnReset = new TemperatureOnReset(&temperatureSensor);
        temperatureOnPower = new TemperatureOnPower(&temperatureSensor);

        temperatureDevice.setOnChange(temperatureOnChange);
        temperatureDevice.setOnError(&errorHandler);
        temperatureDevice.setCallable("reset", temperatureOnReset);
        temperatureDevice.setCallable("power", temperatureOnPower);
        humidityDevice.setOnChange(humidityOnChange);
        humidityDevice.setOnChange("maxThreshold", humidityOnMaxThreshold);
        humidityDevice.setOnError(&errorHandler);
        humidityDevice.setOnError("maxThreshold", humidityOnMaxThresholdError);
        
        Alert& humidityAlert(humidityDevice.createAlert("urn:com:oracle:iot:device:humidity_sensor:too_humid"));
        Alert& temperatureHotAlert(temperatureDevice.createAlert("urn:com:oracle:iot:device:temperature_sensor:too_hot"));
        Alert& temperatureColdAlert(temperatureDevice.createAlert("urn:com:oracle:iot:device:temperature_sensor:too_cold"));

        cout << "Batch update update of temparature device attributes" << endl;

        time_t startTime = temperatureSensor.getStartTime();
        float temp = temperatureSensor.getTemp();
        float minTemp = temperatureSensor.getMinTemp();
        float maxTemp = temperatureSensor.getMaxTemp();
        int minThreshold = temperatureSensor.getMinThreshold();
        int maxThreshold = temperatureSensor.getMaxThreshold();

        temperatureDevice.update()
                            .set("startTime", startTime)
                            .set("temp", temp)
                            .set("unit", temperatureSensor.getUnit())
                            .set("minTemp", minTemp)
                            .set("maxTemp", maxTemp)
                            .set("minThreshold", minThreshold)
                            .set("maxThreshold", maxThreshold)
                            .finish();

        int humidity = humiditySensor.getHumidity();
        int humidityThreshold =
                humiditySensor.getMaxThreshold();

        cout << "Batch update update of humidity device attributes" << endl;
        humidityDevice.update()
                         .set("humidity", humidity)
                         .set("maxThreshold", humidityThreshold)
                         .finish();

        printHSAttributes(humidityDevice);
        printTSAttributes(temperatureDevice);

        bool humidityAlerted = false;
        bool tempAlerted = false;
        bool wasOff = false;
        /* 
         * Emulate device work. 
         * Sensors changed values, then device send data message with new values 
         * to the server. If sensors values is out of threshold then 
         * send alert message.
         * Async message dispatcher send messages to the server and handle 
         * message delivery and errors during send. 
         * Request dispatcher handle incoming requests from the server using 
         * custom request handlers.
         */
        while (keep_running) {
            cout << "Sleep 5 seconds!" << endl;
            osDelay(5000);
            try {
                humidity =
                        humiditySensor.getHumidity();
                humidityThreshold =
                        humiditySensor.getMaxThreshold();

                cout << humiditySensorEndpointId +
                        " : Set : \"humidity\"=" << humidity << endl;

                humidityDevice.set("humidity", humidity);

                if (humidity > humidityThreshold) {
                    cout << "Alert : \"humidity\" = " << humidity << ", \"maxThreshold\" = " << humidityThreshold << endl;
                    humidityAlert.set("humidity", humidity).raise();
                }
                if (!temperatureSensor.isPoweredOn()) {
                    wasOff = true;
                    continue;
                }

                if (wasOff) {
                    wasOff = false;

                    cout << "Power back on, updating all attributes." << endl;

                    startTime =
                            temperatureSensor.getStartTime();
                    temp =
                            temperatureSensor.getTemp();
                    minTemp =
                            temperatureSensor.getMinTemp();
                    maxTemp =
                            temperatureSensor.getMaxTemp();
                    minThreshold =
                            temperatureSensor.getMinThreshold();
                    maxThreshold =
                            temperatureSensor.getMaxThreshold();

                    cout << temperatureDevice.
                                getEndpointId() + " : Set : " +
                            "\"power\"=true" +
                            ",\"temp\"=" << temp <<
                            ",\"unit\"=" << temperatureSensor.getUnit() <<
                            ",\"minTemp\"=" << minTemp <<
                            ",\"maxTemp\"=" << maxTemp <<
                            ",\"minThreshold\"=" <<
                            minThreshold <<
                            ",\"maxThreshold\"=" <<
                            maxThreshold;

                    // After the sensor is powered on,
                    // the attribute values in the
                    // VirtualDevice need to be updated
                    // to reflect the new values.
                    temperatureDevice.update()
                            .set("startTime", startTime)
                            .set("temp", temp)
                            .set("unit", temperatureSensor.getUnit())
                            .set("minTemp", minTemp)
                            .set("maxTemp", maxTemp)
                            .set("minThreshold", minThreshold)
                            .set("maxThreshold", maxThreshold)
                            .finish();
                    continue;
                }

                time_t prevStartTime = temperatureDevice.get<time_t>("startTime");
                float prevMinTemp = temperatureDevice.get<float>("minTemp");
                float prevMaxTemp = temperatureDevice.get<float>("maxTemp");
                float temperature = temperatureSensor.getTemp();
                minTemp = temperatureSensor.getMinTemp();
                maxTemp = temperatureSensor.getMaxTemp();
                startTime = temperatureSensor.getStartTime();
                string id = temperatureSensorEndpointId;

                string sb = id + " : Set : \"temp\"=" + to_string(temperature);

                temperatureDevice.update().set(
                        "temp", temperature);

                if (minTemp != prevMinTemp) {
                    sb += ",minTemp=" + to_string(minTemp);
                    temperatureDevice.set(
                            "minTemp", minTemp);
                }

                if (maxTemp != prevMaxTemp) {
                    sb += ",maxTemp=" + to_string(maxTemp);
                    temperatureDevice.set(
                            "maxTemp", maxTemp);
                }

                if (startTime != prevStartTime) {
                    sb += ",startTime=" + to_string((int)startTime);
                    temperatureDevice.set(
                            "startTime", startTime);
                }
                cout << (sb) << endl;
                temperatureDevice.finish();

                int minTempThreshold = temperatureSensor.getMinThreshold();
                int maxTempThreshold = temperatureSensor.getMaxThreshold();
                if (temperature > maxTempThreshold) {
                    if (!tempAlerted) {
                        cout << "Send hot alert" << endl;
                        temperatureHotAlert.set("temp", temperature)
                                              .set("unit", temperatureSensor.getUnit())
                                              .set("maxThreshold", (float)maxTempThreshold)
                                              .raise();
                    }
                } else if (temperature < minTempThreshold) {
                    if (!tempAlerted) {
                        cout << "Send cold alert" << endl;
                        temperatureColdAlert.set("temp", temperature)
                                               .set("unit", temperatureSensor.getUnit())
                                               .set("minThreshold", (float)minTempThreshold)
                                               .raise();
                    }
                } else {
                    tempAlerted = false;
                }

                printTSAttributes(temperatureDevice);
                printHSAttributes(humidityDevice);
            } catch (std::exception &e) {
                cout << "EXCEPTION CAUGHT: " << e.what() << endl;
                continue;
            }
        } // end while
    } catch (std::exception &e) {
        cout << "EXCEPTION CAUGHT: " << e.what() << endl;
        return 1;
    }
    cout << "OK" << endl;
    return 0;
}

int main() {
    int status;
    // set serial baudrate    
    pc.baud(115200);
    // switch on blue led only
    led_red = 1;
    led_green = 1;
    led_blue = 0;

    // configure ethernet interface
    pc.printf("Init network...\n");
    if (eth.init() == 0 && eth.connect(20000) == 0) { //Use DHCP
        pc.printf("Network initialized\n");
        pc.printf("  IP Address:      %s\n", eth.getIPAddress());
        pc.printf("  Subnet mask:     %s\n", eth.getNetworkMask());
        pc.printf("  Gateway:         %s\n", eth.getGateway());
        pc.printf("  MAC-address:     %s\n", eth.getMACAddress());
        pc.printf("Trying to update time by NTP...\n");
        if (ntp.setTime(eth.getGateway()) == 0 || ntp.setTime("pool.ntp.org") == 0) {
            pc.printf("Set time successfully\n");
            time_t ctTime;
            ctTime = time(NULL);
            pc.printf("Time is set to (UTC): %s\n", ctime(&ctTime));
        } else {
            pc.printf("Can't get time from NTP\n");
        }
        pc.printf("Mount sd card\n");
        pc.printf("  Sd card name: %s\n\n", STRINGIFY(IOTCS_SDCARD_MOUNT_NAME));

        pc.printf("check if bootstrap is required...\n");

        // Execute sample code
        status = sample_main(mbed_ts_path, mbed_ts_password);
        led_blue = 1;
        if (EXIT_SUCCESS == status) {
            led_green = 0;
        } else {
            led_red = 0;
        }
        while (1);
    } else {
        pc.printf("Can not obtain DHCP settings\n");
        led_red = 0;
    }
    eth.disconnect();
}
