/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */
#include "mbed.h"
#include "EthernetInterface.h"
#include "NTPClient.h"
#include "SDFileSystem.h"
#include "cmsis_os.h"
#include <iostream>
#include <memory>
#include <exception>
#include <map>
#include <vector>
#include <signal.h>
#include <stdio.h>
#include <string>

#include <assert.h>
#include <math.h>

#include <VirtualDevice.hpp>
#include <DeviceModel.hpp>
#include <DirectlyConnectedDevice.hpp>
#include <NamedValue.hpp>
#include <Exception.hpp>
#include "HumiditySensor.hpp"
#include "Callbacks.hpp"

#ifndef IOTCS_MBED_TS_PASSWORD
#define IOTCS_MBED_TS_PASSWORD changeit\0
#endif // #ifndef IOTCS_MBED_TS_PASSWORD

#ifndef IOTCS_MBED_TS_PATH
#define IOTCS_MBED_TS_PATH /sd/trusted_assets_store\0
#endif // #ifndef IOTCS_MBED_TS_PATH

#define __STRINGIFY(x) #x
#define STRINGIFY(x) __STRINGIFY(x)

#define CMD_SET_TIME   (0x73)  // 's'

Serial pc(USBTX, USBRX);
EthernetInterface eth;
NTPClient ntp;

DigitalOut led_red(LED_RED);
DigitalOut led_green(LED_GREEN);
DigitalOut led_blue(LED_BLUE);

char *mbed_ts_path = STRINGIFY(IOTCS_MBED_TS_PATH);
char *mbed_ts_password = STRINGIFY(IOTCS_MBED_TS_PASSWORD);

#ifndef IOTCS_SDCARD_MOUNT_NAME
#define IOTCS_SDCARD_MOUNT_NAME sd\0
#endif // #ifndef IOTCS_SDCARD_MOUNT_NAME
// mount sd card with IOTCS_SDCARD_MOUNT_NAME name
SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, STRINGIFY(IOTCS_SDCARD_MOUNT_NAME)); // MOSI, MISO, SCK, CS


using namespace iotdcl;
using namespace std;

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))
#define ARRAY_LEN(X) (sizeof(X)/sizeof(X[0]))

/*
 *                           Constants & Variables
 */
/* used as sending loop condition */
static volatile int keep_running = 1;
char result[30];
string to_string(int value) {
    sprintf(result, "%d", value);
    return string(result);
}
string to_string(float value) {
    sprintf(result, "%f", value);
    return string(result);
}
const string humiditySensorUrn("urn:com:oracle:iot:device:humidity_sensor");

/* print all humidity device attributes */
static void printHSAttributes(VirtualDevice &vd) {
    cout << "humidity attribute value = " << vd.get<int>("humidity") << endl;
    cout << "maxThreshold attribute value = " << vd.get<int>("maxThreshold") << endl;
}

class ErrorHandler: public ErrorCallback {
    public:
        virtual void onError(const ErrorEvent<VirtualDevice>& event) const {
            cerr << "Error occurred: " << event.getMessage();
        }
};
ErrorHandler errorHandler;

int sample_main(char* file, char* password) {
    srand(time(NULL));
    try {
        DirectlyConnectedDevice dcd(file, password);
        if (!dcd.isActivated()) {
            dcd.activate({humiditySensorUrn});
        }

        DeviceModel& humidityModel = dcd.getDeviceModel(humiditySensorUrn);

        HumiditySensor humiditySensor(dcd.getEndpointId());

        const string humiditySensorEndpointId = dcd.getEndpointId();

        /* get humidity device handle */
        VirtualDevice& humidityDevice = dcd.createVirtualDevice(humiditySensorEndpointId, humidityModel);
 
        /* Callbacks */
        HumidityOnChange humidityOnChange(&humiditySensor);
        HumidityOnMaxThreshold humidityOnMaxThreshold(&humiditySensor);
        humidityDevice.setOnChange(&humidityOnChange);
        humidityDevice.setOnChange("maxThreshold", &humidityOnMaxThreshold);
        humidityDevice.setOnError(&errorHandler);
        
        Alert& humidityAlert(humidityDevice.createAlert("urn:com:oracle:iot:device:humidity_sensor:too_humid"));

        cout << "Batch update update of temparature device attributes" << endl;

        int humidity = humiditySensor.getHumidity();
        int humidityThreshold =
                humiditySensor.getMaxThreshold();
        cout << "Batch update update of humidity device attributes" << endl;
        humidityDevice.update()
                         .set("humidity", humidity)
                         .set("maxThreshold", humidityThreshold)
                         .finish();
        printHSAttributes(humidityDevice);

        bool humidityAlerted = false;
        /* 
         * Emulate device work. 
         * Sensors changed values, then device send data message with new values 
         * to the server. If sensors values is out of threshold then 
         * send alert message.
         * Async message dispatcher send messages to the server and handle 
         * message delivery and errors during send. 
         * Request dispatcher handle incoming requests from the server using 
         * custom request handlers.
         */
        while (keep_running) {
            cout << "Sleep 5 seconds!" << endl;
            osDelay(5000);
            try {
                humidity =
                        humiditySensor.getHumidity();
                humidityThreshold =
                        humiditySensor.getMaxThreshold();

                cout << humiditySensorEndpointId +
                        " : Set : \"humidity\"=" << humidity << endl;

                humidityDevice.set("humidity", humidity);

                if (humidity > humidityThreshold) {
                    if (!humidityAlerted) {
                        humidityAlerted = true;
                        cout << "Alert : \"humidity\" = " << humidity << ", \"maxThreshold\" = " << humidityThreshold << endl;
                        humidityAlert.set("humidity", humidity).raise();
                    }
                } else {
                    humidityAlerted = false;
                }
                printHSAttributes(humidityDevice);
            } catch (std::exception &e) {
                cout << "EXCEPTION CAUGHT: " << e.what() << endl;
                continue;
            }
        } // end while
    } catch (std::exception &e) {
        cout << "EXCEPTION CAUGHT: " << e.what() << endl;
        return 1;
    }
    cout << "OK" << endl;
    return 0;
}

int main() {
    int status;
    // set serial baudrate    
    pc.baud(115200);
    // switch on blue led only
    led_red = 1;
    led_green = 1;
    led_blue = 0;

    // configure ethernet interface
    pc.printf("Init network...\n");
    if (eth.init() == 0 && eth.connect(20000) == 0) { //Use DHCP
        pc.printf("Network initialized\n");
        pc.printf("  IP Address:      %s\n", eth.getIPAddress());
        pc.printf("  Subnet mask:     %s\n", eth.getNetworkMask());
        pc.printf("  Gateway:         %s\n", eth.getGateway());
        pc.printf("  MAC-address:     %s\n", eth.getMACAddress());
        pc.printf("Trying to update time by NTP...\n");
        if (ntp.setTime(eth.getGateway()) == 0 || ntp.setTime("pool.ntp.org") == 0) {
            pc.printf("Set time successfully\n");
            time_t ctTime;
            ctTime = time(NULL);
            pc.printf("Time is set to (UTC): %s\n", ctime(&ctTime));
        } else {
            pc.printf("Can't get time from NTP\n");
        }
        pc.printf("Mount sd card\n");
        pc.printf("  Sd card name: %s\n\n", STRINGIFY(IOTCS_SDCARD_MOUNT_NAME));

        pc.printf("check if bootstrap is required...\n");

        // Execute sample code
        status = sample_main(mbed_ts_path, mbed_ts_password);
        led_blue = 1;
        if (EXIT_SUCCESS == status) {
            led_green = 0;
        } else {
            led_red = 0;
        }
        while (1);
    } else {
        pc.printf("Can not obtain DHCP settings\n");
        led_red = 0;
    }
    eth.disconnect();
}
