/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

/**
 * @file iotcs_port_diagnostic.h
 * @brief The file contains porting layer for functionality required by diagnostic capability.
 */

#ifndef IOTCS_PORT_DIAGNOSTIC_H
#define	IOTCS_PORT_DIAGNOSTIC_H

#ifdef	__cplusplus
extern "C" {
#endif

#include "iotcs_config.h"

    /**
     * @brief Writes ip address of the device as a null terminated string to a given buffer
     * Returned string is sent to IoT server in response to \a /info request 
     * of \a urn:oracle:iot:dcd:capability:diagnostics device model
     * as a value of \a ipAddress property.
     * @note Optional API. Called by the Library if IOTCS_USE_DIAGNOSTIC_CAPABILITY option is defined.
     * @param buffer output buffer
     * @param buf_len length of the output buffer
     * @return The number of characters written to \a buffer, not counting the terminating null character.
     * @return Negative value otherwise (including cases when \a buffer is too small for the string)
     */
    int iotcs_port_get_ip_address(char* buffer, int buf_len);

    /**
     * @brief Writes mac address of the device as a null terminated string to a given buffer
     * Returned string is sent to IoT server in response to \a /info request 
     * of \a urn:oracle:iot:dcd:capability:diagnostics device model
     * as a value of \a macAddress property.
     * @note Optional API. Called by the Library if IOTCS_USE_DIAGNOSTIC_CAPABILITY option is defined.
     * @param buffer output buffer
     * @param buf_len length of the output buffer
     * @return The number of characters written to \a buffer, not counting the terminating null character.
     * @return Negative value otherwise (including cases when \a buffer is too small for the string)
     */
    int iotcs_port_get_mac_address(char* buffer, int buf_len);

    /**
     * @brief Returns device start time in milliseconds since epoch
     * Returned value is sent to IoT server in response to \a /info request 
     * of \a urn:oracle:iot:dcd:capability:diagnostics device model
     * as a value of \a startTime property.
     * @note Optional API. Called by the Library if IOTCS_USE_DIAGNOSTIC_CAPABILITY option is defined.
     * @return Device start time if success
     * @return Negative value otherwise
     */
    int64_t iotcs_port_get_start_time(void);

    /**
     * @brief Returns version as a null terminated string
     * Returned string is sent to IoT server in response to \a /info request 
     * of \a urn:oracle:iot:dcd:capability:diagnostics device model
     * as a value of \a version property.
     * @note Optional API. Called by the Library if IOTCS_USE_DIAGNOSTIC_CAPABILITY option is defined.
     * @return Version string address if success
     * @return NULL otherwise
     */
    const char* iotcs_port_get_version(void);

    /**
     * @brief The total disk space in bytes on this directly connected device
     * Returned value is sent to IoT server in response to \a /info request 
     * of \a urn:oracle:iot:dcd:capability:diagnostics device model
     * as a value of \a totalDiskSpace property.
     * @note Optional API. Called by the Library if IOTCS_USE_DIAGNOSTIC_CAPABILITY option is defined.
     * @return total disk space in bytes on this directly connected device.
     */
    uint64_t iotcs_port_get_total_disk_space(void);

    /**
     * @brief The free disk space in bytes on this directly connected device
     * Returned value is sent to IoT server in response to \a /info request 
     * of \a urn:oracle:iot:dcd:capability:diagnostics device model
     * as a value of \a freeDiskSpace property.
     * @note Optional API. Called by the Library if IOTCS_USE_DIAGNOSTIC_CAPABILITY option is defined.
     * @return free disk space in bytes on this directly connected device.
     */
    uint64_t iotcs_port_get_free_disk_space(void);

#ifdef	__cplusplus
}
#endif

#endif	/* IOTCS_PORT_DIAGNOSTIC_H */

