# 
# Copyright (c) 2015, 2019, Oracle and/or its affiliates. All rights reserved.
#
# This software is dual-licensed to you under the MIT License (MIT) and the Universal Permissive License (UPL).  See the LICENSE file in the root directory for license terms.  You may choose
# either license, or both.
#

################################################################################
## custom arguments                                                           ##
################################################################################

################################################################################
## Library could be presented like set of elements:                           ##
##                                                                            ##
##   |--------------------|--------------------|                              ##
##   | 1. Library         |  2.     TAM        |                              ##
##   |        shared part |                    |                              ##
##   |--------------------|--------------------|                              ##
##   | 3.  Port layer     |  4.  Port TAM      |                              ##
##   |--------------------|--------------------|                              ##
##                                                                            ##
##  Build system use several compiler flags.                                  ##
##  These flags cover different elements. For example:                        ##
##  - LIB_OPTS                      = 1                                       ##
##  - PORT_OPTS                     = 3 + 4                                   ##
##  - TAM_OPTS                      = 2 + 4                                   ##
##  - common custom compiler flags  = 1 + 2 + 3 + 4                           ##
##                                                                            ##
##  - 1   = LIB_OPTS                                                          ##
##  - 2   = TAM_OPTS                                                          ##
##  - 3   = PORT_OPTS                                                         ##
##  - 4   = TAM_OPTS PORT_OPTS                                                ##
##  - 1 + 2 + 3 + 4 = common compiler flags                                   ##
##                                                                            ##
##  Lib flags include custom LIB_OPTS and internal LIB_FLAGS                  ##
##                                                                            ##
##  Port flags include custom PORT_OPTS and internal PORT_FLAGS               ##
##                                                                            ##
##  TAM flags include custom TAM_OPTS and internal TAM_FLAGS                  ##
##                                                                            ##
################################################################################
# custom compiler flags for port layer
PORT_OPTS?=
# custom compiler flags for library
LIB_OPTS?=
# custom compiler flags for TAM
TAM_OPTS?=
# + common custom compiler arguments

################################################################################
## internal port-dependent variables                                          ##
##                   should be overwritten in port implementation             ##
################################################################################
# object files of porting layer
OBJ_PORT?=
# object files of TAM porting layer
TAM_OBJ_PORT?=

################################################################################
## internal port-independent variables shouldn't be overwritten               ##
################################################################################
# base file check
ifeq ($(LIB_CONFIG),)
	$(error No library configuration file for LIB_CFG=$(LIB_CFG) !)
endif
ifeq ($(LIB_SHARED_CUSTOM_CONFIG),)
	$(info No custom shared library part configuration file !)
endif
ifeq ($(LIB_PORT_CUSTOM_CONFIG),)
	$(error No custom port library part configuration file !)
endif
ifeq ($(COMPILER_CONFIG),)
	$(error No compiler configuration file for CC_CFG=$(CC_CFG) !)
endif
ifeq ($(LIB_COMMON_DEFS),)
	$(error No library common defenition file !)
endif
ifeq ($(LIB_PORT_IMPL),)
	$(error No library port implemenetation file !)
endif

## variable depending on the configuration ##
# used for verbose output log
ifneq ($(VERBOSE),true)
	A=@
else
	V=-v
endif

# used for output directory name
ifeq ($(DEBUG), true)
	DEBUG_SUFFIX=_d
	LOG_LEVEL=dbg
endif
# used for output object directory name
CONF_SUBDIR:=
ifeq ($(MESSAGING_THREAD_SAFETY), true)
	CONF_SUBDIR:=ts
else
	CONF_SUBDIR:=nots
endif
ifeq ($(MESSAGE_DISPATCHER), true)
	CONF_SUBDIR:=$(CONF_SUBDIR)_md
endif
ifeq ($(VIRTUALIZATION_SUPPORT), true)
	CONF_SUBDIR:=$(CONF_SUBDIR)_vs
endif
ifeq ($(GATEWAY), true)
	CONF_SUBDIR:=$(CONF_SUBDIR)_gw
endif

## additional flags for library ##
## add default includes
CPPINCLUDES=$(INCLUDE_OPT)$(PORT_INC_DIR) $(INCLUDE_OPT)$(INC_DIR) $(INCLUDE_OPT)$(SHARED_DIR) $(INCLUDE_OPT)$(OUT_LIB_PATH)
ifneq ($(DISABLE_MQTT), true)
	CPPINCLUDES+=$(INCLUDE_OPT)$(PORT_SRC_DIR) $(INCLUDE_OPT)$(SHARED_DIR)/external/mqtt/include
endif
CPPFLAGS+=$(WARN_LVL_OPT) $(CPPINCLUDES) $(COMPILE_OPT)

# check debug mode
ifeq ($(DEBUG), true)
	CPPFLAGS+=$(DEBUG_OPT)
endif

## base directories ##
SHARED_DIR=$(SRC_DIR)/shared
BOOTSTRAPPER_DIR=$(SRC_DIR)/bootstrapper
INC_DIR=$(SRC_DIR)/include
SAMPLE_DIR=$(PROJ_DIR)/samples
DOC_DIR=$(PROJ_DIR)/make/doxygen
BST_MAKE_DIR=$(PROJ_DIR)/make/bootstrapper
VRT_SUBDIR=virtualization
MSG_SUBDIR=messaging
PRT_SUBDIR=porting-layer

## base files ##
LICENSE_FILE=$(LICENSE_PATH)/LICENSE

# documentation files
DOC_VERSION_FILE=version
DOC_FILES=$(INC_DIR)/*
# messaging API Headers
_MESSAGING_API_HEADERS =                     \
	advanced/iotcs_storage_object.h          \
	advanced/iotcs_message.h                 \
	advanced/iotcs_messaging.h               \
	iotcs_device.h                           \
	iotcs.h

# device virtualization API Headers
_VIRTUALIZATION_API_HEADERS =                \
	iotcs_storage_object.h            \
	iotcs_device.h                           \
	iotcs_virtual_device.h                   \
	iotcs.h

# porting layer API Headers
_PORTING_LAYER_API_HEADERS =              \
	iotcs_port_queue.h                    \
	iotcs_port_crypto.h                   \
	iotcs_port_diagnostic.h               \
	iotcs_port_memory.h                   \
	iotcs_port_mutex.h                    \
	iotcs_port_ssl.h                      \
	iotcs_port_system.h                   \
	iotcs_port_thread.h

## base variables ##
# platform variables
PLATFORM_OS:=$(subst /,_,$(shell uname -m))
PLATFORM_OS:=$(shell echo $(PLATFORM_OS) | tr A-Z a-z)
FULL_VERSION_NUMBER=$(shell cat $(DOC_DIR)/$(DOC_VERSION_FILE))
# documentation variables
DOXYGEN?=doxygen
MSG_API_DOC_PART?=E70344-25
VRT_API_DOC_PART?=E70343-25
PRT_API_DOC_PART?=E80003-18
PLATFORM_API='C'
VRT_API='Device Virtualization'
MSG_API='Messaging'
PORT_API='Porting Layer'

MESSAGING_API_HEADERS = $(patsubst %,$(INC_DIR)/%,$(_MESSAGING_API_HEADERS))
VIRTUALIZATION_API_HEADERS = $(patsubst %,$(INC_DIR)/%,$(_VIRTUALIZATION_API_HEADERS))
PORTING_LAYER_API_HEADERS = $(patsubst %,$(PORT_INC_DIR)/%,$(_PORTING_LAYER_API_HEADERS))

## output directories ##
TAM_OBJ_DIR=$(OBJ_DIR)/tam
PORT_OBJ_DIR=$(OBJ_DIR)/$(PORT)_port
PORT_TAM_OBJ_DIR=$(TAM_OBJ_DIR)/$(PORT)_port
OUT_DOC_SUBDIR=apidocs

## output variables ##
LINE_TRAP="===================================================="
define SCM
/**\n
endef
define CMT
 *
endef
define ECM
\n */\n
endef
LIB_EXT=a
SHORT_LIB_NAME?=iotdeviceclient
SHORT_TAM_LIB_NAME=tam
OUT_LIB_NAME=lib$(SHORT_LIB_NAME).$(LIB_EXT)
OUT_TAM_LIB_NAME=lib$(SHORT_TAM_LIB_NAME).$(LIB_EXT)

## output files ##
OUT_CONFIG_HEADER=$(OUT_LIB_PATH)/iotcs_config.h

# object files
_OBJ_FILE_NAMES =                           \
	iotcs/iotcs.o                           \
	iotcs/iotcs_client.o                    \
	json/json_helper.o                      \
	json/json_writer.o                      \
	protocol/protocol_request.o             \
	messaging/msg.o                         \
	messaging/msg_alert.o                   \
	messaging/msg_data.o                    \
	messaging/msg_resource.o                \
	messaging/msg_request.o                 \
	messaging/msg_response.o                \
	util/util_array_queue.o                 \
	util/util.o                             \
	util/util_key_value.o                   \
	util/util_resource.o                    \
	util/util_memory.o                      \
	log/log.o                               \
	messaging/msg_request_dispatcher.o      \
	device_model/device_model_capability.o  \
	util/util_buffer.o                      \
	external/json_parser/jsmn.o             \
	external/http_parser/http_parser.o
	
ifeq ($(STORAGE_SUPPORT), true)
	_OBJ_FILE_NAMES+=scs/storage_util.o \
	                 scs/storage_dispatcher.o \
	                 scs/storage_object.o
endif

ifneq ($(DISABLE_HTTP), true)
	_OBJ_FILE_NAMES+=protocol/http/http_wrapper.o
endif

ifneq ($(DISABLE_MQTT), true)
	_OBJ_FILE_NAMES+=protocol/mqtt/mqtt_wrapper.o \
		external/mqtt/src/MQTTClient.o      \
		external/mqtt/src/MQTTConnectClient.o \
		external/mqtt/src/MQTTConnectServer.o \
		external/mqtt/src/MQTTDeserializePublish.o \
		external/mqtt/src/MQTTFormat.o \
		external/mqtt/src/MQTTPacket.o \
		external/mqtt/src/MQTTSerializePublish.o \
		external/mqtt/src/MQTTSubscribeClient.o \
		external/mqtt/src/MQTTSubscribeServer.o \
		external/mqtt/src/MQTTUnsubscribeClient.o \
		external/mqtt/src/MQTTUnsubscribeServer.o
endif

_GATEWAY_OBJ_FILE_NAMES =                   \
	iotcs/iotcs_gateway.o

_ASYNC_MD_OBJ_FILE_NAMES =                  \
	messaging/msg_dispatcher.o				\
	messaging/msg_dispatcher.o	            \
	iotcs/iotcs_thread_pool.o

_POLICY_OBJ_FILE_NAMES =                    \
	policy/device_policy.o

_MESSAGE_PERS_OBJ_FILE_NAMES =                  \
	messaging/message_persistence.o	    \
	messaging/sqlite3.o	

_STATIC_HEAP_OBJ_FILE_NAMES =               \
	external/umm_malloc/umm_malloc.o

_VIRT_OBJ_FILE_NAMES =                      \
	device_model/device_model_parser.o      \
	device_model/device_model.o

TAM_OBJ_FILE_NAMES =                        \
	trusted_assets_manager/iotcs_tam.o
OBJ_FILES?=
OBJ_FILES+=$(patsubst %,$(OBJ_DIR)/%,$(_OBJ_FILE_NAMES))
ifeq ($(GATEWAY), true)
	OBJ_FILES+=$(patsubst %,$(OBJ_DIR)/%,$(_GATEWAY_OBJ_FILE_NAMES))
endif
$(info "IMPLICIT_EDGE_COMPUTING "$IMPLICIT_EDGE_COMPUTING)
ifeq ($(MESSAGE_DISPATCHER), true)
	OBJ_FILES+=$(patsubst %,$(OBJ_DIR)/%,$(_ASYNC_MD_OBJ_FILE_NAMES))
endif
ifeq ($(IMPLICIT_EDGE_COMPUTING), true)
	OBJ_FILES+=$(patsubst %,$(OBJ_DIR)/%,$(_POLICY_OBJ_FILE_NAMES))
endif
ifeq ($(MESSAGE_PERSISTENCE),true)
ifeq ($(PORT),mbed)
	MESSAGE_PERSISTENCE:=false
else
	OBJ_FILES+=$(patsubst %,$(OBJ_DIR)/%,$(_MESSAGE_PERS_OBJ_FILE_NAMES))
endif
endif
ifeq ($(VIRTUALIZATION_SUPPORT), true)
	OBJ_FILES+=$(patsubst %,$(OBJ_DIR)/%,$(_VIRT_OBJ_FILE_NAMES))
endif

ifeq ($(MESSAGING_THREAD_SAFETY), true)
	USE_STATIC_HEAP:=false
else
	ifeq ($(USE_STATIC_HEAP), true)
	OBJ_FILES+=$(patsubst %,$(OBJ_DIR)/%,$(_STATIC_HEAP_OBJ_FILE_NAMES))
	endif
endif

TAM_OBJ_FILES=$(patsubst %,$(TAM_OBJ_DIR)/%,$(TAM_OBJ_FILE_NAMES))

## temporary directories ##
TMP_DIR=$(OUTPUT_DIR)/tmp
# documentation directories
TMP_MSG_DOC_SUBDIR=msg_doc_src
TMP_VRT_DOC_SUBDIR=vrt_doc_src
TMP_PORT_DOC_SUBDIR=port_doc_src

## temporary variables ##

## temporary files ##
# should be created if environment checked
CHECK_DEPENDENCY=checked.o

# targets for make directories
$(OBJ_DIR) $(TAM_OBJ_DIR) $(OUT_LIB_PATH) $(TMP_DIR) $(OUTPUT_DIR) $(PORT_OBJ_DIR):
	$(A)mkdir -p $@

# targets for compile shared sources
$(OBJ_DIR)/%.o: $(SHARED_DIR)/%.c
	$(A)mkdir -p $(dir $@)
	$(A)$(CC) $(CPPFLAGS) $(CPP_OPTS) $(CFLAGS) $(C_OPTS) $(LIB_FLAGS) $(LIB_OPTS) $(OBJECT_FILE_NAME_OPT)$@ $<

# targets for compile TAM sources
$(TAM_OBJ_DIR)/%.o: $(SHARED_DIR)/%.c
	$(A)mkdir -p $(dir $@)
	$(A)$(CC) $(CPPFLAGS) $(CPP_OPTS) $(CFLAGS) $(C_OPTS) $(TAM_FLAGS) $(TAM_OPTS) $(OBJECT_FILE_NAME_OPT)$@ $<

# documentation related targets
.pre_doc: $(OUTPUT_DIR)
	@printf "\nGenerate documentation bundle...\n"

.doc_impl: .vrt_doc .msg_doc .port_doc
	$(A)cp -R $(V) $(TMP_DIR)/$(OUT_DOC_SUBDIR) $(OUTPUT_DIR)

.vrt_doc:
	@printf "\n\nGenerate Virtualization API documentation...\n"
	$(A)mkdir -p $(TMP_DIR)/$(TMP_VRT_DOC_SUBDIR)
	$(A)mkdir -p $(TMP_DIR)/$(OUT_DOC_SUBDIR)/$(VRT_SUBDIR)
	$(A)cp $(VIRTUALIZATION_API_HEADERS) $(TMP_DIR)/$(TMP_VRT_DOC_SUBDIR)
	$(A)cp $(DOC_DIR)/virtualization_overview.txt $(TMP_DIR)/$(TMP_VRT_DOC_SUBDIR)
	$(A)cd $(TMP_DIR)/$(TMP_VRT_DOC_SUBDIR); PLATFORM_API=$(PLATFORM_API) API_TYPE=$(VRT_API) RELEASE_VERSION=$(FULL_VERSION_NUMBER) DOC_DIR=./../../$(DOC_DIR) DOC_PART=$(VRT_API_DOC_PART) $(DOXYGEN) ./../../$(DOC_DIR)/documentation.conf
	$(A)cp $(LICENSE_FILE) $(TMP_DIR)/$(TMP_VRT_DOC_SUBDIR)/doc
	$(A)cp -R $(V) $(TMP_DIR)/$(TMP_VRT_DOC_SUBDIR)/doc/* $(TMP_DIR)/$(OUT_DOC_SUBDIR)/$(VRT_SUBDIR)

.msg_doc:
	@printf "\n\nGenerate Messaging API documentation...\n"
	$(A)mkdir -p $(TMP_DIR)/$(TMP_MSG_DOC_SUBDIR)
	$(A)mkdir -p $(TMP_DIR)/$(OUT_DOC_SUBDIR)/$(MSG_SUBDIR)
	$(A)cp $(MESSAGING_API_HEADERS) $(TMP_DIR)/$(TMP_MSG_DOC_SUBDIR)
	$(A)cp $(DOC_DIR)/messaging_overview.txt $(TMP_DIR)/$(TMP_MSG_DOC_SUBDIR)
	$(A)cd $(TMP_DIR)/$(TMP_MSG_DOC_SUBDIR); PLATFORM_API=$(PLATFORM_API) API_TYPE=$(MSG_API) RELEASE_VERSION=$(FULL_VERSION_NUMBER) DOC_DIR=./../../$(DOC_DIR) DOC_PART=$(MSG_API_DOC_PART) $(DOXYGEN) ./../../$(DOC_DIR)/documentation.conf
	$(A)cp $(LICENSE_FILE) $(TMP_DIR)/$(TMP_MSG_DOC_SUBDIR)/doc
	$(A)cp -R $(V) $(TMP_DIR)/$(TMP_MSG_DOC_SUBDIR)/doc/* $(TMP_DIR)/$(OUT_DOC_SUBDIR)/$(MSG_SUBDIR)

.port_doc:
	@printf "\nGenerate the port layer API documentation...\n"
	$(A)mkdir -p $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR)
	$(A)mkdir -p $(TMP_DIR)/$(OUT_DOC_SUBDIR)/$(PRT_SUBDIR)
	$(A)cp $(PORTING_LAYER_API_HEADERS) $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR)
	$(A)cp $(DOC_DIR)/porting_layer_overview.txt $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR)
	$(A)cp $(DOC_DIR)/porting_guide.html $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR)
	$(A)cp $(DOC_DIR)/Arch.jpg $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR)
	$(A)cd $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR); PLATFORM_API=$(PLATFORM_API) API_TYPE=$(PORT_API) RELEASE_VERSION=$(FULL_VERSION_NUMBER) DOC_DIR=./../../$(DOC_DIR) DOC_PART=$(PRT_API_DOC_PART) $(DOXYGEN) ./../../$(DOC_DIR)/documentation.conf
	$(A)cp $(LICENSE_FILE) $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR)/doc
	$(A)cp -R $(V) $(TMP_DIR)/$(TMP_PORT_DOC_SUBDIR)/doc/* $(TMP_DIR)/$(OUT_DOC_SUBDIR)/$(PRT_SUBDIR)

.post_doc:
	$(A)rm -rf $(TMP_DIR)
	@printf "Documentation generated !\n"

# check environment
.PHONY: check
check: .check_impl

# print build configuration
.PHONY: print_conf
print_conf:
	@printf "%s\n" "$(LINE_TRAP)";                            \
	printf "PLATFORM_OS     = %s\n" "$(PLATFORM_OS)";         \
	printf "PORT            = %s\n" "$(PORT)";                \
	printf "LIB_CFG         = %s\n" "$(LIB_CFG)";             \
	printf "CC_CFG          = %s\n" "$(CC_CFG)";              \
	printf "OUT_CC_SUBDIR   = %s\n" "$(OUT_CC_SUBDIR)";       \
	printf "TOOLS_PREFIX    = %s\n" "$(TOOLS_PREFIX)";        \
	$(if $(CPP_OPTS), printf "CPP_OPTS        = %s\n" "$(CPP_OPTS)"; )   \
	$(if $(C_OPTS), printf "C_OPTS          = %s\n" "$(C_OPTS)"; )       \
	$(if $(CXX_OPTS), printf "CXX_OPTS        = %s\n" "$(CXX_OPTS)"; )   \
	$(if $(AS_OPTS), printf "AS_OPTS         = %s\n" "$(AS_OPTS)"; )     \
	$(if $(LD_OPTS), printf "LD_OPTS         = %s\n" "$(LD_OPTS)"; )     \
	$(if $(LIB_OPTS), printf "LIB_OPTS        = %s\n" "$(LIB_OPTS)"; )   \
	$(if $(PORT_OPTS), printf "PORT_OPTS       = %s\n" "$(PORT_OPTS)"; ) \
	$(if $(TAM_OPTS), printf "TAM_OPTS        = %s\n" "$(TAM_OPTS)"; )   \
	printf "%s\n" "$(LINE_TRAP)"

# print library configuration
.PHONY: print_libconf
print_libconf: .print_common_libconf .print_libconf_impl
	@printf "%s\n" "$(LINE_TRAP)"

.print_common_libconf:
	@printf "IoTCS Library \"%s\" configuration:\n" "$(FULL_VERSION_NUMBER)";  \
	printf "%s\n" "$(LINE_TRAP)";                                              \
	printf "  MESSAGING_THREAD_SAFETY     = %s\n" "$(MESSAGING_THREAD_SAFETY)";\
	printf "  MESSAGE_DISPATCHER          = %s\n" "$(MESSAGE_DISPATCHER)";     \
	printf "  IMPLICIT_EDGE_COMPUTING     = %s\n" "$(IMPLICIT_EDGE_COMPUTING)";\
	printf "  VIRTUALIZATION_SUPPORT      = %s\n" "$(VIRTUALIZATION_SUPPORT)"; \
	printf "  GATEWAY                     = %s\n" "$(GATEWAY)";                \
	printf "  USE_STATIC_HEAP             = %s\n" "$(USE_STATIC_HEAP)";        \
	$(if $(filter true,$(USE_STATIC_HEAP)),                                    \
		printf "  STATIC_HEAP_SIZE            = %s\n" "$(STATIC_HEAP_SIZE)";)  \
	printf "  USE_DRAFT_DEVICE_MODELS     = %s\n" "$(USE_DRAFT_DEVICE_MODELS)";\
	printf "  DEVICE_MODEL_DIR            = %s\n" "$(DEVICE_MODEL_DIR)";       \
	printf "  USE_SERVER_TIME             = %s\n" "$(USE_SERVER_TIME)";        \
	printf "  DEBUG                       = %s\n" "$(DEBUG)";                  \
	printf "  DISABLE_MQTT                = %s\n" "$(DISABLE_MQTT)";           \
	printf "  DISABLE_MQTT_ACCEPTED_BYTES = %s\n" "$(DISABLE_MQTT_ACCEPTED_BYTES)"; \
	printf "  DISABLE_HTTP                = %s\n" "$(DISABLE_HTTP)";           \
	printf "  DEFAULT_TAM                 = %s\n" "$(DEFAULT_TAM)";            \
	printf "  LONG_POLLING                = %s\n" "$(LONG_POLLING)";           \
	printf "  USE_DIAGNOSTIC_CAPABILITY   = %s\n" "$(USE_DIAGNOSTIC_CAPABILITY)"; \
	printf "  LOG_LEVEL                   = %s\n" "$(LOG_LEVEL)";              \
	printf "  LOG_USE_FILE_LINE           = %s\n" "$(LOG_USE_FILE_LINE)";      \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_CORE)),              \
		printf "  LOG_LEVEL_CHANNEL_CORE      = %s\n" "$(LOG_LEVEL_CHANNEL_CORE)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_DM)),                \
		printf "  LOG_LEVEL_CHANNEL_DM        = %s\n" "$(LOG_LEVEL_CHANNEL_DM)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_EXT)),               \
		printf "  LOG_LEVEL_CHANNEL_EXT       = %s\n" "$(LOG_LEVEL_CHANNEL_EXT)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_JSON)),              \
		printf "  LOG_LEVEL_CHANNEL_JSON      = %s\n" "$(LOG_LEVEL_CHANNEL_JSON)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_MSG)),               \
		printf "  LOG_LEVEL_CHANNEL_MSG       = %s\n" "$(LOG_LEVEL_CHANNEL_MSG)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PROTO)),             \
		printf "  LOG_LEVEL_CHANNEL_PROTO     = %s\n" "$(LOG_LEVEL_CHANNEL_PROTO)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_TAM)),               \
		printf "  LOG_LEVEL_CHANNEL_TAM       = %s\n" "$(LOG_LEVEL_CHANNEL_TAM)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_UTIL)),              \
		printf "  LOG_LEVEL_CHANNEL_UTIL      = %s\n" "$(LOG_LEVEL_CHANNEL_UTIL)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_LOG)),               \
		printf "  LOG_LEVEL_CHANNEL_LOG       = %s\n" "$(LOG_LEVEL_CHANNEL_LOG)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_QUEUE)),       \
		printf "  LOG_LEVEL_CHANNEL_PORT_QUEUE = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_QUEUE)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_CRYPTO)),       \
		printf "  LOG_LEVEL_CHANNEL_PORT_CRYPTO = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_CRYPTO)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_TAM)),          \
		printf "  LOG_LEVEL_CHANNEL_PORT_TAM  = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_TAM)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_DIAG)),         \
		printf "  LOG_LEVEL_CHANNEL_PORT_DIAG = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_DIAG)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_MEM)),          \
		printf "  LOG_LEVEL_CHANNEL_PORT_MEM  = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_MEM)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_MUTEX)),        \
		printf "  LOG_LEVEL_CHANNEL_PORT_MUTEX  = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_MUTEX)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_SSL)),          \
		printf "  LOG_LEVEL_CHANNEL_PORT_SSL  = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_SSL)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_SYSTEM)),        \
		printf "  LOG_LEVEL_CHANNEL_PORT_SYSTEM = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_SYSTEM)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_THREAD)),        \
		printf "  LOG_LEVEL_CHANNEL_PORT_THREAD = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_THREAD)"; ) \
	$(if $(filter-out IOTCS_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_MQTT)),          \
		printf "  LOG_LEVEL_CHANNEL_PORT_MQTT = %s\n" "$(LOG_LEVEL_CHANNEL_PORT_MQTT)"; )

.print_lib_src:
	@printf "Compile source files of %s library shared part...\n" "$(PORT)"

.print_port_src:
	@printf "Compile source files of %s library port layer...\n" "$(PORT)"

.print_tam_src:
	@printf "Compile source files of %s TAM part...\n" "$(PORT)"

.print_port_tam_src:
	@printf "Compile source files of %s TAM port layer...\n" "$(PORT)"

# generate library configuration header
.PHONY: gen_libconf
gen_libconf: .gen_libconf_header .gen_common_libconf .gen_libconf_impl .gen_libconf_footer

.gen_libconf_header:
	$(A)rm -f $(OUT_CONFIG_HEADER)
	@printf "\nGenerate config header file...\n"
	@printf "$(SCM)$(CMT) This set of compile-time options may be used to enable\n" > $(OUT_CONFIG_HEADER); \
	printf "$(CMT) or disable features selectively$(ECM)" >> $(OUT_CONFIG_HEADER); \
	printf "#ifndef IOTCS_CONFIG_H\n#define IOTCS_CONFIG_H\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "#ifdef __cplusplus\nextern \"C\" {\n#endif\n\n" >> $(OUT_CONFIG_HEADER);

.gen_libconf_footer:
	@printf "\n#ifdef __cplusplus\n}\n#endif\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "#endif /* IOTCS_CONFIG_H */\n" >> $(OUT_CONFIG_HEADER);
	@printf "Config header generated !\n\n"

.gen_common_libconf: .gen_lib_cfg .gen_lib_common .gen_tam_common

.gen_lib_cfg:
	@printf "Generate library configuration part...\n"
	@printf "$(SCM)$(CMT) %s \n" "$(LINE_TRAP)" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Base library configuration \n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) %s $(ECM)\n" "$(LINE_TRAP)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable tread safety for Messaging API$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(MESSAGING_THREAD_SAFETY)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_MESSAGING_THREAD_SAFETY\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable async message dispatcher support\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Requires: IOTCS_MESSAGING_THREAD_SAFETY enabled$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(MESSAGE_DISPATCHER)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_MESSAGE_DISPATCHER\n\n" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(IMPLICIT_EDGE_COMPUTING)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_IMPLICIT_EDGE_COMPUTING\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable Virtualization API support\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Requires: IOTCS_MESSAGE_DISPATCHER enabled$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(VIRTUALIZATION_SUPPORT)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_VIRTUALIZATION_SUPPORT\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable indirect activation support$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(GATEWAY)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_GATEWAY\n\n" >> $(OUT_CONFIG_HEADER);
	$(if $(filter-out true,$(MESSAGE_PERSISTENCE)), printf "//" >> $(OUT_CONFIG_HEADER);)
	$(if $(filter mbed,$(PORT)), printf "//" >> $(OUT_CONFIG_HEADER);)
	printf "#define IOTCS_MESSAGE_PERSISTENCE\n\n" >> $(OUT_CONFIG_HEADER);

.pre_gen_lib_common:
	@printf "Generate shared library settings...\n"
	@printf "$(SCM)$(CMT) %s \n" "$(LINE_TRAP)" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Shared library settings\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) %s $(ECM)\n" "$(LINE_TRAP)" >> $(OUT_CONFIG_HEADER);

.gen_lib_common: .pre_gen_lib_common .gen_lib_features .gen_lib_const .gen_lib_logging

.gen_lib_features:
	@printf "$(SCM)$(CMT) Enable static heap instead on libc's malloc/free\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Requires: IOTCS_MESSAGING_THREAD_SAFETY disabled$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(and $(filter-out true,$(MESSAGING_THREAD_SAFETY)), $(filter true,$(USE_STATIC_HEAP))),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_USE_STATIC_HEAP\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Override static heap size in bytes (by default 8192)\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Requires: IOTCS_USE_STATIC_HEAP enabled$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(and $(filter-out true,$(MESSAGING_THREAD_SAFETY)), $(filter true,$(USE_STATIC_HEAP))),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_STATIC_HEAP_SIZE %s\n\n" "$(STATIC_HEAP_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable creation of draft Device Models$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(USE_DRAFT_DEVICE_MODELS)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_USE_DRAFT_DEVICE_MODELS\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable static usage time from server$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(USE_SERVER_TIME)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_USE_SERVER_TIME\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Specify to enable device model caching on local file system$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(DEVICE_MODEL_DIR),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_DEVICE_MODEL_DIR \"%s\"\n\n" "$(DEVICE_MODEL_DIR)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable debug mode$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(DEBUG)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_DEBUG\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Disable mqtt support$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(DISABLE_MQTT)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_DISABLE_MQTT\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Disable mqtt accepted bytes support\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Requires: IOTCS_DISABLE_MQTT disabled$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(and $(filter-out true,$(DISABLE_MQTT)),$(filter true,$(DISABLE_MQTT_ACCEPTED_BYTES))),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_DISABLE_MQTT_ACCEPTED_BYTES\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Disable http support$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(DISABLE_HTTP)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_DISABLE_HTTP\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable the long polling implementation$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(LONG_POLLING)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_LONG_POLLING\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) enable the storage cloud server support implementation.$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(STORAGE_SUPPORT)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_STORAGE_SUPPORT\n\n" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(DISABLE_STORAGE_OBJECT_PREFIX)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define DISABLE_STORAGE_OBJECT_PREFIX 1\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Enable diagnostic capability API usage in library$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(USE_DIAGNOSTIC_CAPABILITY)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_USE_DIAGNOSTIC_CAPABILITY\n\n" >> $(OUT_CONFIG_HEADER);

.gen_lib_const:
	@printf "$(SCM)$(CMT) The size of buffer for device model filename in bytes$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_DEVICE_MODEL_FILENAME_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_DEVICE_MODEL_FILENAME_BUFFER_SIZE %s\n\n" "$(IOTCS_DEVICE_MODEL_FILENAME_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The size of buffer for device model capability server response in bytes$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_CAPABILITY_RESPONSE_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_CAPABILITY_RESPONSE_BUFFER_SIZE %s\n\n" "$(IOTCS_CAPABILITY_RESPONSE_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The polling timeout in milliseconds for internal async message dispatcher$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_POLLING_TIMEOUT_MS),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_POLLING_TIMEOUT_MS %s\n\n" "$(IOTCS_POLLING_TIMEOUT_MS)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The maximum number messages to send in one request$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_MAX_MESSAGES_FOR_SEND),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_MAX_MESSAGES_FOR_SEND %s\n\n" "$(IOTCS_MAX_MESSAGES_FOR_SEND)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The size of buffer for reason of async message send failure in bytes$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_ASYNC_FAIL_REASON_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_ASYNC_FAIL_REASON_BUFFER_SIZE %s\n\n" "$(IOTCS_ASYNC_FAIL_REASON_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The size of buffer for request message in bytes$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_REQUEST_MESSAGE_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_REQUEST_MESSAGE_BUFFER_SIZE %s\n\n" "$(IOTCS_REQUEST_MESSAGE_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The number of request messages$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_REQUEST_MESSAGE_NUMBER),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_REQUEST_MESSAGE_NUMBER %s\n\n" "$(IOTCS_REQUEST_MESSAGE_NUMBER)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The maximum number of resource for registration$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_MAX_RESOURCE_NUMBER),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_MAX_RESOURCE_NUMBER %s\n\n" "$(IOTCS_MAX_RESOURCE_NUMBER)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_REQUEST_HANDLER_THREAD_POOL_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_REQUEST_HANDLER_THREAD_POOL_SIZE %s\n\n" "$(IOTCS_REQUEST_HANDLER_THREAD_POOL_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The server host buffer length in bytes\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) The default value is 128 + 14, where 128 is got from TAM store\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) and 14 is the length of string 'server.host=' + '/n' + '/0'$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_SERVER_HOST_BUFFER_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_SERVER_HOST_BUFFER_LENGTH %s\n\n" "$(IOTCS_SERVER_HOST_BUFFER_LENGTH)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The client ID buffer length in bytes\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) The default value is 20 + 12, where 12 is the length of\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) string 'client.id'= + '/n' + '/0'$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_CLIENT_ID_BUFFER_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_CLIENT_ID_BUFFER_LENGTH %s\n\n" "$(IOTCS_CLIENT_ID_BUFFER_LENGTH)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The shared secret buffer length in bytes\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) The default value is 50 + 15, where 15 is the length\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) string 'client.secret' + '/n' + '/0'$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_SHARED_SECRET_BUFFER_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_SHARED_SECRET_BUFFER_LENGTH %s\n\n" "$(IOTCS_SHARED_SECRET_BUFFER_LENGTH)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The signature buffer length in bytes$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_SIGNATURE_BUFFER_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_SIGNATURE_BUFFER_LENGTH %s\n\n" "$(IOTCS_SIGNATURE_BUFFER_LENGTH)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The assets signature buffer length in bytes$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_ASSETS_SIGNATURE_BUFFER_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_ASSETS_SIGNATURE_BUFFER_LENGTH %s\n\n" "$(IOTCS_ASSETS_SIGNATURE_BUFFER_LENGTH)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The public key buffer length in bytes\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) The default value is 350. It's enough for public key with length 2048 bits$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_PUBLIC_KEY_BUFFER_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_PUBLIC_KEY_BUFFER_LENGTH %s\n\n" "$(IOTCS_PUBLIC_KEY_BUFFER_LENGTH)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The server port length in bytes\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) The default value is 19, where 12 is the length of 'server.port=' + 5\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) is max port value '65535' + 1('/n') + 1('/0') = 19$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_SERVER_PORT_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_SERVER_PORT_LENGTH %s\n\n" "$(IOTCS_SERVER_PORT_LENGTH)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The payload buffer size in bytes. It is used for storing messages payload.\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) The less then IOTCS_MIN_PAYLOAD_BUFFER_SIZE value is not acceptable \n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) for IOTCS_PAYLOAD_BUFFER_SIZE and compile time error is generated in this case. \n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Also could be reused for other operation (e.g. reading of the trusted assets store).$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_PAYLOAD_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_PAYLOAD_BUFFER_SIZE %s\n\n" "$(IOTCS_PAYLOAD_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Buffer is used for MQTT frames.\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) NOTE: Publish frames doesn't contain a payload, the payload is written to socket from payload buffer directly.$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_MQTT_SEND_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_MQTT_SEND_BUFFER_SIZE %s\n\n" "$(IOTCS_MQTT_SEND_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The mqtt topic name buffer size$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_MQTT_SUB_TOPIC_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_MQTT_SUB_TOPIC_BUFFER_SIZE %s\n\n" "$(IOTCS_MQTT_SUB_TOPIC_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The timeout for waiting of the response for mqtt request$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_MQTT_CMD_TIMEOUT_MS),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_MQTT_CMD_TIMEOUT_MS %s\n\n" "$(IOTCS_MQTT_CMD_TIMEOUT_MS)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Represents how much time the library will provide to user to set the resource handlers.\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) NOTE: Requests without user handler could be processed later during this time.$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_SETTLE_TIME_MS),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_SETTLE_TIME_MS %s\n\n" "$(IOTCS_SETTLE_TIME_MS)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The size for default buffer used for logging$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_LOG_MSG_BUFFER_SIZE),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_LOG_MSG_BUFFER_SIZE %s\n\n" "$(IOTCS_LOG_MSG_BUFFER_SIZE)" >> $(OUT_CONFIG_HEADER);
	printf "$(SCM)$(CMT) Represents how much jsmntok_t instances should be allocated for jsmn json parser.$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_JSON_TOKEN_NUM),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_JSON_TOKEN_NUM %s\n\n" "$(IOTCS_JSON_TOKEN_NUM)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Length of the available buffer for response from storage cloud server.\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) NOTE: Also this buffer uses for chunked data sending to scs server. Increasing of this value will increase throughput.$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(IOTCS_SCS_RESPONSE_BUFFER_LENGTH),, printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_SCS_RESPONSE_BUFFER_LENGTH %s\n\n" "$(IOTCS_SCS_RESPONSE_BUFFER_LENGTH)" >> $(OUT_CONFIG_HEADER); \

# $(1) - log level variable
# $(2) - log level variable name
define SET_LOG_LEVEL
$(if $(filter $(1),none no critical crit error err warning warn info debug dbg all), \
	printf "#define IOTCS_$(2) " >> $(OUT_CONFIG_HEADER); \
	$(if $(or $(filter $(1),none), $(filter $(1),no)), printf "0" >> $(OUT_CONFIG_HEADER);) \
	$(if $(or $(filter $(1),critical),$(filter $(1),crit)), printf "1" >> $(OUT_CONFIG_HEADER);) \
	$(if $(or $(filter $(1),error),$(filter $(1),err)), printf "2" >> $(OUT_CONFIG_HEADER);) \
	$(if $(or $(filter $(1),warning),$(filter $(1),warn)), printf "3" >> $(OUT_CONFIG_HEADER);) \
	$(if $(filter $(1),info), printf "4" >> $(OUT_CONFIG_HEADER);) \
	$(if $(or $(filter $(1),debug),$(filter $(1),dbg)), printf "5" >> $(OUT_CONFIG_HEADER);) \
	$(if $(filter $(1),all), printf "5" >> $(OUT_CONFIG_HEADER);) \
	printf "\n\n" >> $(OUT_CONFIG_HEADER); \
, \
	printf "#define IOTCS_$(2) %s\n\n" "$(1)" >> $(OUT_CONFIG_HEADER); \
)
endef

# Supported log levels:
# * 0 = none|no       - No log
# * 1 = critical|crit - Critical log entry: unrecoverable internal error - library can't work properly anymore
# * 2 = error|err     - Error log entry: recoverable error, user action is required
# * 3 = warning|warn  - Warning log entry: recoverable error, no user action is required
# * 4 = info          - Information log entry: not an error
# * 5 = debug|dbg     - Debug log entry
# * 5 = all

.gen_lib_logging:
	@printf "$(SCM)$(CMT) %s \n" "$(LINE_TRAP)" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) Logging settings\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) %s $(ECM)\n" "$(LINE_TRAP)" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) Supported log levels:\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) 0 = none|no       - No log\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) 1 = critical|crit - Critical log entry: unrecoverable internal error - library can't work properly anymore\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) 2 = error|err     - Error log entry: recoverable error, user action is required\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) 3 = warning|warn  - Warning log entry: recoverable error, no user action is required\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) 4 = info          - Information log entry: not an error\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) 5 = debug|dbg     - Debug log entry\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) 5 = all$(ECM)\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The global library log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL),LOG_LEVEL) \
	printf "$(SCM)$(CMT) Enable include base file name and line number in the log entry\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(CMT) It is implicitly turned on by default for debug mode$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(if $(filter-out true,$(LOG_USE_FILE_LINE)), printf "//" >> $(OUT_CONFIG_HEADER);) \
	printf "#define IOTCS_LOG_USE_FILE_LINE " >> $(OUT_CONFIG_HEADER); \
	$(if $(filter $(LOG_USE_FILE_LINE),0 1), printf "$(LOG_USE_FILE_LINE)" >> $(OUT_CONFIG_HEADER);) \
	printf "\n\n" >> $(OUT_CONFIG_HEADER); \
	printf "$(SCM)$(CMT) The iotcs module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_CORE),LOG_LEVEL_CHANNEL_CORE) \
	printf "$(SCM)$(CMT) The device model channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_DM),LOG_LEVEL_CHANNEL_DM) \
	printf "$(SCM)$(CMT) The external module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_EXT),LOG_LEVEL_CHANNEL_EXT) \
	printf "$(SCM)$(CMT) The json module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_JSON),LOG_LEVEL_CHANNEL_JSON) \
	printf "$(SCM)$(CMT) The messaging module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_MSG),LOG_LEVEL_CHANNEL_MSG) \
	printf "$(SCM)$(CMT) The protocol module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PROTO),LOG_LEVEL_CHANNEL_PROTO) \
	printf "$(SCM)$(CMT) The TAM module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_TAM),LOG_LEVEL_CHANNEL_TAM) \
	printf "$(SCM)$(CMT) The util module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_UTIL),LOG_LEVEL_CHANNEL_UTIL) \
	printf "$(SCM)$(CMT) The logging module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_LOG),LOG_LEVEL_CHANNEL_LOG) \
	printf "$(SCM)$(CMT) The port blocking queue module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_QUEUE),LOG_LEVEL_CHANNEL_PORT_QUEUE) \
	printf "$(SCM)$(CMT) The port crypto module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_CRYPTO),LOG_LEVEL_CHANNEL_PORT_CRYPTO) \
	printf "$(SCM)$(CMT) The port TAM module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_TAM),LOG_LEVEL_CHANNEL_PORT_TAM) \
	printf "$(SCM)$(CMT) The port diagnostic module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_DIAG),LOG_LEVEL_CHANNEL_PORT_DIAG) \
	printf "$(SCM)$(CMT) The port memory module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_MEM),LOG_LEVEL_CHANNEL_PORT_MEM) \
	printf "$(SCM)$(CMT) The port mutex module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_MUTEX),LOG_LEVEL_CHANNEL_PORT_MUTEX) \
	printf "$(SCM)$(CMT) The port SSL module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_SSL),LOG_LEVEL_CHANNEL_PORT_SSL) \
	printf "$(SCM)$(CMT) The port system module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_SYSTEM),LOG_LEVEL_CHANNEL_PORT_SYSTEM) \
	printf "$(SCM)$(CMT) The port thread module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_THREAD),LOG_LEVEL_CHANNEL_PORT_THREAD) \
	printf "$(SCM)$(CMT) The port MQTT module channel log level$(ECM)" >> $(OUT_CONFIG_HEADER); \
	$(call SET_LOG_LEVEL,$(LOG_LEVEL_CHANNEL_PORT_MQTT),LOG_LEVEL_CHANNEL_PORT_MQTT)

.gen_tam_common:
	@printf "Generate tam library setting...\n"

# used for library settings set
ifdef LIB_CFG_LIST
BUILD_SUFFIX=_$(LIB_CFG)

ifneq ($(firstword $(MAKEFILE_LIST)), )
	CUR_MAKEFILE=-f $(firstword $(MAKEFILE_LIST))
endif

.pre_build$(BUILD_SUFFIX):
	@printf "\nStart build library for configuration \"%s\"...\n" "$(LIB_CFG)"

.build_impl$(BUILD_SUFFIX): $(LIB_CFG_LIST)

$(LIB_CFG_LIST):
	@printf "\nBuild library for configuration \"%s\"\n" "$@"
# remove unwanted arguments from MAKEOVERRIDES
	$(eval MAKEOVERRIDES:=$(filter-out LIB_CFG=%,$(MAKEOVERRIDES)))
	$(A)$(MAKE) $(CUR_MAKEFILE) build LIB_CFG=$@ $(MAKEOVERRIDES)

.post_build$(BUILD_SUFFIX):
	@printf "\nBuild library for configuration \"%s\" done !\n" "$(LIB_CFG)"
endif
