/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL). See the LICENSE file in the root
 * directory for license terms. You may choose either license, or both.
 */

#include <stdio.h>
#include <string.h>
#include <limits.h>

#include "advanced/iotcs_messaging.h"
#include "advanced/iotcs_message.h"
#include "messaging/msg_private.h"
#include "device_model/device_model_private.h"
#include "iotcs/iotcs_private.h"
#include "device_model/device_model_capability.h"
#include "util/util_thread_private.h"
#include "iotcs_port_thread.h"
#include "iotcs_port_system.h"
#include "iotcs_port_queue.h"
#include "iotcs_port_ssl.h"

#define STOP_MESSAGE_BASE 1
#define UPDATE_POLLING_INTERVAL_MESSAGE_BASE 2

#ifndef IOTCS_SETTLE_TIME_MS
#define IOTCS_SETTLE_TIME_MS 5000
#endif

#ifndef IOTCS_MAX_MESSAGES_FOR_SEND
#define IOTCS_MAX_MESSAGES_FOR_SEND 5
#endif

#define MSG_STOP_RECEIVE_REQUEST ((iotcs_request_message *)1)

#define MSG_INF_POLLING_INTERVAL -1

#ifndef IOTCS_ASYNC_FAIL_REASON_BUFFER_SIZE
#define IOTCS_ASYNC_FAIL_REASON_BUFFER_SIZE 256
#endif
static char g_async_send_fail_reason_buf[IOTCS_ASYNC_FAIL_REASON_BUFFER_SIZE];

static iotcs_message STOP_MESSAGE = {.base = (const iotcs_message_base*) STOP_MESSAGE_BASE};
static iotcs_message UPDATE_POLLING_INTERVAL_MESSAGE = {.base = (const iotcs_message_base*) UPDATE_POLLING_INTERVAL_MESSAGE_BASE};
static iotcs_port_message_queue g_send_queue;
static iotcs_port_thread g_send_thread;
static iotcs_port_thread g_recv_thread;
static volatile int64_t g_polling_timeout_ms;

static iotcs_message_dispatcher_delivery_callback g_delivery_cb = NULL;
iotcs_message_dispatcher_error_callback g_error_cb = NULL;

static cl_message_dispatcher_counters g_message_dispatcher_counters;

#ifdef IOTCS_LONG_POLLING
static iotcs_bool g_use_long_polling = IOTCS_FALSE;
#endif

static int64_t g_settle_time;

static void set_settle_time() {
    g_settle_time = iotcs_port_get_current_time_millis() + IOTCS_SETTLE_TIME_MS;
}

iotcs_bool cl_is_settle_time_passed() {
    return iotcs_port_get_current_time_millis() > g_settle_time ? IOTCS_TRUE : IOTCS_FALSE;
}

void iotcs_message_dispatcher_set_delivery_callback(iotcs_message_dispatcher_delivery_callback delivery_cb) {
    g_delivery_cb = delivery_cb;
}

void iotcs_message_dispatcher_set_error_callback(iotcs_message_dispatcher_error_callback error_cb) {
    g_error_cb = error_cb;
}

iotcs_result iotcs_message_dispatcher_queue(iotcs_message *message) {
    if (!message) {
        return IOTCS_RESULT_INVALID_ARGUMENT;
    }

    return iotcs_port_message_queue_put(g_send_queue, message, -1);
}

void cl_message_dispatcher_counters_add_bytes_sent(int64_t value) {
    if (value > 0) {
        g_message_dispatcher_counters.total_bytes_sent += value;
    }
}

void cl_message_dispatcher_counters_add_bytes_received(int64_t value) {
    if (value > 0) {
        g_message_dispatcher_counters.total_bytes_received += value;
    }
}

int64_t cl_message_dispatcher_get_polling_interval(void) {
    return g_polling_timeout_ms == MSG_INF_POLLING_INTERVAL ? 0 : g_polling_timeout_ms;
}

void cl_message_dispatcher_set_polling_interval(int64_t value) {
    g_polling_timeout_ms = value > 0 ? value : MSG_INF_POLLING_INTERVAL;
    iotcs_port_message_queue_put(g_send_queue, &UPDATE_POLLING_INTERVAL_MESSAGE, -1);
}

cl_message_dispatcher_counters* cl_message_dispatcher_counters_get(void) {
    g_message_dispatcher_counters.load = ((float) iotcs_port_message_queue_count(g_send_queue)) / ((float) IOTCS_PORT_MESSAGE_QUEUE_SIZE);
    return &g_message_dispatcher_counters;
}

void cl_message_dispatcher_counters_reset() {
    g_message_dispatcher_counters.total_bytes_received = 0;
    g_message_dispatcher_counters.total_bytes_sent = 0;
    g_message_dispatcher_counters.total_messages_received = 0;
    g_message_dispatcher_counters.total_messages_retried = 0;
    g_message_dispatcher_counters.total_messages_sent = 0;
    g_message_dispatcher_counters.total_protocols_errors = 0;
}

static void send_all(iotcs_message *message, int count) {
    int posted = 0;
    cl_fail_string_desc desc = {g_async_send_fail_reason_buf, IOTCS_ASYNC_FAIL_REASON_BUFFER_SIZE};

    if (!message || count <= 0) {
        return;
    }

    while (count > 0) {
        iotcs_result current_result = cl_internal_send(message, count, &posted, &desc);
        if (current_result == IOTCS_RESULT_OK) {
            g_message_dispatcher_counters.total_messages_sent += posted;
        } else {
            g_message_dispatcher_counters.total_protocols_errors += posted;
        }

        int i;
        for (i = 0; i < posted; i++) {
            if (current_result == IOTCS_RESULT_OK) {
                if (g_delivery_cb) {
                    (*g_delivery_cb)(&message[i]);
                }
            } else {
                if (g_error_cb) {
                    (*g_error_cb)(&message[i], current_result, g_async_send_fail_reason_buf);
                }
            }
        }

        message += posted;
        count -= posted;
    };
}

void* messaging_send_thread(void *arg) {
    (void) arg;
    int stop_message_received = 0;
    iotcs_message message[IOTCS_MAX_MESSAGES_FOR_SEND];
    iotcs_result result;

    set_settle_time();

    while (!stop_message_received) {
        int count = 0;
        result = iotcs_port_message_queue_get(g_send_queue, &message[count],
                g_polling_timeout_ms < INT_MAX ? (int) g_polling_timeout_ms : INT_MAX);

        if (result == IOTCS_RESULT_OK) {
            do {
                if (message[count].base == (iotcs_message_base*) STOP_MESSAGE_BASE) {
                    stop_message_received = 1;
                    break;
                }

                /*if update polling interval message is not first in a chain then just send all previous messages*/
                if (message[count].base == (iotcs_message_base*) UPDATE_POLLING_INTERVAL_MESSAGE_BASE) {
                    break;
                }

                count++;
            } while (count < IOTCS_MAX_MESSAGES_FOR_SEND &&
                    IOTCS_RESULT_OK == (result = iotcs_port_message_queue_get(g_send_queue, &message[count], 0)));

            send_all(&message[0], count);
        } else {
#ifdef IOTCS_LONG_POLLING
            if (!g_use_long_polling)
#endif
            {/* No messages to send - go to server and ask it one time */
                cl_get_requests_messages();
            }
        }
    }

#ifdef IOTCS_LONG_POLLING    
    UTIL_LOCK_LP();
    cl_long_polling_receive_should_be_stopped();
    UTIL_UNLOCK_LP();
#endif
    cl_put_special_request_message(MSG_STOP_RECEIVE_REQUEST, -1);

    if (g_recv_thread != NULL) {
        iotcs_port_thread_join(g_recv_thread);
    }

    // Then finalize send thread
    while (IOTCS_RESULT_OK == iotcs_port_message_queue_get(g_send_queue, &message[0], 0)) {
        send_all(&message[0], 1);
    }

#ifndef IOTCS_LONG_POLLING
    /* Free all received messages */
    {
        iotcs_request_message *request;
        while (NULL != (request = cl_internal_receive(0))) {
            g_message_dispatcher_counters.total_messages_received++;
            cl_internal_request_message_free(request);
        }
    }
#endif

    iotcs_port_thread_cleanup();

    return NULL;
}

void* messaging_recv_thread(void *arg) {
    (void) arg;
    iotcs_request_message *request;
#if (IOTCS_REQUEST_HANDLER_THREAD_POOL_SIZE <= 1)
    iotcs_message message;
#endif

    while (1) {
#ifndef IOTCS_LONG_POLLING
        request = cl_get_request_message_blocking(-1);

        if (request == MSG_STOP_RECEIVE_REQUEST) {
            break;
        }
#else
        if (cl_is_long_polling_receive_should_be_stopped()) {
            break;
        }

        if (g_use_long_polling) {
            request = cl_get_request_message_blocking(0);
        } else {
            request = cl_get_request_message_blocking(-1);
        }

        if (request == MSG_STOP_RECEIVE_REQUEST) {
            break;
        }
#endif

        if (request != NULL) {
            /* Process all received messages */
            g_message_dispatcher_counters.total_messages_received++;

            if (IOTCS_RESULT_OK == device_model_process_capability_request(request)) {
                continue; /* it was a capability request handled by library */
            }
#ifdef IOTCS_VIRTUALIZATION_SUPPORT            
  #if  defined(IOTCS_MESSAGE_DISPATCHER) && (IOTCS_REQUEST_HANDLER_THREAD_POOL_SIZE > 1)
           /* dispatch the request using one of request handler thread pool threads */
           cl_internal_request_handler_pool_dispatcher_dispatch(cl_internal_dispatch_request,request);
  #else
           cl_internal_dispatch_request(request, &message);
  #endif   
#else 
  #if  defined(IOTCS_MESSAGE_DISPATCHER) && (IOTCS_REQUEST_HANDLER_THREAD_POOL_SIZE > 1)
           cl_internal_request_handler_pool_dispatcher_dispatch(cl_internal_request_dispatcher_dispatch,request);
  #else
           cl_internal_request_dispatcher_dispatch(request, &message);
  #endif
#endif

#if (IOTCS_REQUEST_HANDLER_THREAD_POOL_SIZE <= 1)
           cl_internal_async_message_dispatcher_process_response_message(request, &message);
#endif
        }
#ifdef IOTCS_LONG_POLLING 
        else if (g_use_long_polling) {
            char* response = NULL;
            if (IOTCS_RESULT_OK == cl_post_long_polling_request(&response, NULL, IOTCS_LP_BIG_TIMEOUT_MS)) {
                if (response) {
                    cl_process_message_from_json(response);
                }
            }
        }
#endif
    }

    iotcs_port_thread_cleanup();

    return NULL;
}

void cl_internal_async_message_dispatcher_process_response_message(iotcs_request_message *request, iotcs_message* message) {
    //if settle time isn't passed and request wasn't processed then try to put it into receive queue
    //we will try to hold 1 slot for it until is_settle_time_passed() == IOTCS_FALSE 
    if (message->u.response.status_code == PROTOCOL_RESPONSE_CODE_NOT_FOUND &&
            cl_is_settle_time_passed() == IOTCS_FALSE &&
            strstr(message->u.response.request->url, "manage/resources/") == NULL) {
        cl_put_special_request_message(request, 0);
    } 
    else  {
        iotcs_port_message_queue_put(g_send_queue, message, -1);
    }
}

iotcs_result cl_internal_async_message_dispatcher_init(iotcs_message_dispatcher_delivery_callback delivery_cb,
        iotcs_message_dispatcher_error_callback error_cb, int polling_timeout_ms, iotcs_bool long_polling_request_required) {

    g_polling_timeout_ms = polling_timeout_ms;
    g_delivery_cb = delivery_cb;
    g_error_cb = error_cb;
#ifdef IOTCS_LONG_POLLING
    g_use_long_polling = long_polling_request_required;
#endif

    if (NULL == (g_send_queue = iotcs_port_message_queue_create())) {
        return IOTCS_RESULT_FAIL;
    }

    if (NULL == (g_send_thread = iotcs_port_thread_create(messaging_send_thread))) {
        iotcs_port_message_queue_destroy(g_send_queue);
        return IOTCS_RESULT_FAIL;
    }

    if (NULL == (g_recv_thread = iotcs_port_thread_create(messaging_recv_thread))) {
        iotcs_port_message_queue_put(g_send_queue, &STOP_MESSAGE, -1);
        iotcs_port_thread_join(g_send_thread);
        iotcs_port_message_queue_destroy(g_send_queue);
        return IOTCS_RESULT_FAIL;
    }

    return IOTCS_RESULT_OK;
}

void cl_internal_async_message_dispatcher_finalize(void) {
    iotcs_port_message_queue_put(g_send_queue, &STOP_MESSAGE, -1);
    iotcs_port_thread_join(g_send_thread);
    iotcs_port_message_queue_destroy(g_send_queue);
}
