/*
 * An example of a directly connected device which is capable of communicating
 * directly with Oracle IoT Cloud Service. This sample illustrates
 * C code for sending data to the cloud service and does not fully explore the Client Library API.
 *
 * The sample uses the virtualization API. It presents a simple messenger.
 *
 * The device implements a device model. A device model is a set of related
 * attributes, actions, and message formats that can be represented in
 * a real device. For this example the "Hello World" device model is used.
 * You must upload this device model to the server before running this example.
 */

#include "mbed.h"
#include "EthernetInterface.h"
#include "NTPClient.h"
#include "SDFileSystem.h"
/* include iot cs device model APIs */
#include "iotcs_virtual_device.h"
/* include methods for device client*/
#include "iotcs_device.h"

/* print error message and terminate the program execution */
#define error(message) {                                            \
                           printf("Error occurred: %s\n", message); \
                           led_red = 0;                             \
                           return IOTCS_RESULT_FAIL;                \
                       }

Serial pc(USBTX, USBRX);
EthernetInterface eth;
NTPClient ntp;

DigitalOut led_red(LED_RED);
DigitalOut led_green(LED_GREEN);
DigitalOut led_blue(LED_BLUE);

/* Current device value */
static const char* device_current_value;
/* Device model handle */
static iotcs_device_model_handle device_model_handle = NULL;
/* Device handle */
static iotcs_virtual_device_handle device_handle = NULL;

SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd"); // MOSI, MISO, SCK, CS
const char *ts_path = "/sd/trusted_assets_store";
const char *ts_password = "changeit";

int sample_main(void);

int main() {
    // set serial baudrate
    pc.baud(115200);
    // switch on blue led only
    led_red = 1;
    led_green = 1;
    led_blue = 0;

    // configure ethernet interface
    if (eth.init() == 0 && eth.connect(20000) == 0) { //Use DHCP
        pc.printf("Network initialized\n");
        pc.printf("  IP Address:      %s\n", eth.getIPAddress());
        pc.printf("  Subnet mask:     %s\n", eth.getNetworkMask());
        pc.printf("  Gateway:         %s\n", eth.getGateway());
        pc.printf("  MAC-address:     %s\n", eth.getMACAddress());
        led_blue = 1;
        if (sample_main() == IOTCS_RESULT_OK) {
            led_green = 0;
        } else {
            led_red = 0;
        }
    } else {
        printf("Can not obtain DHCP settings");
        led_red = 0;
    }
    eth.disconnect();
}

int sample_main(void) {
    /* This is the URN of your device model. */
    const char* device_urns[] = {
        "urn:test:helloworld",
        NULL
    };

    iotcs_result rv;
  
    /*
     * Initialize the library before any other calls.
     * Initiate all subsystems like ssl, TAM, request dispatcher,
     * async message dispatcher, etc., that are needed for correct library work.
     */
  
    if (iotcs_init(ts_path, ts_password) != IOTCS_RESULT_OK) {
        error("Initialization failed");
        return IOTCS_RESULT_FAIL;
    }
 
    /*
     * Activate the device, if it's not already activated.
     * Always check if the device is activated before calling activate.
     * The device model URN is passed into the activate call to tell
     * the server the device model(s) that are supported by this
     * directly connected device.
     */
 
    if (!iotcs_is_activated()) {
        if (iotcs_activate(device_urns) != IOTCS_RESULT_OK) {
            error("Sending activation request failed");
            return IOTCS_RESULT_FAIL;
        }
    }
 
    /* get device model handle */
    if (iotcs_get_device_model(device_urns[0], &device_model_handle) != IOTCS_RESULT_OK) {
        error("iotcs_get_device_model method failed\n");
        return IOTCS_RESULT_FAIL;
    }
 
    /* get device handle */
    if (iotcs_create_virtual_device_handle(iotcs_get_endpoint_id(), device_model_handle, &device_handle) != IOTCS_RESULT_OK) {
        error("iotcs_get_device_handle method failed\n");
        return IOTCS_RESULT_FAIL;
    }
 
    /* set device value */
    rv = iotcs_virtual_device_set_string(device_handle, "message", "Hello world!");
 
    if (rv != IOTCS_RESULT_OK) {
        error("iotcs_virtual_device_set_string method failed\n");
        return IOTCS_RESULT_FAIL;
    }
 
    /* get current device value */
    if (iotcs_virtual_device_get_string(device_handle, "message", &device_current_value) != IOTCS_RESULT_OK) {
        error("iotcs_virtual_device_get_string method failed\n");
        return IOTCS_RESULT_FAIL;
    }
 
    pc.printf("Message is %s\n", device_current_value);
 
    /* free device handle */
    iotcs_free_virtual_device_handle(device_handle);
    /* free device model handle */
    iotcs_free_device_model(device_model_handle);
 
    /*
     * Calling finalization of the library ensures that communications channels are closed,
     * previously allocated temporary resources are released.
     */
    iotcs_finalize();
    pc.printf("OK\n");
    
    return IOTCS_RESULT_OK;
}
