/*
 * Copyright (c) 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package oracle.iot.client.enterprise;

/**
 * An Action to be sent to the server. The action is sent by calling the
 * {@link #call()} method.
 * <p>
 * The {@code set} method returns the {@code Action} instance to allow
 * the fields of the action to be set in fluent style.
 * @see VirtualDevice#createAction(String)
 */
public abstract class Action {

    /**
     * Set the value of an argument in the {@code Action}. The arguments are
     * determined by the action in the device model. The arguments can be set
     * in any order.
     * The value is validated according to the constraints in the device model.
     * If the value is not valid, or the named arguement does not exist in the
     * device model, an IllegalArgumentException is raised.
     * <p>If the action has an {@code arguments} array in the device model, the
     * {@code argName} must match one of the argument names in the array.
     * To support device models which have only a single, un-named argument,
     * {@code argName} may be passed as {@code ""} (an empty String) or {@code null}.
     * For an action that takes no arguments, the {@code set}, the code only needs to
     * invoke the {@link #call} method. An attempt to set an argument value for an
     * action that does not take arguments will result in an IllegalArgumentException.
     * <p>All arguments defined in the action that do not have default values
     * must be set before calling {@code call()}.</p>
     * @param <T> the type of the value.
     * @param argName the name of an argument from the action
     * @param value the value to set
     * @return this Action instance
     * @throws IllegalArgumentException if the value is not valid,
     * or {@code field} is {@code null}
     * @see VirtualDevice#createAction(String)
     */
    public abstract <T> Action set(String argName, T value);

    /**
     * Send the action.
     * <p>
     * The onError handler of the parent virtual device will be called
     * if there is error sending the action.
     * <p>All fields defined in the action that do not have default values
     * must be set before invoking {@code call()}; otherwise, an (unchecked)
     * IllegalStateException is thrown.</p>
     * @throws IllegalStateException if {@code call()} is invoked
     * before setting all arguments that do not have default values
     */
    public abstract void call();

    /*  */
    protected Action() {}

}
