/*
 * Copyright (c) 2016, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl.http;

import android.webkit.CookieManager;
import com.oracle.iot.client.RestApi;
import oracle.iot.client.enterprise.UserAuthenticationException;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.GeneralSecurityException;

/**
 */
public final class HttpClientImpl extends HttpClient {

	private static final String CSRF_TOKEN_REQUEST_PATH = "/iot/webapi/v2/private/server";
	private static final String CSRF_TOKEN_NAME = "X-CSRF-TOKEN";

		@Override
		protected void beforeConnect(HttpsURLConnection connection)
				throws IOException, GeneralSecurityException {

			if (!RestApi.V2.isWebApi()) return;

			final String csrfToken = getCsrfToken(connection);

			if(csrfToken != null)
				connection.setRequestProperty(CSRF_TOKEN_NAME, csrfToken);

			// Add Cookies
			final String cookie = CookieManager.getInstance().getCookie(connection.getURL().toString());

			if (cookie.contains(CSRF_TOKEN_NAME))
				connection.setRequestProperty("Cookie", cookie);
			else if (csrfToken != null)
				connection.setRequestProperty("Cookie", cookie + "; " + CSRF_TOKEN_NAME + "=" + csrfToken);
		}

		private String getCsrfToken(HttpsURLConnection connection)
				throws IOException, GeneralSecurityException {
			URL url = connection.getURL();
			URL csrfTokenUrl = new URL("https", url.getHost(), url.getPort(), CSRF_TOKEN_REQUEST_PATH);
			final HttpsURLConnection con = (HttpsURLConnection) csrfTokenUrl.openConnection();

			con.setSSLSocketFactory(connection.getSSLSocketFactory());
			con.setInstanceFollowRedirects(false);
			con.setConnectTimeout(getResponseTimeout());
			con.setReadTimeout(getResponseTimeout());
			con.setRequestMethod("GET");

			String cookie = CookieManager.getInstance().getCookie(con.getURL().toString());
			con.setRequestProperty("Cookie", cookie);

			con.connect();

			final int responseCode = con.getResponseCode();
			String csrfToken = null;
			if (responseCode == 200) {
				csrfToken = con.getHeaderField(CSRF_TOKEN_NAME);
			} else if (responseCode == 302) {
				throw new UserAuthenticationException("User Authentication failed!", con.getHeaderField("Location"));
			}

			con.disconnect();
			return csrfToken;
		}

	public HttpClientImpl(SSLSocketFactory sslSocketFactory, URL url) throws MalformedURLException {
		super(sslSocketFactory, url);
	}

}
