/*
 * Copyright (c) 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl.enterprise;

import com.oracle.iot.client.DeviceModelAction;
import com.oracle.iot.client.DeviceModelFormat;
import com.oracle.iot.client.impl.DeviceModelImpl;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import oracle.iot.client.DeviceModel;
import oracle.iot.client.enterprise.Action;
import oracle.iot.client.enterprise.VirtualDevice;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * ActionImpl
 */
public class ActionImpl extends Action {

    private final VirtualDeviceImpl virtualDevice;
    private final DeviceModelAction deviceModelAction;
    private final Map<String, Object> argumentValues;

    ActionImpl(VirtualDeviceImpl virtualDevice, String actionName) {

        this.deviceModelAction = getDeviceModelAction(virtualDevice, actionName);
        
        if (this.deviceModelAction == null) {
            throw new IllegalArgumentException("'" + actionName + "' not found");
        }

        this.virtualDevice = (VirtualDeviceImpl) virtualDevice;
        this.argumentValues = new HashMap<String, Object>();
    }

    @Override
    public <T> Action set(String argumentName, T value) {

        DeviceModelAction.Argument argument = getArgument(argumentName);
        if (argument == null) {
            throw new IllegalArgumentException(argumentName + " not in model");
        }

        Object castValue = value;
        switch (argument.getArgType()) {
            case NUMBER:
                if (!(value instanceof Number)) {
                    throw new IllegalArgumentException("value for '" + argumentName + "' is not a NUMBER");
                }
                checkBounds(argument, (Number)value);
                break;
            case INTEGER:
                if (!(value instanceof Integer)) {
                    throw new IllegalArgumentException("value for '" + argumentName + "' is not an INTEGER");
                }
                checkBounds(argument, (Number)value);
                break;
            case DATETIME:
                if (!(value instanceof Date) && !(value instanceof Long)) {
                    throw new IllegalArgumentException("value for '" + argumentName + "' is not a DATETIME");
                }
                if (value instanceof Date) {
                    castValue = ((Date)value).getTime();
                }
                break;
            case BOOLEAN:
                if (!(value instanceof Boolean)) {
                    throw new IllegalArgumentException("value for '" + argumentName + "' is not a BOOLEAN");
                }
                break;
            case STRING:
                if (!(value instanceof String)) {
                    throw new IllegalArgumentException("value for '" + argumentName + "' is not a STRING");
                }
                break;
            case URI:
                if (!(value instanceof oracle.iot.client.ExternalObject)) {
                    throw new IllegalArgumentException("value for '" + argumentName + "' is not an ExternalObject");
                }
                break;
        }

        argumentValues.put(argumentName, castValue);

        return this;
    }

    private void checkBounds(DeviceModelAction.Argument argument, Number value)
        throws IllegalArgumentException {

        final Number upperBound = argument.getUpperBound();
        final Number lowerBound = argument.getLowerBound();

        // Assumption here is that lowerBound <= upperBound
        final double val = ((Number) value).doubleValue();
        if (upperBound != null) {
            final double upper = upperBound.doubleValue();
            if (Double.compare(val, upper) > 0) {
                throw new IllegalArgumentException(deviceModelAction.getName() +
                    " '" + argument.getName() + "' out of range: " +
                    val + " > " + upper);
            }
        }
        if (lowerBound != null) {
            final double lower = lowerBound.doubleValue();
            if(Double.compare(val, lower) < 0) {
                throw new IllegalArgumentException(deviceModelAction.getName() +
                    " '" + argument.getName() + "' out of range: " +
                    val + " < " + lower);
            }
        }

    }


    @Override
    public void call() {

        final String actionName = deviceModelAction.getName();
        // With Controller
        final String action_format = virtualDevice.isDeviceApp() ?
            VirtualDeviceImpl.DEVICE_APP_ACTION_FORMAT : VirtualDeviceImpl.DEVICE_ACTION_FORMAT;
        final String resource =
            String.format(action_format,
                virtualDevice.getEnterpriseClient().getApplication().getId(),
                virtualDevice.getEndpointId(),
                virtualDevice.getDeviceModel().getURN(),
                actionName);

        final String payload = getPayload();

        Controller.update(virtualDevice, "POST", resource, payload,
            new Controller.Callback() {
                public void onError(VirtualDevice virtualDevice,
                    String resource, String payload,
                    String reason) {
                    // on error, give user back the data they passed
                    ((VirtualDeviceImpl)virtualDevice).processOnError(virtualDevice, actionName, payload,
                        reason);
                }
            });    }

    private DeviceModelAction.Argument getArgument(String fieldName) {
        
        if (deviceModelAction == null) return null;
        
        List<DeviceModelAction.Argument> argumentList = deviceModelAction.getArguments();
        for(DeviceModelAction.Argument argument : argumentList) {
            if (argument.getName().equals(fieldName)) {
                return argument;
            }
        }
        return null;
    }
    
    private static DeviceModelAction getDeviceModelAction(
            VirtualDevice virtualDevice,
            String actionName) {

        DeviceModel dm = virtualDevice.getDeviceModel();
        if (!(dm instanceof DeviceModelImpl)) return null;
        
        DeviceModelImpl deviceModel = (DeviceModelImpl)dm;
        return deviceModel.getDeviceModelActions().get(actionName);
    }

    private String getPayload() {
        String payload = "{}";
        try {
            final JSONObject jsonObject = new JSONObject();
            List<DeviceModelAction.Argument> argumentList = deviceModelAction.getArguments();
            for(DeviceModelAction.Argument argument : argumentList) {
                Object value = argumentValues.get(argument.getName());
                if (value == null) {
                    value = argument.getDefaultValue();
                    if (value == null) {
                        throw new IllegalStateException(
                            "missing required argument '" + argument.getName()
                                + "' to action '" + deviceModelAction.getName() + "'");
                    }
                }
                jsonObject.put(argument.getName(), value);
            }
            payload = jsonObject.toString();

        } catch (JSONException e) {
            getLogger().log(Level.WARNING, e.toString());
        }
        return payload;
    }

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }   
}
