/*
 * Copyright (c) 2016, 2019, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl.device.http;

import com.oracle.iot.client.TransportException;
import com.oracle.iot.client.RestApi;
import com.oracle.iot.client.impl.http.HttpSecureConnection;
import com.oracle.iot.client.device.DirectlyConnectedDevice;
import com.oracle.iot.client.HttpResponse;
import com.oracle.iot.client.impl.device.SendReceiveImpl;
import com.oracle.iot.client.message.Message;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.security.GeneralSecurityException;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * HttpSendReceiveImpl is an implementation of the send(Message...) method of
 * {@link DirectlyConnectedDevice}. The send is
 * synchronous. Receive should be considered as a synchronous call. The
 * implementation has a buffer for receiving request messages. If there are
 * no messages in the buffer, the receive implementation will send a message
 * to the server to receive any pending requests the server may have.
 * <p>
 * The size of the buffer can be configured by setting the property {@code TBD}.
 */
public final class HttpSendReceiveImpl extends SendReceiveImpl {

    private static final String LONG_POLL_OFFSET_PROP =
        "com.oracle.iot.client.device.long_polling_timeout_offset";
    private static final String MIN_ACCEPT_BYTES_HEADER = "X-min-acceptBytes";
    private static final int USE_DEFAULT_TIMEOUT_VALUE = -1;
    private static final int LONG_POLL_OFFSET =
        Integer.getInteger(LONG_POLL_OFFSET_PROP, 100);
    private final HttpSecureConnection secureConnection;

    public HttpSendReceiveImpl(HttpSecureConnection secureConnection) {
        this(secureConnection, !SendReceiveImpl.isLongPollingDisabled());
    }

    HttpSendReceiveImpl(HttpSecureConnection secureConnection, boolean useLongPolling) {
        super(useLongPolling);
        this.secureConnection = secureConnection;
    }

    /*************************************************************************
     *
     * Methods and classes for handling outgoing message dispatch
     * Implementation and API in this section should be private
     *
     *************************************************************************/

    /*
     * Called from post(Collection<MessageQueueEntry>). This simply makes
     * the other code easier to read.
     */
    @Override
    protected void post(byte[] payload) throws IOException,
        GeneralSecurityException {
        post(payload, USE_DEFAULT_TIMEOUT_VALUE);
    }

    @Override
    protected void post(byte[] payload, int timeout) throws IOException,
        GeneralSecurityException {

        StringBuffer restApi = new StringBuffer(RestApi.V2.getReqRoot())
            .append("/messages");

        int usedBytes = 0;
        int acceptBytes = 0;

        // If we're not using long-polling, then each request needs to
        // have the acceptBytes parameter.
        // If payload is null, then we're polling the server
        // for request messages and we need the acceptBytes parameter
        // plus the iot.sync paramater. This is not dependent on long-polling!
        // A null payload means "poll".
        if (!useLongPolling || payload == null) {

            usedBytes = getUsedBytes();

            // subtract 2 for the length bytes.
            acceptBytes = Math.max(0, getRequestBufferSize() - usedBytes - 2);

            // There is no point in making the HTTP request if we're
            // polling for request messages and there isn't any buffer,
            if (payload == null && acceptBytes == 0) {
                return;
            }
        }

        restApi.append("?acceptBytes=").append(acceptBytes);

        boolean longPolling = useLongPolling && payload == null;
        if (longPolling) {
            restApi.append("&iot.sync");

            if (timeout > 0) {
                // iot.timeout is in seconds
                int iotTimeout = timeout / 1000;
                if (iotTimeout == 0) {
                    iotTimeout = 1;
                }

                restApi.append("&iot.timeout=");
                restApi.append(iotTimeout);

                /*
                 * Add an offset to the transport level timeout
                 * as a failover mechanism, just in case.
                 */
                timeout = (iotTimeout * 1000) + LONG_POLL_OFFSET;
            }
        }

        final String uri = restApi.toString();

        if (getLogger().isLoggable(Level.FINER)) {

            StringBuilder msg = new StringBuilder("POST ")
                .append(uri);
            if (getLogger().isLoggable(Level.FINEST)) {
                msg.append(" : ")
                    .append(Message.prettyPrintJson(payload));
            }
            getLogger().log(Level.FINER, msg.toString());
        }

        HttpResponse response;
        try {
            response =
                secureConnection.post(uri, payload, timeout);
        } catch (IOException ie) {

            if (getLogger().isLoggable(Level.FINER)) {
                getLogger().log(Level.FINER,"POST " + uri + ": " + ie.getMessage());
            }
            // Do not throw the SocketTimeoutException if the post was for long polling.
            // The SocketTimeoutException is expected for long polling and just
            // means that the server closed its end.
            if (longPolling && ie instanceof java.net.SocketTimeoutException) {
                return;
            }

            throw ie;
        }

        final int status = response.getStatus();

        if (longPolling && status == HttpURLConnection.HTTP_GATEWAY_TIMEOUT) {
            if (getLogger().isLoggable(Level.FINER)) {
                // 504 logs too much useless data
                getLogger().log(Level.FINER,"POST " + uri + ": HTTP 504");
            }
            // 504 is expected when long polling.
            return;
        }

        if (getLogger().isLoggable(Level.FINER)) {
            getLogger().log(Level.FINER, response.getVerboseStatus("POST", uri));
        }

        final Map<String, List<String>> map = response.getHeaders();
        final List<String> header = map.get(MIN_ACCEPT_BYTES_HEADER);
        if (header != null && !header.isEmpty()) {
            final int minAcceptBytes = Integer.parseInt(header.get(0));

            //
            // The next request the server has for this client is larger
            // than the bytes available in the request buffer.
            //
            // A properly configured and responsive system should never
            // get in this state.
            //

            final int requestBufferSize = getRequestBufferSize();
            if (minAcceptBytes > requestBufferSize - 2) {
                //
                // The request the server has is larger than buffer.
                //
                getLogger().log(Level.SEVERE,
                    "The server has a request of " + minAcceptBytes +
                        " bytes for this client, which is too large for the " +
                        requestBufferSize + " byte request buffer. Please " +
                        "restart the client with larger value for " +
                        REQUEST_BUFFER_SIZE_PROPERTY);
            } else {
                //
                // The message(s) from the last time have not been
                // processed.
                //
                getLogger().log(Level.WARNING,
                    "The server has a request of " + minAcceptBytes +
                        " bytes for this client, which cannot be sent " +
                        " because the " + requestBufferSize +
                        " byte request buffer is filled with " + usedBytes +
                        " of unprocessed requests");
            }
        }

        if (status != HttpURLConnection.HTTP_OK && status != HttpURLConnection.HTTP_ACCEPTED) {
            throw new TransportException(status,
                response.getVerboseStatus("POST", restApi.toString()));
        }

        if (acceptBytes > 0) bufferRequest(response.getData());
    }

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }   
}
