/*
 * Copyright (c) 2015, 2019, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl.device;

import com.oracle.iot.client.device.DirectlyConnectedDevice;
import com.oracle.iot.client.message.Message;
import com.oracle.iot.client.message.MessageParsingException;
import com.oracle.iot.client.message.RequestMessage;

import com.oracle.iot.client.message.ResponseMessage;
import java.util.Locale;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.security.GeneralSecurityException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * SendReceiveImpl is an implementation of the send(Message...) method of
 * {@link DirectlyConnectedDevice}. The send is
 * synchronous. Receive should be considered as a synchronous call. The
 * implementation has a buffer for receiving request messages. If there are
 * no messages in the buffer, the receive implementation will send a message
 * to the server to receive any pending requests the server may have.
 * <p>
 * The size of the buffer can be configured by setting the property {@code TBD}.
 */
public abstract class SendReceiveImpl {

    // Requests messages from the server are buffered. This is the
    // amount of space reserved for the buffer. Overridden by setting
    // oracle.iot.client.device.request_buffer_size
    private static final short DEFAULT_REQUEST_BUFFER_SIZE = 4192;

    // This is the minimum number of milliseconds since the last time
    // a message was sent before polling for request messages again. Used
    // when long polling is disabled. Overridden by setting
    // oracle.iot.client.device.send_receive_timeout
    private static final short DEFAULT_SEND_RECEIVE_TIMEOUT = 100; // in milliseconds

    // Set this property to override the default request buffer size.
    protected static final String REQUEST_BUFFER_SIZE_PROPERTY = "oracle.iot.client.device.request_buffer_size";

    // Set this property to override the default send-receive timeout.
    private static final String SEND_RECEIVE_TIMEOUT_PROPERTY = "oracle.iot.client.device.send_receive_timeout";

    // Set this property to true to disable long polling.
    private static final String DISABLE_LONG_POLLING_PROPERTY = "com.oracle.iot.client.disable_long_polling";

    // Constant to indicate that the default timeout value is to be used in an HTTP request.
    private static final int USE_DEFAULT_TIMEOUT_VALUE = -1;

    protected static boolean isLongPollingDisabled() {
        final String disableLongPollingPropertyValue = System.getProperty(DISABLE_LONG_POLLING_PROPERTY);
        return "".equals(disableLongPollingPropertyValue) || Boolean.parseBoolean(disableLongPollingPropertyValue);
    }

    private static short getShortPropertyValue(String key, short defaultValue) {
        short imax = Utils.getShortProperty(key, defaultValue);
        short max = (imax > defaultValue ? imax : defaultValue);
        return (max > 0 ? max : defaultValue);
    }

    private static short getSendReceiveTimeout(String key, short defaultValue) {
        short imax = Utils.getShortProperty(key, defaultValue);
        if (imax == 0)
            // allow to set 0 send_receive_timeout which would mean don't 'post(null)'
            return 0;
        short max = (imax > defaultValue ? imax : defaultValue);
        // set to property value if it's greater then default
        return (max > 0 ? max : defaultValue);
    }

    //
    // Messages from server are buffered. The buffer is a circular buffer of
    // bytes. The first two bytes at the head of the buffer always contains
    // the number of bytes in the message that follows. The tail points to the
    // next available chunk of buffer.  When bytes are written to the buffer,
    // the number of bytes are written to requestBuffer[tail] + requestBuffer[tail+1], 
    // and the received bytes are written starting from requestBuffer[tail+2].
    // If tail is greater than the length of the buffer size, tail wraps back
    // around to zero. In this way, the buffer wraps from tail
    // to head. Head keeps moving as messages are read. Tail keeps moving as
    // requests are received. Bytes in the buffer never have to be moved.
    //
    // The number of bytes available are sent to the server in a query parameter
    // to the messages REST API:
    // POST /iot/api/v2/messages?acceptBytes=<number-of-bytes>
    //
    private final byte[] requestBuffer;
    private int head;
    private int tail;

    protected final boolean useLongPolling;

    // Minumum time between calls to receive before going to check if
    // there are messages on the server. This prevents the POST of an
    // empty message just to see if the server has request
    // messages from happening in a tight loop. Used when
    // long polling is disabled.
    private final short  sendReceiveTimeLimit;

    // Last time a message was posted. Used in conjunction with
    // sendReceiveTimeLimit to see if a POST of an empty message
    // should be made poll for request messages.
    private long sendCallTime ;
    

    /**
     */
    protected SendReceiveImpl(boolean useLongPolling) {

        final short requestBufferSize =
                getShortPropertyValue(REQUEST_BUFFER_SIZE_PROPERTY,
                                      DEFAULT_REQUEST_BUFFER_SIZE);

        this.sendReceiveTimeLimit =
            getSendReceiveTimeout(SEND_RECEIVE_TIMEOUT_PROPERTY,
                                  DEFAULT_SEND_RECEIVE_TIMEOUT);

        // TODO: configurable.
        //requestBuffer = new byte[getRequestBufferSize()];
        requestBuffer = new byte[requestBufferSize];
        head = tail = 0;
        this.sendCallTime = -1;
        this.useLongPolling = useLongPolling;
    }

    /*
     * This is where a call to DirectlyConnectedDevice#send(Message...)
     * ends up.
     */
    public final void send(Message... messages)
            throws IOException, GeneralSecurityException {

        byte[] payload = null;

        if (messages != null && messages.length > 0) {
            JSONArray jsonArray = new JSONArray();
            for (Message message : messages) {

                // Special handling for LL actionCondition
                //     If this is a request message that loops back to the sender,
                //         don't deliver it, but queue it up for the next call to receive.
                //     If this is a response message that loops back to the sender,
                //         then don't send it to the server.
                if (message instanceof RequestMessage) {
                    if (message.getDestination() != null && message.getDestination().equals(message.getSource())) {
                        final JSONObject jsonObject = message.toJson();
                        // buffer request expects an array
                        final String jsonString = "[" + jsonObject.toString() + "]";
                        final byte[] buf = jsonString.getBytes(Charset.forName("UTF-8"));
                        if (buf.length < (getRequestBufferSize() - getUsedBytes())) {
                            bufferRequest(buf);
                            continue;
                        } else {
                            throw new IOException("could not loopback request message");
                        }
                    }
                } else if (message instanceof ResponseMessage) {
                    if (message.getDestination() != null && message.getDestination().equals(message.getSource())) {
                        continue;
                    }
                }
                jsonArray.put(message.toJson());
            }

            payload = jsonArray.toString().getBytes(Charset.forName("UTF-8"));
        }

        if (payload != null && payload.length > 2) {
            post(payload);
        }

        if (!useLongPolling) {
            this.sendCallTime = System.currentTimeMillis();
        }
    }

    /*
     * this is where a call to DirectlyConnectedDevice#receive(long) ends up.
     *
     * Timeout (in milliseconds) is used when using HTTP long polling,
     * so it is only need for the HttpSendReceiveImpl,
     * which passes it to HttpSecureConnection, which passes it to HttpClient,
     * which passes it to HttpsUrlConnection.setReadTimeout, in which 
     * timeout of zero is interpreted as an infinite timeout.
     *
     * However, if a negative value is given, it will be converted to be the
     * default response timeout.
     */
    public final synchronized RequestMessage receive(long timeout)
            throws IOException, GeneralSecurityException {

        // If head == tail, requestBuffer is empty. 
        if (head == tail) {
            if (!useLongPolling) {
                long receiveCallTime = System.currentTimeMillis();
                if (this.sendReceiveTimeLimit == 0 ||
                        (receiveCallTime - this.sendCallTime) <
                        this.sendReceiveTimeLimit) {
                    // time delta between last send and this receive is too
                    // small do not make a call to the network to get request
                    // messages
                    return null;
                } else {
                    post(null, USE_DEFAULT_TIMEOUT_VALUE);
                    this.sendCallTime = System.currentTimeMillis();
                }
            } else {
                post(null, (int)timeout);
            }
        }

        if (head != tail) {

            int nBytes = 0;
            nBytes += ((requestBuffer[(head++) % requestBuffer.length] ) & 0xFF) << 8;
            nBytes += ((requestBuffer[(head++) % requestBuffer.length] ) & 0xFF);

            int offset = head;

            // keep head < requestBuffer.length to avoid overflow of head.
            // Also, this gets past the JSON object or array in the buffer.
            // If something goes bad when we're parsing out the JSON,
            // then we will have skipped over this bad JSON on the next call
            // to receive.
            head = (head + nBytes) % requestBuffer.length;

            try {
                // Read the comments in bufferRequest, then return here.
                //
                // The nBytes starting at buffer[head] are either a JSONArray
                // or a JSONObject. To extract the JSON, we deserialize
                // the buffered bytes using a JSONTokener. First we create
                // a String from the buffered bytes (Android only provides
                // a JSONTokener(String)). 
                final byte[] bytes = new byte[nBytes];
                for (int b = 0; b < nBytes; b++) {
                    bytes[b] = requestBuffer[offset++ % requestBuffer.length];
                }
                final String json = new String(bytes, "UTF-8");

                // This will be the JSONObject that we pull from the buffer. 
                // The JSONObject represents a RequestMessage. 
                JSONObject jsonObject = null;

                final JSONTokener jsonTokener = new JSONTokener(json);
                while (jsonTokener.more()) {
                    final Object obj = jsonTokener.nextValue();
                    if (obj instanceof JSONArray){
                        // The server sends us a JSONArray. Typically, the
                        // array will have only one element. But if the array
                        // contains more than one element, we want to return
                        // the first element, and retain the others in the buffer.
                        final JSONArray jsonArray = (JSONArray)obj;
                        for (int e=jsonArray.length()-1; 0 <= e; --e) {

                            // At the end of this loop, jsonObject will be the first
                            // request message in the request buffer
                            jsonObject = jsonArray.getJSONObject(e);

                            // If this is not the last JSONObject (the one we return),
                            // we re-write the JSONObject back to the buffer.
                            // The next time receive is called, we only deserialize
                            // the one object, not the whole array.
                            //
                            // If the buffer has '<nbytes>[{"a":1},{"b":2},{"c":3}]',
                            // what we end up with after re-writing the extra elements is
                            // '<nbytes>{"b":2}<nbytes>{"c":3}'
                            //
                            // Since we're re-writing a smaller amount of bytes, we
                            // don't have to worry about wrapping backwards over tail. 
                            //
                            if (e > 0) {
                                final String s = jsonObject.toString();
                                final byte[] data = s.getBytes("UTF-8");
                                for (int d = data.length-1; 0 <= d; --d) {
                                    // note that we're writing this backwards...
                                    head = --head >= 0 ? head : requestBuffer.length - 1;
                                    requestBuffer[head % requestBuffer.length] = data[d];
                                }

                                // now write the length into two bytes...
                                head = --head >= 0 ? head : requestBuffer.length - 1;
                                requestBuffer[head % requestBuffer.length] =
                                    (byte)((0x00ff & data.length));

                                head = --head >= 0 ? head : requestBuffer.length - 1;
                                requestBuffer[head % requestBuffer.length] =
                                    (byte)((0xff00 & data.length) >> 8);

                                // Now head is pointing to the first byte of the
                                // two byte length value.
                            }
                        }
                    } else if (obj instanceof JSONObject) {
                        // A JSONObject in the buffer is from re-writing the
                        // elements of the JSONArray
                        jsonObject = (JSONObject) obj;
                        break;
                    } else {
                        getLogger().log(Level.WARNING, "unexpected element ignored: " + String.valueOf(obj));
                    }
                }

                if (jsonObject != null) {

                    RequestMessage requestMessage =
                        new RequestMessage.Builder().fromJson(jsonObject).build();

                    if (getLogger().isLoggable(Level.FINER)) {
                        getLogger().log(Level.FINER, "dequeued '"
                            + requestMessage.getMethod().toUpperCase(Locale.ROOT) + " "
                            + requestMessage.getURL() + " "
                            + requestMessage.getBodyString() + "', destination = "
                            + requestMessage.getDestination());
                    }

                    return requestMessage;
                }

            } catch (UnsupportedEncodingException e) {
                // UTF-8 is a required encoding, so this cannot happen.
            } catch (JSONException e) {
                getLogger().log(Level.SEVERE, e.getMessage());
            } catch (MessageParsingException e) {
                // thrown from RequestMessage.Builder fromJson
                getLogger().log(Level.SEVERE, e.getMessage());
            } catch (IllegalArgumentException e) {
                // thrown from RequestMessage constructor
                getLogger().log(Level.SEVERE, e.getMessage());
            }

        }

        return null;
    }

    /*************************************************************************
     *
     * Methods and classes for handling outgoing message dispatch
     * Implementation and API in this section should be private
     *
     *************************************************************************/

    /*
     * Called from post(Collection<MessageQueueEntry>). This simply makes
     * the other code easier to read.
     */
    abstract protected void post(byte[] payload) throws IOException, GeneralSecurityException;

    /*
     * Messages are received when posting, an application when using HTTP long
     * polling can set the read timeout (in milliseconds) when receiving,
     * the so it is only need for the HttpSendReceiveImpl,
     * which passes it to HttpSecureConnection, which passes it to HttpClient,
     * which passes it to HttpsUrlConnection.setReadTimeout, in which 
     * timeout of zero is interpreted as an infinite timeout.
     *
     * However, if a negative value is given, it will be converted to be the
     * default response timeout.
     */
    abstract protected void post(byte[] payload, int timeout)
        throws IOException, GeneralSecurityException;

    // head and tail are kept so that they are less than requestBuffer.length.
    // This avoids the (unlikely) situation that they would overflow.
    // But this also means that tail might be less than head, since the buffer
    // is circular. This has to be taken into account when calculating the
    // available bytes. Remember that tail is the pointer to the next
    // available chunk and does not count toward used bytes (no need to
    // adjust used byte count by 1). If tail == head, then used bytes is
    // zero - perfect!
    final synchronized protected int getUsedBytes() {
        head = head % requestBuffer.length;
        tail = tail % requestBuffer.length;
        return tail >= head
                ? tail - head
                : (tail + requestBuffer.length) - head;
    }

    final protected int getRequestBufferSize() {
        return requestBuffer.length;
    }

    final synchronized protected void bufferRequest(final byte[] data) {

        // if data.length == 2, then it is an empty json array and there are
        // no values in the message.
        if (data != null && data.length > 2) {

            //
            // We have two choices here. One is to buffer the entire JSON array,
            // and the other is to split the JSON array up into its individual
            // JSON objects and buffer the individual objects. To buffer as
            // individual objects, we can create a JSONArray from the bytes, then
            // iterate over the data and write back the bytes of each element
            // to the requestBuffer. Then in the receive method, we'd just deserialize
            // the individual JSON objects. But this means that we're parsing the
            // JSON twice (once here, and again in the receive method). It also 
            // means that we have to create a String from the bytes twice
            // (since Android's JSONTokener only accepts a String).
            // If we just buffer the entire array, then we only have to create
            // a String and tokenize once in the receive method. But, if we store
            // the array, then end up deserializing the entire array only to return
            // the first element. To avoid deserializing the array over and over,
            // the receive method re-writes the individual JSONObjects back to
            // the buffer.
            //

            // Write the length of the data.
            requestBuffer[tail++ % requestBuffer.length] =
                (byte)((0xff00 & data.length) >> 8);
            requestBuffer[tail++ % requestBuffer.length] =
                (byte)((0x00ff & data.length));

            for(int pos = 0; pos < data.length; pos++) {
                requestBuffer[tail++ % requestBuffer.length] = data[pos];
            }

            // adjust tail to keep it < requestBuffer.length to avoid overflow.
            tail = tail % requestBuffer.length;

            if (getLogger().isLoggable(Level.FINER)) {
                // subtract 2 for the length bytes.
                final int availableBytes = getRequestBufferSize() - getUsedBytes() - 2;
                getLogger().log(Level.FINER,
                    "buffered " + data.length + " bytes of request data from server. " +
                        availableBytes + " available bytes of " +
                        getRequestBufferSize() + " remaining in buffer."
                );
            }

            if (getLogger().isLoggable(Level.FINEST)) {
                getLogger().log(Level.FINEST, "buffered: " + Message.prettyPrintJson(data));
            }

        }

    }

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }   
}
