/*
 * Copyright (c) 2017, 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl.device;

import com.oracle.iot.client.DeviceModelAction;
import com.oracle.iot.client.DeviceModelAction.Argument;
import com.oracle.iot.client.DeviceModelAttribute;
import com.oracle.iot.client.DeviceModelAttribute.Type;
import com.oracle.iot.client.device.DirectlyConnectedDevice;
import com.oracle.iot.client.device.util.RequestDispatcher;
import com.oracle.iot.client.impl.DeviceModelImpl;
import com.oracle.iot.client.message.Message;
import com.oracle.iot.client.message.RequestMessage;
import com.oracle.iot.client.message.ResponseMessage;
import com.oracle.iot.client.message.StatusCode;
import java.util.List;
import oracle.iot.client.DeviceModel;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Device
 */
public final class DeviceAnalogImpl implements DeviceAnalog {

    public DeviceAnalogImpl(DirectlyConnectedDevice directlyConnectedDevice, DeviceModelImpl deviceModel, String endpointId) {
        this.directlyConnectedDevice = directlyConnectedDevice;
        this.deviceModel = deviceModel;
        this.endpointId = endpointId;
        attributeValueMap = new HashMap<String,Object>();
    }

    @Override
    public String getEndpointId() {
        return this.endpointId;
    }

    @Override
    public DeviceModel getDeviceModel() {
        return deviceModel;
    }

    @Override
    public void setAttributeValue(String attribute, Object value) {

        if (value == null) {
            throw new IllegalArgumentException("value cannot be null");
        }

        final DeviceModelAttribute deviceModelAttribute = deviceModel.getDeviceModelAttributes().get(attribute);
        if (deviceModelAttribute == null) {
            throw new IllegalArgumentException(deviceModel.getURN() + " does not contain attribute " + attribute);
        }

        final DeviceModelAttribute.Type type = deviceModelAttribute.getType();
        final boolean badValue;
        switch (type) {
            case NUMBER:
                badValue = !(value instanceof Number);
                break;
            case STRING:
            case URI:
                badValue = !(value instanceof String);
                break;
            case BOOLEAN:
                badValue = !(value instanceof Boolean);
                break;
            case INTEGER:
                badValue = !(value instanceof Integer);
                break;
            case DATETIME:
                badValue = !(value instanceof Long);
                break;
            default:
                throw new InternalError("unknown type '" + type + "'");
        }

        if (badValue) {
            throw new IllegalArgumentException(
                    "cannot set '"+ deviceModel.getURN() + ":attribute/" + attribute + "' to " + value.toString()
            );
        }

        attributeValueMap.put(attribute, value);

    }

    @Override
    public Object getAttributeValue(String attribute) {

        final DeviceModelAttribute deviceModelAttribute = deviceModel.getDeviceModelAttributes().get(attribute);
        if (deviceModelAttribute == null) {
            throw new IllegalArgumentException(deviceModel.getURN() + " does not contain attribute " + attribute);
        }

        Object value = attributeValueMap.get(attribute);
        if (value == null) {
            value = deviceModelAttribute.getDefaultValue();
        }
        return value;
    }

    @Override
    public void call(String actionName, Map<String, ?> args) {

        final Map<String,DeviceModelAction> deviceModelActionMap = deviceModel.getDeviceModelActions();
        if (deviceModelActionMap == null) {
            getLogger().log(Level.WARNING, deviceModel.getURN() + " does not contain action '" + actionName + "'");
            return;
        }

        final DeviceModelAction deviceModelAction = deviceModelActionMap.get(actionName);
        if (deviceModelAction == null) {
            getLogger().log(Level.WARNING, deviceModel.getURN() + " does not contain action '" + actionName + "'");
            return;
        }

        // What this is doing is pushing a request message into the message
        // queue for the requested action. To the LL, it is handled just like
        // any other RequestMessage. But we don't want this RM going to the
        // server, so the source and destination are set to be the same
        // endpoint id. In SendReceiveImpl, if there is a RequestMessage
        // with a source == destination, it is treated it specially.
        final RequestMessage.Builder requestMessageBuilder =
                new RequestMessage.Builder()
                .source(getEndpointId())
                .destination(getEndpointId())
                .url("deviceModels/" + getDeviceModel().getURN() + "/actions/" + actionName)
                .method("POST");

        final List<Argument> argumentList = deviceModelAction.getArguments();
        try {
            final JSONObject actionArgs = new JSONObject();
            for (Argument argument : argumentList) {
                Object value = args.get(argument.getName());
                if (value == null) {
                    value = argument.getDefaultValue();
                    if (value == null) {
                        getLogger().log(Level.WARNING,
                            "Action not called: missing argument '" + argument.getName()
                                + " in call to '" + actionName + "'");
                        return;
                    }
                }
                encodeArg(actionArgs, deviceModelAction, argument, value);
            }
            requestMessageBuilder.body(actionArgs.toString());
        } catch (JSONException e) {
            getLogger().log(Level.WARNING, e.toString());
            return;
        } catch (IllegalArgumentException e) {
            getLogger().log(Level.WARNING, e.toString());
            return;
        }
        final RequestMessage requestMessage = requestMessageBuilder.build();

        final boolean useLongPolling = !Boolean.getBoolean("com.oracle.iot.client.disable_long_polling");

        //
        // Assumption here is that, if you are using long polling, you are using message dispatcher.
        // This could be a bad assumption. But if long polling is disabled, putting the message on
        // the request buffer will work regardless of whether message dispatcher is used.
        //
        if (useLongPolling) {
            try {
                final ResponseMessage responseMessage = RequestDispatcher.getInstance().dispatch(requestMessage);
                if (responseMessage.getStatusCode() == StatusCode.NOT_FOUND) {
                    getLogger().log(Level.INFO,
                            "Endpoint " + getEndpointId() + " has no handler for " + requestMessage.getURL());
                }
            } catch (Exception e) {
                getLogger().log(Level.WARNING, e.getMessage());
            }

        } else {
            // Not long polling, push request message back on request buffer
            try {
                directlyConnectedDevice.send(requestMessage);
            } catch (IOException e) {
                getLogger().log(Level.WARNING, e.getMessage());
            } catch (GeneralSecurityException e) {
                getLogger().log(Level.WARNING, e.getMessage());
            }
        }


    }

    private void encodeArg(JSONObject jsonObject, DeviceModelAction deviceModelAction, Argument argument, Object value)
        throws IllegalArgumentException, JSONException {
        final String actionName = deviceModelAction.getName();
        final String argumentName = argument.getName();
        switch (argument.getArgType()) {
            case NUMBER:
                if (!(value instanceof Number)) {
                    throw new IllegalArgumentException(
                        actionName + " value for '" + argument + "' is not a NUMBER");
                }
                checkBounds(deviceModelAction, argument, (Number) value);
                jsonObject.put(argumentName, (Number) value);
                break;
            case INTEGER:
                if (!(value instanceof Integer)) {
                    throw new IllegalArgumentException(
                        actionName + " value for '" + argumentName + "' is not an INTEGER");
                }
                checkBounds(deviceModelAction, argument, (Integer) value);
                jsonObject.put(argumentName, (Number) value);
                break;
            case DATETIME:
                if (!(value instanceof Date) && !(value instanceof Long)) {
                    throw new IllegalArgumentException(
                        actionName + " value for '" + argumentName + "' is not a DATETIME");
                }
                if (value instanceof Date) {
                    jsonObject.put(argumentName, ((Date) value).getTime());
                } else {
                    jsonObject.put(argumentName, (Long) value);
                }
                break;
            case BOOLEAN:
                if (!(value instanceof Boolean)) {
                    throw new IllegalArgumentException(
                        actionName + " value for '" + argumentName + "' is not a BOOLEAN");
                }
                jsonObject.put(argumentName, (Boolean) value);
                break;
            case STRING:
                if (!(value instanceof String)) {
                    throw new IllegalArgumentException(
                        actionName + " value for '" + argumentName + "' is not a STRING");
                }
                jsonObject.put(argumentName, (String) value);
                break;
            case URI:
                if (!(value instanceof oracle.iot.client.ExternalObject) &&
                    !(value instanceof com.oracle.iot.client.ExternalObject)) {
                    throw new IllegalArgumentException(
                        actionName + " value for '" + argumentName + "' is not an ExternalObject");
                }
                if (value instanceof oracle.iot.client.ExternalObject) {
                    jsonObject
                        .put(argumentName, ((oracle.iot.client.ExternalObject) value).getURI());
                } else {
                    jsonObject
                        .put(argumentName, ((com.oracle.iot.client.ExternalObject) value).getURI());
                }
                break;
            default:
                throw new IllegalArgumentException(
                    actionName + " argument '" + argumentName + "' has an unknown type");
        }
    }

    private void checkBounds(
        DeviceModelAction deviceModelAction,
        DeviceModelAction.Argument argument,
        Number value)
        throws IllegalArgumentException {

        final Number upperBound = argument.getUpperBound();
        final Number lowerBound = argument.getLowerBound();

        // Assumption here is that lowerBound <= upperBound
        final double val = ((Number) value).doubleValue();
        if (upperBound != null) {
            final double upper = upperBound.doubleValue();
            if (Double.compare(val, upper) > 0) {
                throw new IllegalArgumentException(deviceModelAction.getName() +
                    " '" + argument.getName() + "' out of range: " +
                    val + " > " + upper);
            }
        }
        if (lowerBound != null) {
            final double lower = lowerBound.doubleValue();
            if(Double.compare(val, lower) < 0) {
                throw new IllegalArgumentException(deviceModelAction.getName() +
                    " '" + argument.getName() + "' out of range: " +
                    val + " < " + lower);
            }
        }

    }

    // DeviceAnalog API
    @Override
    public void queueMessage(Message message) {
        try {
            directlyConnectedDevice.offer(message);
        } catch (IOException e) {
            getLogger().log(Level.INFO, e.getMessage());
        } catch (GeneralSecurityException e) {
            getLogger().log(Level.INFO, e.getMessage());
        }
    }

    private final DirectlyConnectedDevice directlyConnectedDevice;
    private final DeviceModelImpl deviceModel;
    private final String endpointId;
    private final Map<String,Object> attributeValueMap;

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }
}
