/*
 * Copyright (c) 2017, 2019, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */
package com.oracle.iot.client.impl;

import com.oracle.iot.client.TransportException;
import com.oracle.iot.client.HttpResponse;
import com.oracle.iot.client.RestApi;
import com.oracle.iot.client.SecureConnection;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.channels.FileChannel;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.Map;
import java.util.List;
import java.util.Set;

import org.json.JSONException;
import org.json.JSONObject;
import com.oracle.iot.client.StorageObject;

/**
 * The StorageConnection transfers content to the Storage Cloud Service.
 * Service. There is one StorageConnection instance per client.
 */
public abstract class StorageConnectionBase implements StorageConnection {
    
    private static final String STORAGE_CLOUD_HOST;
    private static final String DEFAULT_STORAGE_CLOUD_HOST = "storage.oraclecloud.com";
    private static final boolean virtualStorageDirectories;
    /**
     * REST resource for storage authentication
     */
    private static final String REST_STORAGE_AUTHENTICATION =
        RestApi.V2.getReqRoot()+"/provisioner/storage";
    private static final String AUTH_TOKEN_HEADER = "X-Auth-Token";
    private static final int DEFAULT_RETRY_LIMIT = 2;
    private static final int ERR_BAD_CHECKSUM = -1;
    private static final int ERR_UPLOAD_CANCELLED = -2;
    private static final int ERR_WRITING_REQUEST_BODY_TO_SERVER = -3;
    /**
     * Number of times to retry a transfer if the token expires.
     */
    private static final int RETRY_LIMIT;
    /**
     * Default chunk size is 4096.
     */
    private static final int DEFAULT_CHUNK_SIZE = 4096;
    /**
     * Property for setting chunk size. No limits imposed on chunk size,
     * but HttpURLConnection may impose its own limits.
     */
    private static final String CHUNK_SIZE_PROPERTY = "com.oracle.iot.client.storage_connection_chunk_size";
    /**
     * Size of chunks for chunked streaming mode.
     */
    private static final int CHUNK_SIZE = Integer.getInteger(CHUNK_SIZE_PROPERTY, DEFAULT_CHUNK_SIZE);

    private final SecureConnection secureConnection;
    private String storageContainerUrl = null;
    private String authToken = null;
    
    private boolean closed;

    static {
        Integer val = Integer.getInteger("com.oracle.iot.client.storage_connection_retry_limit");
        RETRY_LIMIT = ((val != null) && (val > 0))? val : DEFAULT_RETRY_LIMIT;

        STORAGE_CLOUD_HOST = System.getProperty("com.oracle.iot.client.storage_connection_host_name", DEFAULT_STORAGE_CLOUD_HOST);

        final String value = System.getProperty("oracle.iot.client.disable_storage_object_prefix");
        virtualStorageDirectories = value == null || "".equals(value) || !Boolean.parseBoolean(value);
    }

    
    protected StorageConnectionBase(SecureConnection secureConnection) {
        this.secureConnection = secureConnection;
        this.closed = false;
    }

    @Override
    final public void sync(StorageObject storageObject) throws IOException, GeneralSecurityException {
        if (storageObject.getInputStream() != null) {
            upload(storageObject);
        }
        else if (storageObject.getOutputStream() != null) {
            download(storageObject);
        }
        else throw new IllegalArgumentException("InputStream and OutputStream are not set.");
    }

    private void upload(StorageObject storageObject) throws IOException {
        int responseCode = transfer(storageObject);
        if (responseCode != HttpURLConnection.HTTP_CREATED) {
            throw new TransportException(responseCode, "Upload " + storageObject.getURI());
        }
    }

    private void download(StorageObject storageObject) throws IOException {
        int responseCode = transfer(storageObject);
        if (responseCode != HttpURLConnection.HTTP_OK && responseCode != HttpURLConnection.HTTP_PARTIAL) {
            throw new TransportException(responseCode, "Download " + storageObject.getURI());
        }
    }
    
    final public StorageObject createStorageObject(String clientId,
            String name, String contentType)
            throws IOException, GeneralSecurityException {
        final String path;
        if (virtualStorageDirectories && clientId != null) {
            path = clientId + "/" + name;
        } else {
            path = name;
        }

        final String contentStorageLocation =
            getStorageContainerUrl() + "/" + path;
        final StorageObject storageObject =
            createStorageObject(contentStorageLocation, name,
                                contentType, null, null, -1);

        return storageObject;
    }

    final public StorageObject createStorageObject(String storageUrl)
            throws IOException, GeneralSecurityException {

        final URL url;
        final String name;
        try {
            url = new URL(storageUrl);
            String fullContainerUrl = getStorageContainerUrl() + "/";
            if (!storageUrl.startsWith(fullContainerUrl)) {
                throw new GeneralSecurityException("Storage container URL does not match.");
            }

            if (virtualStorageDirectories) {
                name = storageUrl.substring(storageUrl.lastIndexOf('/') + 1);
            } else {
                name = storageUrl.substring(fullContainerUrl.length());
            }
        } catch (MalformedURLException ex) {
            throw new IllegalArgumentException("Storage Cloud URL is invalid", ex);
        }

        int i=0;
        while (i < RETRY_LIMIT) {
            if (authToken == null) {
                authToken = authenticate();
            }
            if (authToken != null) {
            final HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod("HEAD");
            connection.setRequestProperty(AUTH_TOKEN_HEADER, authToken);
            connection.connect();

            final int length = connection.getContentLength();
            final String type = connection.getContentType();
            final String encoding = connection.getContentEncoding();
            final String date = connection.getHeaderField("Last-Modified");
            final int responseCode = connection.getResponseCode();

                if (responseCode == HttpURLConnection.HTTP_OK) {
                    return createStorageObject(storageUrl, name, type, encoding, date, length);
                } else if (responseCode != HttpURLConnection.HTTP_UNAUTHORIZED) {
                throw new TransportException(responseCode, "HEAD " + storageUrl);
            } else {
                    authToken = null;
                }
            }
            i++;
        }
        throw new TransportException(HttpURLConnection.HTTP_UNAUTHORIZED, "HEAD " + storageUrl);
    }

    protected abstract StorageObject createStorageObject(
                                           String storageContainerUrl,
                                           String name,
                                           String type,
                                           String encoding,
                                           String date,
                                           int length);

    public static boolean isStorageCloudURI(String uri) {
        try {
            if (new URI(uri).getHost().contains(STORAGE_CLOUD_HOST)) {
                return true;
            }
        } catch (Exception ex) {
        }
        return false;
    }

    @Override
    final public synchronized void close() throws IOException {
        if (!closed) {
            closed = true;
        }
    }

    private String getStorageContainerUrl() throws IOException, GeneralSecurityException {
        if (storageContainerUrl == null) {
            StorageAuthenticationResponse storageAuthenticationResponse
                    = getStorageAuthentication();
            storageContainerUrl = storageAuthenticationResponse.getStorageContainerUrl();
            authToken = storageAuthenticationResponse.getAuthToken();
        }
        return storageContainerUrl;
    }

    private int transfer(StorageObject storageObject) throws IOException {

        final InputStream inputStream = storageObject.getInputStream();
        final boolean upload = inputStream != null;
        if (upload && inputStream.markSupported()) {
            inputStream.mark(Integer.MAX_VALUE);
        }

        final String scPath = storageObject.getURI() == null
            ? storageContainerUrl + "/" + storageObject.getName()
            : storageObject.getURI();

        int retries = 0;
        while (retries < RETRY_LIMIT) {

            if (retries > 0 && upload) {
                if (inputStream.markSupported()) {
                    inputStream.reset();
                } else if (inputStream instanceof FileInputStream) {
                    FileInputStream fs = (FileInputStream)inputStream;
                    FileChannel fc = fs.getChannel();
                    fc.position(0);
                } else {
                    // cannot reset stream for retry
                    getLogger().log(Level.SEVERE, "Cannot reset input stream for retry");
                    return HttpURLConnection.HTTP_INTERNAL_ERROR;
                }
            }

            if (authToken == null) {
                authToken = authenticate();
            }

            final int responseCode;
            if (authToken != null) {
                responseCode = transferContent(scPath, storageObject);
            } else {
                // could not get authToken
                return HttpURLConnection.HTTP_UNAUTHORIZED;
            }

            switch (responseCode) {
                case HttpURLConnection.HTTP_OK:
                case HttpURLConnection.HTTP_CREATED:
                case HttpURLConnection.HTTP_PARTIAL:
                    return responseCode;
                case ERR_UPLOAD_CANCELLED:
                    getLogger().log(Level.INFO, (upload ? "Upload" : "Download") + " cancelled " + storageObject.getURI());
                    return upload ? HttpURLConnection.HTTP_CREATED : HttpURLConnection.HTTP_OK;
                case HttpURLConnection.HTTP_UNAUTHORIZED:
                    // retry with new token
                    authToken = null;
                    break;
                case ERR_BAD_CHECKSUM:
                case ERR_WRITING_REQUEST_BODY_TO_SERVER:
                    // retry
                    break;
                default:
                    return responseCode;
            }
            retries++;
        }
        getLogger().log(Level.INFO, "Retries exceeded " + (upload ? "uploading" : "downloading") + " " + storageObject.getURI());
        return HttpURLConnection.HTTP_INTERNAL_ERROR;
    }

    private int transferContent(final String scPath,
            StorageObject storageObject) throws IOException {

        assert authToken != null;
        if (authToken == null) {
            return HttpURLConnection.HTTP_UNAUTHORIZED;
        }

        MessageDigest digest = null;
        try {
            digest = MessageDigest.getInstance("MD5");
        } catch (NoSuchAlgorithmException ex) {
            getLogger().log(Level.WARNING, "Storage Cloud Service: checksum could not be verified.", ex);
        }
        final String contentType = storageObject.getType();
        final String encoding = storageObject.getEncoding();
        final InputStream inputStream = storageObject.getInputStream();
        final OutputStream outputStream = storageObject.getOutputStream();
        final boolean upload = inputStream != null;
        final URL url = new URL(scPath);
        final HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        if (upload) {
            connection.setRequestMethod("PUT");
            connection.setChunkedStreamingMode(CHUNK_SIZE);
            connection.setRequestProperty("Content-Type", contentType != null ? contentType : "application/octet-stream");
            if (encoding != null) {
                connection.setRequestProperty("Content-Encoding", encoding);
            }
            connection.setDoOutput(true);
            Map<String, String> metadata = storageObject.getCustomMetadata();
            for (Map.Entry<String, String> entry : metadata.entrySet()) {
                String k = entry.getKey();
                String v = entry.getValue();
                connection.setRequestProperty("X-Object-Meta-" + k, v);
            }
        } else {
            connection.setRequestMethod("GET");
            connection.setDoOutput(false);
        }

        connection.setRequestProperty(AUTH_TOKEN_HEADER, authToken);
        
        connection.connect();
        
        int responseCode;
        long transferredBytes = 0;
        final OutputStream os = upload ? connection.getOutputStream() : outputStream;
        final InputStream is = upload ? inputStream : connection.getInputStream();

        byte[] b = new byte[CHUNK_SIZE];
        try {

            int len;
            while ((len = is.read(b)) != -1) {
                if (((StorageObjectDelegate) storageObject).isCancelled()) {
                    return ERR_UPLOAD_CANCELLED;
                }
                os.write(b, 0, len);
                if (digest != null) {
                    digest.update(b, 0, len);
                }
                ((StorageObjectDelegate) storageObject)
                    .setTransferredBytes(transferredBytes += len);
            }
       } catch (IOException e) {
            if ("Error writing request body to server".equalsIgnoreCase(e.getMessage())) {
                getLogger().log(Level.SEVERE, e.toString());
                connection.disconnect();
                return ERR_WRITING_REQUEST_BODY_TO_SERVER;
            }
            throw e;
          } finally {
  
            // Close only the HttpURLConnection stream.
            // Closing StorageObject streams is responsibility of the caller to sync(StorageObject).
            if (upload) {
                try {
                    os.close();
                } catch (IOException ignored) {
                    // not expected to happen
                }
            } else {
                try {
                    is.close();
                } catch (IOException ignored) {
                    // not expected to happen
                }
            }
        }

        responseCode = connection.getResponseCode();
        final String checksum = connection.getHeaderField("ETag");

        final String date = connection.getHeaderField("Last-Modified");
        
        if (checksum != null && digest != null) {
            //Verify MD5 checksum (objects < 5GB)
            String digestOut = bytesToHexString(digest.digest());
            if (!checksum.equals(digestOut)) {
                getLogger().log(Level.INFO, "Storage Cloud Service: checksum mismatch");
                return ERR_BAD_CHECKSUM;
            }
        }
        
        if (upload && responseCode == HttpURLConnection.HTTP_CREATED) {
            ((StorageObjectDelegate)storageObject).setAttributes(date, transferredBytes);
        } else if (!upload && (responseCode == HttpURLConnection.HTTP_OK || responseCode == HttpURLConnection.HTTP_PARTIAL)) {
            Map<String, List<String>> headers = connection.getHeaderFields();
            Set<Map.Entry<String, List<String>>> entrySet = headers.entrySet();
            for (Map.Entry<String, List<String>> entry : entrySet) {
                try {
                    String key = entry.getKey();
                    List<String> values = entry.getValue();
                    // if there are no key or no value then drop it
                    if (key == null || values == null || values.size() == 0) {
                        continue;
                    }
                    if (key.startsWith("X-Object-Meta-")) {
                        storageObject.setCustomMetadata(key.substring("X-Object-Meta-".length()), values.get(0));
                    }
                } catch (IllegalArgumentException | NullPointerException ex) {
                }
            }
        }

        return responseCode;
    }

    /**
     * GET the storage authentication.
     * @return the StorageAuthenticationResponse to the request
     * @throws IOException
     * @throws GeneralSecurityException 
     */
    private StorageAuthenticationResponse getStorageAuthentication()
            throws IOException, GeneralSecurityException {

        HttpResponse response =
            secureConnection.get(REST_STORAGE_AUTHENTICATION);

        int status = response.getStatus();

        if (status != HttpURLConnection.HTTP_OK) {
            throw new TransportException(status, response.getVerboseStatus("GET",
                REST_STORAGE_AUTHENTICATION));
        }

        String jsonResponse = new String(response.getData(), "UTF-8");
        JSONObject json;
        try {
            json = new JSONObject(jsonResponse);
        } catch (JSONException e) {
            throw new RuntimeException(e);
        }

        StorageAuthenticationResponse storageAuthenticationResponse =
            StorageAuthenticationResponse.fromJson(json);
        if (getLogger().isLoggable(Level.FINER)) {
            getLogger().log(Level.FINER,
                storageAuthenticationResponse.toString());
        }

        return storageAuthenticationResponse;
    }
    
    private String authenticate() {
        try {
            StorageAuthenticationResponse storageAuthenticationResponse
                    = getStorageAuthentication();
            storageContainerUrl = storageAuthenticationResponse.getStorageContainerUrl();
            return storageAuthenticationResponse.getAuthToken();
        } catch (Exception e) {
            getLogger().log(Level.INFO, "IoT storage API cannot be accessed: " + e.getMessage());
        }
        return null;
    }

    private static String bytesToHexString(byte[] in) {
        final StringBuilder sb = new StringBuilder();
        for(byte b : in) {
            sb.append(String.format("%02x", b));
        }
        return sb.toString();
    }

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client.StorageConnection");
    private static Logger getLogger() { return LOGGER; }   
}
