/*
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl;

import com.oracle.iot.client.DeviceModelAction;
import com.oracle.iot.client.DeviceModelAttribute;
import com.oracle.iot.client.DeviceModelAttribute.Type;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.List;
import java.util.Locale;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * DeviceModelActionImpl
 */
class DeviceModelActionImpl extends DeviceModelAction {

    private final String name;
    private final String description;
    private final List<Argument> argumentList;
    private final String alias;

    static class ArgumentImpl extends DeviceModelAction.Argument {

        private final String name;
        private final DeviceModelAttribute.Type argType;
        private final Number lowerBound;
        private final Number upperBound;
        private final Object defaultValue;

        ArgumentImpl(String name, DeviceModelAttribute.Type argType,
            Number lowerBound, Number upperBound, Object defaultValue) {
            super();
            this.name = name;
            this.argType = argType;
            if (this.argType == DeviceModelAttribute.Type.INTEGER ||
                this.argType == DeviceModelAttribute.Type.NUMBER) {
                this.lowerBound = lowerBound;
                this.upperBound = upperBound;
            } else {
                this.lowerBound = this.upperBound = null;
            }
            this.defaultValue = defaultValue;
        }

        @Override
        public String getName() {
            return name;
        }

        @Override
        public DeviceModelAttribute.Type getArgType() {
            return argType;
        }

        @Override
        public Number getLowerBound() {
            return lowerBound;
        }

        @Override
        public Number getUpperBound() {
            return upperBound;
        }

        @Override
        public Object getDefaultValue() {
            return defaultValue;
        }

        @Override
        public String toString() {
            StringBuilder b = new StringBuilder("{name = ");
            b.append(name);
            b.append(", type = ");
            b.append(argType);
            b.append(", lowerBound = ");
            b.append(lowerBound);
            b.append(", upperBound = ");
            b.append(upperBound);
            b.append(", default = ");
            b.append(String.valueOf(defaultValue));
            b.append("}");
            return b.toString();
        }

        static Argument fromJSON(JSONObject jsonObject) throws JSONException {
            final String name = jsonObject.optString("name", "value");
            final String type = jsonObject.getString("type");
            // possible IllegalArgumentException here
            final Type argType;
            try {
                argType = Type.valueOf(type);
            } catch (IllegalArgumentException e) {
                throw new JSONException("Invalid type: " + type);
            }
            final String range = jsonObject.optString("range", null);
            final Number min;
            final Number max;
            if (range != null) {
                final NumberFormat nf = NumberFormat.getNumberInstance(Locale.ROOT);
                final String[] strings = range.split(",");
                try {
                    min = nf.parse(strings[0]);
                    max = nf.parse(strings[1]);
                } catch (ParseException e) {
                    throw new JSONException("Invalid range: " + range);
                }
            } else {
                min = null;
                max = null;
            }
            final Object defaultValue = jsonObject.opt("defaultValue");
            return new ArgumentImpl(name, argType, min, max, defaultValue);
        }

    }

    DeviceModelActionImpl(String name, String description,
                                 List<Argument> argumentList,
                                 String alias) {
        this.name = name;
        this.description = description;
        this.argumentList = argumentList;
        this.alias = alias;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public List<Argument> getArguments() {
        return argumentList;
    }

    @Override
    public String getAlias() {
        return alias;
    }

    @Override
    public String toString() {
        StringBuilder b = new StringBuilder("name = ");
        b.append(name);
        b.append(", description = ");
        b.append(description);
        b.append(", arguments = [");
        boolean first = true;
        final String newline = System.getProperty("line.separator");
        for (Argument argument : argumentList) {
            if (!first || (first = false)) {
                b.append(',').append(newline);
            }
            b.append('\t').append(argument.toString());
        }
        b.append("], alias = ");
        b.append(alias);
        return b.toString();
    }
}
