/*
 * Copyright (c) 2017, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.device.persistence;

import com.oracle.iot.client.message.Message;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Support for persisting messages for guaranteed delivery.
 */
public class BatchByPersistenceImpl extends BatchByPersistence {

    /* package */ static final String TABLE_NAME = "BATCH_BY";

    /**
     * Create an implementation of MessagePersistence.
     * @param messagePersistence the application {@code android.content.Context}
     */
    public BatchByPersistenceImpl(MessagePersistence messagePersistence) {
        super();
        this.messageCollection = new ArrayList<>(1);
        this.delegate = (MessagePersistenceImpl)messagePersistence;
    }

    @Override
    public void save(Message message, String endpointId) {
        synchronized (messageCollection) {
            try {
                messageCollection.add(message);
                this.delegate.save(TABLE_NAME, messageCollection, endpointId);
            } finally {
                messageCollection.clear();
            }
        }
    }

    @Override
    public void save(Collection<Message> m, String endpointId) {
        this.delegate.save(TABLE_NAME, m, endpointId);
    }

    @Override
    public void delete(Collection<Message> m) {
        this.delegate.delete(TABLE_NAME, m);
    }

    @Override
    public List<Message> load(String endpointId) {
        return this.delegate.load(TABLE_NAME, endpointId);
    }

    private final MessagePersistenceImpl delegate;
    private final Collection<Message> messageCollection;


}
