/*
 * Copyright (c) 2017, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.device.persistence;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteStatement;
import com.oracle.iot.client.message.Message;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Support for persisting messages for guaranteed delivery.
 */
public class MessagePersistenceImpl extends MessagePersistence {

    // If schema changeds, update the version and handle change in onUpgrade method.
    private static final int DB_VERSION = 1;

    // Default name of the database
    private static final String DB_NAME = PersistenceMetaData.getDBName();

    private static final String TABLE_NAME = "MESSAGES";

    /**
     * Create an implementation of MessagePersistence.
     * @param context the application {@code android.content.Context}
     */
    public MessagePersistenceImpl(Object context) {
        super();

        // If context is null, an in-memory database is created, but the database name also needs to be null.
        this.dbHelper = new DBHelper((Context)context, DB_NAME);
    }

    @Override
    public void save(Collection<Message> messages, String endpointId) {
        save(TABLE_NAME, messages, endpointId);
    }

    /*package*/
    synchronized void save(String tableName, Collection<Message> messages, String endpointId) {

        SQLiteDatabase sqLiteDatabase = null;
        SQLiteStatement sqLiteStatement = null;
        try {

            sqLiteDatabase = dbHelper.getWritableDatabase();
            sqLiteDatabase.beginTransaction();

            sqLiteStatement = sqLiteDatabase.compileStatement( "INSERT INTO " + tableName + " (TIMESTAMP,UUID,ENDPOINT_ID,MESSAGE) VALUES (?, ?, ?, ?)");

            for (Message message : messages) {

                sqLiteStatement.clearBindings();

                final byte[] blob;
                try {
                    blob = message.toJson().toString().getBytes("UTF-8");
                } catch (UnsupportedEncodingException cannot_happen) {
                    // Cannot happen. UTF-8 is a required encoding
                    // Throw runtime exception to make compiler happy
                    throw new RuntimeException(cannot_happen);
                }

                sqLiteStatement.bindLong(1, message.getEventTime());
                sqLiteStatement.bindString(2, message.getClientId());
                sqLiteStatement.bindString(3, endpointId);
                sqLiteStatement.bindBlob(4, blob);
                sqLiteStatement.execute();
            }

            sqLiteDatabase.setTransactionSuccessful();
        } catch (SQLiteException e) {
            getLogger().log(Level.WARNING, "SQL exception during saving messages to database.", e);
        } finally {
            if (sqLiteStatement != null) {
                sqLiteStatement.close();
            }
            if (sqLiteDatabase != null) {
                // if transaction was not set successful, it is rolled back
                sqLiteDatabase.endTransaction();
            }
        }
    }

    @Override
    public void delete(Collection<Message> messages) {
        delete(TABLE_NAME, messages);
    }

    /* package */ synchronized void delete(String tableName, Collection<Message> messages) {
        SQLiteDatabase sqLiteDatabase = null;
        SQLiteStatement sqLiteStatement = null;
        try {

            sqLiteDatabase = dbHelper.getWritableDatabase();
            sqLiteDatabase.beginTransaction();

            sqLiteStatement = sqLiteDatabase.compileStatement( "DELETE FROM " + tableName + " WHERE UUID = ?");

            for (Message message : messages) {
                sqLiteStatement.bindString(1, message.getClientId());
                sqLiteStatement.execute();
            }

            sqLiteDatabase.setTransactionSuccessful();
        } catch (SQLiteException e) {
            getLogger().log(Level.WARNING, "SQL exception during saving messages to database.", e);
        } finally {
            if (sqLiteStatement != null) {
                sqLiteStatement.close();
            }
            if (sqLiteDatabase != null) {
                // if transaction was not set successful, it is rolled back
                sqLiteDatabase.endTransaction();
            }
        }
    }

    @Override
    public List<Message> load(String endpointId) {
        return load(TABLE_NAME, endpointId);
    }

    /* package */ synchronized List<Message> load(String tableName, String endpointId) {

        SQLiteDatabase sqLiteDatabase = null;
        Cursor cursor = null;
        try {

            sqLiteDatabase = dbHelper.getWritableDatabase();

            cursor = sqLiteDatabase.rawQuery(
                    "SELECT MESSAGE FROM " + tableName + " WHERE ENDPOINT_ID = ? ORDER BY TIMESTAMP",
                    new String[]{endpointId}
            );

            final List<Message> messages = new ArrayList<>();

            if (cursor != null && cursor.moveToFirst()) {
                do {
                    final byte[] bytes = cursor.getBlob(cursor.getColumnIndex("MESSAGE"));
                    final List<Message> list = Message.fromJson(bytes);
                    final Message message = list.isEmpty() ? null : list.get(0);
                    messages.add(message);

                } while (cursor.moveToNext());
            }
            return messages;

        } catch (SQLiteException e) {
            getLogger().log(Level.WARNING, "SQL exception during saving messages to database.", e);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
        }
        return Collections.emptyList();
    }

    private static class DBHelper extends SQLiteOpenHelper {

        private DBHelper(Context context, String dbName) {
            super(context, context != null ? dbName : null, CURSOR_FACTORY_IS_NULL, DB_VERSION);
        }

        @Override
        public void onCreate(SQLiteDatabase sqLiteDatabase) {

            sqLiteDatabase.execSQL(
                    "CREATE TABLE IF NOT EXISTS " + TABLE_NAME
                            + "(TIMESTAMP BIGINT NOT NULL,"
                            + "UUID VARCHAR(40) NOT NULL,"
                            + "ENDPOINT_ID VARCHAR(40) NOT NULL,"
                            + "MESSAGE BLOB, PRIMARY KEY (UUID));"
                            + "CREATE INDEX IF NOT EXISTS endpoint_id ON "
                            + TABLE_NAME + "(ENDPOINT_ID);"
            );

            sqLiteDatabase.execSQL(
                "CREATE TABLE IF NOT EXISTS " + BatchByPersistenceImpl.TABLE_NAME
                    + "(TIMESTAMP BIGINT NOT NULL,"
                    + "UUID VARCHAR(40) NOT NULL,"
                    + "ENDPOINT_ID VARCHAR(40) NOT NULL,"
                    + "MESSAGE BLOB, PRIMARY KEY (UUID));"
                    + "CREATE INDEX IF NOT EXISTS endpoint_id ON "
                    + BatchByPersistenceImpl.TABLE_NAME + "(ENDPOINT_ID);"
            );
        }

        @Override
        public void onUpgrade(SQLiteDatabase sqLiteDatabase, int i, int i1) {
            // Implement this method if DB_VERSION changes
        }
    }

    private final SQLiteOpenHelper dbHelper;

    // self documenting code
    private final static SQLiteDatabase.CursorFactory CURSOR_FACTORY_IS_NULL = null;

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }

}
