/*
 * Copyright (c) 2016, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */
package com.oracle.iot.client.impl.trust;

import java.security.GeneralSecurityException;

import android.content.Context;

import com.oracle.iot.client.trust.TrustedAssetsManager;
import com.oracle.iot.client.trust.TrustException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import java.io.File;
import java.io.OutputStream;
import java.io.FileOutputStream;
import java.security.KeyStore;
import java.util.Enumeration;
import java.util.logging.Logger;
import android.util.Log;


public class TrustedAssetsManagerFactory {
    // Configuration file within assets directory
    static final String TA_STORE_CONFIG_FILE_PATH = "trustedAssets.properties";

    // Properties
    static final String TA_STORE_PROPERTY =
        "oracle.iot.client.trustedAssetsStore";
    static final String TA_STORE_PASSWORD_PROPERTY =
        "oracle.iot.client.trustedAssetsStorePassword";
    static final String TA_STORE_LOCAL_RW_PROPERTY =
        "oracle.iot.client.trustedAssetsStoreRw";
    static final String TA_STORE_LOCAL_RW_PASSWORD_PROPERTY =
        "oracle.iot.client.trustedAssetsStoreRwPassword";

    // Default property values
    static final String DEFAULT_TA_STORE = "trustedAssetsStore.bks";
    static final String DEFAULT_TA_STORE_LOCAL_RW = "trustedAssetsKeyStore.bks";

    public static TrustedAssetsManager create(Object cntxt)
            throws GeneralSecurityException {
        if (null == cntxt) {
            throw new NullPointerException("Context can't be null.");
        }
        Context context = (Context)cntxt;
        Properties properties = getConfigurationFromAssets(context);
        String taStoreRoFileName = properties.getProperty(
            TA_STORE_PROPERTY, DEFAULT_TA_STORE);
        String taStoreRoPassword = properties.getProperty(
            TA_STORE_PASSWORD_PROPERTY, "");
        String taStoreRwFileName = properties.getProperty(
            TA_STORE_LOCAL_RW_PROPERTY, DEFAULT_TA_STORE_LOCAL_RW);
        String taStoreRwPassword = properties.getProperty(
            TA_STORE_LOCAL_RW_PASSWORD_PROPERTY, "");

        // Check if the private Keystore exist otherwise copy it from the RO Keystore
        File taStoreRwFile = new File(context.getFilesDir(), taStoreRwFileName);
        KeyStore.PasswordProtection passwordParamRo =
            new KeyStore.PasswordProtection(taStoreRoPassword.toCharArray());
        if ( !taStoreRwFile.exists()) {
            InputStream is = null;
            OutputStream os = null;
            try {
				is = (InputStream)context.getApplicationContext().
                        getAssets().open(taStoreRoFileName);
				os = context.openFileOutput(taStoreRwFileName,
                        Context.MODE_PRIVATE);
				byte[] buf = new byte[1024];
				int bytesRead;
				while ((bytesRead = is.read(buf)) > 0) {
					os.write(buf, 0, bytesRead);
				}
                if (taStoreRoFileName.endsWith(".bks")) {
					return new DefaultTrustedAssetsManager(taStoreRwFile.getPath(),
										taStoreRoPassword, context);
                } else {
				/** The trustFile is UnifiedTrustFormat file **/
                    return new UnifiedTrustedAssetsManager(taStoreRwFile.getPath(),
										taStoreRoPassword, context);
                }
            } catch (Exception e) {
                getLogger().severe(e + e.getMessage() +
                    "Error while creating trust assets");
                throw new TrustException("Error loading trust assets...", e);
            } finally {
                try {
                    if (is != null) {
                        is.close();
                    }
                } catch (IOException ex) {
                }
                try {
                    if (os != null) {
                        os.close();
                    }
                } catch (IOException ex) {
                }
            }
        }
        return create(taStoreRwFile.getPath(), taStoreRoPassword, context);
    }

    private static Properties getConfigurationFromAssets(Context context) throws TrustException {
		InputStream configInStream = null;
		try {
			configInStream = (InputStream) context.getApplicationContext().getAssets().open(TA_STORE_CONFIG_FILE_PATH);
			Properties config = new Properties();
			config.load(configInStream);
			return config;
		} catch (Exception ex) {
			throw new TrustException("Configuration file cannot be loaded");
		} finally {
			try {
				if (configInStream != null) {
					configInStream.close();
				}
			} catch (IOException ex) {
			}
		}
    }

    public static TrustedAssetsManager create(String path, String password,
            Object context) throws GeneralSecurityException {
        if (path.endsWith(".bks")) {
            return new DefaultTrustedAssetsManager(path, password, null);
        }

        return new UnifiedTrustedAssetsManager(path, password, null);
    }
    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }
}
