/*
 * Copyright (c) 2015, 2018, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.impl.device;

import com.oracle.iot.client.DeviceModelAction;
import com.oracle.iot.client.DeviceModelAttribute;
import com.oracle.iot.client.VirtualDeviceAttribute;
import com.oracle.iot.client.device.DirectlyConnectedDevice;
import com.oracle.iot.client.device.util.MessageDispatcher;
import com.oracle.iot.client.device.util.RequestDispatcher;
import com.oracle.iot.client.device.util.RequestHandler;
import com.oracle.iot.client.device.util.StorageDispatcher;
import com.oracle.iot.client.impl.DeviceModelImpl;
import com.oracle.iot.client.impl.StorageConnectionBase;
import com.oracle.iot.client.impl.VirtualDeviceAttributeBase;
import com.oracle.iot.client.impl.VirtualDeviceBase;
import com.oracle.iot.client.message.DataItem;
import com.oracle.iot.client.message.DataMessage;
import com.oracle.iot.client.message.Message;
import com.oracle.iot.client.message.RequestMessage;
import com.oracle.iot.client.message.ResponseMessage;
import com.oracle.iot.client.message.StatusCode;

import oracle.iot.client.AbstractVirtualDevice;
import oracle.iot.client.DeviceModel;
import oracle.iot.client.StorageObject;
import oracle.iot.client.device.Alert;
import oracle.iot.client.device.Data;
import oracle.iot.client.device.VirtualDevice;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Locale;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * VirtualDeviceImpl
 */
public final class VirtualDeviceImpl
        extends VirtualDevice
        implements VirtualDeviceBase.Adapter<VirtualDevice>, RequestHandler {

    private final VirtualDeviceBase<VirtualDevice> base;

    // used from AlertImpl
    final DirectlyConnectedDevice directlyConnectedDevice;

    private final Map<String,VirtualDeviceAttributeBase<VirtualDevice, Object>> attributeMap;

    private static final ErrorCallbackBridge ERROR_CALLBACK_BRIDGE =
            new ErrorCallbackBridge();

    public VirtualDeviceImpl(DirectlyConnectedDevice directlyConnectedDevice,
                             String endpointId,
                             DeviceModelImpl deviceModel) {
        super();
        this.base = new VirtualDeviceBase(this, endpointId, deviceModel);
        this.directlyConnectedDevice = directlyConnectedDevice;
        this.attributeMap = createAttributeMap(this, deviceModel);
        final MessageDispatcher messageDispatcher =
                MessageDispatcher.getMessageDispatcher(directlyConnectedDevice);

        final RequestDispatcher requestDispatcher =
                messageDispatcher.getRequestDispatcher();

        // handler for all attribute and action requests
        requestDispatcher.registerRequestHandler(endpointId, "deviceModels/"+deviceModel.getURN(), this);

        ERROR_CALLBACK_BRIDGE.add(this);
        messageDispatcher.setOnError(ERROR_CALLBACK_BRIDGE);
    }

    private static Map<String, VirtualDeviceAttributeBase<VirtualDevice, Object>> createAttributeMap(VirtualDeviceImpl virtualDevice, DeviceModel deviceModel) {

        final Map<String, VirtualDeviceAttributeBase<VirtualDevice, Object>> map =
                new HashMap<String, VirtualDeviceAttributeBase<VirtualDevice, Object>>();

        if (deviceModel instanceof DeviceModelImpl) {
            DeviceModelImpl deviceModelImpl = (DeviceModelImpl) deviceModel;
            for(DeviceModelAttribute attribute : deviceModelImpl.getDeviceModelAttributes().values()) {
                VirtualDeviceAttributeImpl<Object> vda = new VirtualDeviceAttributeImpl(virtualDevice, attribute);
                map.put(attribute.getName(), vda);
                String alias = attribute.getAlias();
                if (alias != null && alias.length() != 0) {
                    map.put(alias, vda);
                }
            }
        }
        return map;
    }

    /**
     * VirtualDeviceBase.Adapter API
     * {@inheritDoc}
     */
    @Override
    public VirtualDeviceAttributeBase<VirtualDevice, Object> getAttribute(String attributeName) {

        final VirtualDeviceAttributeBase<VirtualDevice, Object> virtualDeviceAttribute =
                attributeMap.get(attributeName);

        if (virtualDeviceAttribute == null) {
            throw new IllegalArgumentException
                ("no such attribute '" + attributeName +
                        "'.\n\tVerify that the URN for the device model you created " +
                        "matches the URN that you use when activating the device in " +
                        "the Java application.\n\tVerify that the attribute name " +
                        "(and spelling) you chose for your device model matches the " +
                        "attribute you are setting in the Java application.");
        }

        return virtualDeviceAttribute;
    }


    /**
     * VirtualDeviceBase.Adapter API
     * {@inheritDoc}
     */
    @Override
    public void setValue(VirtualDeviceAttributeBase<VirtualDevice, Object> attribute, Object value) {

        if (attribute == null) {
            throw new IllegalArgumentException("attribute may not be null");
        }

        attribute.set(value);
    }

    /**
     * Set all the attributes in an update batch. Errors are handled in the
     * set call, including calling the on error handler.
     * {@inheritDoc}
     */
    @Override
    public void updateFields(Map<VirtualDeviceAttributeBase<VirtualDevice, Object>, Object> updatedAttributes) {

        final Set<Map.Entry<VirtualDeviceAttributeBase<VirtualDevice, Object>, Object>> entries
                = updatedAttributes.entrySet();

        final Iterator<Map.Entry<VirtualDeviceAttributeBase<VirtualDevice, Object>, Object>> iterator
                = entries.iterator();

        while (iterator.hasNext()) {
            final Map.Entry<VirtualDeviceAttributeBase<VirtualDevice, Object>, Object> entry = iterator.next();
            final VirtualDeviceAttributeBase<VirtualDevice, Object> attribute = entry.getKey();
            final Object value = entry.getValue();

            try {

                // Here, we assume
                // 1. That attribute is not null. If the attribute were not found
                //    an exception would have been thrown from the VirtualDevice
                //    set(String attributeName, T value) method.
                // 2. That the set method validates the value. The call to
                //    update here should not throw an exception because the
                //    value is bad.
                // 3. The author of this code knew what he was doing.

                attribute.update(value);

            } catch (Exception e) {
            }
        }

        processOnChange(updatedAttributes);

    }

    @Override
    public DeviceModel getDeviceModel() {
        return base.getDeviceModel();
    }

    @Override
    public String getEndpointId() {
        return base.getEndpointId();
    }

    @Override
    public <T> T get(String attributeName) {
        VirtualDeviceAttributeBase<VirtualDevice, Object> attribute = getAttribute(attributeName);
        return (T)attribute.get();
    }

    @Override
    public <T> T getLastKnown(String attributeName) {
        VirtualDeviceAttributeBase<VirtualDevice, Object> attribute = getAttribute(attributeName);
        return (T)attribute.getLastKnown();
    }

    @Override
    public <T> VirtualDevice set(String attributeName, T value) {
        base.set(attributeName, value);
        return this;
    }

    @Override
    public VirtualDevice update() {
        base.update();
        return this;
    }

    @Override
    public void finish() {
        base.finish();
    }

    @Override
    public void setOnChange(String attributeName, ChangeCallback callback) {
        VirtualDeviceAttribute attribute = getAttribute(attributeName);
        attribute.setOnChange(callback);
    }

    @Override
    public void setOnChange(ChangeCallback callback) {
        base.setOnChange(callback);
    }

    ChangeCallback getChangeCallback() {
        return base.getOnChangeCallback();
    }

    @Override
    public void setOnError(ErrorCallback callback) {
        base.setOnError(callback);
    }

    ErrorCallback getErrorCallback() {
        return base.getOnErrorCallback();
    }

    @Override
    public void setOnError(String attributeName, ErrorCallback<VirtualDevice> callback) {
        VirtualDeviceAttribute<VirtualDevice,?> attribute = getAttribute(attributeName);
        attribute.setOnError(callback);
    }

    @Override
    public ResponseMessage handleRequest(RequestMessage requestMessage) throws Exception {

            //
            // Dispatch the request message to the appropriate handler for the method
            //

            final String method = requestMessage.getMethod().toUpperCase(Locale.ROOT);

            StatusCode responseStatus = StatusCode.BAD_REQUEST;

            if ("POST".equals(method)) {
                String methodOverride =
                        requestMessage.getHeaderValue("X-HTTP-Method-Override");
                if ("PATCH".equalsIgnoreCase(methodOverride)) {
                    responseStatus = handlePatch(requestMessage);
                } else {
                    responseStatus = handlePost(requestMessage);
                }

            } else if ("PUT".equals(method)) {
                responseStatus = handlePut(requestMessage);

            } else if ("PATCH".equals(method)) {
                responseStatus = handlePatch(requestMessage);

//        } else if ("GET".equals(method)) {
//            For a GET, the server should return the value

            } else {
                getLogger().severe("unexpected method: " + method);
            }

            return new ResponseMessage.Builder(requestMessage)
                    .statusCode(responseStatus)
                    .build();
    }

    // Patch is always a multi-attribute PUT
    private StatusCode handlePatch(RequestMessage requestMessage) {
        NamedValue<?> root = null;
        VirtualDeviceBase.NamedValueImpl<?> last = null;
        try {
            byte[] rawData = requestMessage.getBody();
            String json = new String(rawData, "UTF-8");
            JSONObject jsonObject = new JSONObject(json);

            if (base.getOnChangeCallback() == null) {
                /*
                 * If there is no device level onChange callback then all
                 * attributes in the message must have handlers.
                 */
                Iterator<String> keys = jsonObject.keys();
                while (keys.hasNext()) {
                    String attributeName = keys.next();
                    Object jsonValue = jsonObject.get(attributeName);

                    /*
                     * Throws IllegalArgumentException if attribute is not
                     * in model.
                     */
                    VirtualDeviceAttributeImpl attribute =
                        (VirtualDeviceAttributeImpl)getAttribute(attributeName);

                    if (attribute.getOnChange() == null) {
                        getLogger().log(Level.INFO, "No handler for: '" +
                                requestMessage.getMethod().toUpperCase(Locale.ROOT) + " " + requestMessage.getURL());
                        return StatusCode.NOT_FOUND;
                    }
                }
            }

            Iterator<String> keys = jsonObject.keys();
            while (keys.hasNext()) {
                String attributeName = keys.next();
                Object jsonValue = jsonObject.get(attributeName);

                // Throws IllegalArgumentException if attribute is not in model
                VirtualDeviceAttributeImpl attribute =
                    (VirtualDeviceAttributeImpl) getAttribute(attributeName);

                DeviceModelAttribute dma = attribute.getDeviceModelAttribute();
                Object newValue = getValue(dma.getType(), jsonValue, attributeName);

                // Throws IllegalArgumentException if newValue isn't right type
                attribute.validate(dma, newValue);

                final Object oldValue = attribute.get();
                // Do not update the value, or call onChange, if the value hasn't changed.
                if (oldValue != null ? oldValue.equals(newValue) : newValue == null) {
                    continue;
                }

                VirtualDeviceBase.NamedValueImpl<?> nameValue =
                    new VirtualDeviceBase.NamedValueImpl<Object>(attributeName,
                    newValue);

                if (attribute.getOnChange() != null) {
                    attribute.getOnChange().onChange(
                        new VirtualDeviceBase.ChangeEvent<VirtualDevice>(
                        this, nameValue));
                }

                if (last != null) {
                    last.setNext(nameValue);
                    last = nameValue;
                } else {
                    root = last = nameValue;
                }
            }
        } catch (Exception e) {
            //
            // The onChange method may throw exceptions. Since the CL has no
            // knowledge of what these might be, Exception is caught here.
            //
            // Possible exceptions from CL internals are JSONException,
            // IllegalArgumentException, NullPointerException, and
            // ClassCastException.
            //
            getLogger().log(Level.SEVERE, e.getMessage(), e);
            return StatusCode.BAD_REQUEST;
        }

        // root == null just means that no values were updated,
        // which can happen if the new values equal the old values.
        if (root == null) {
            return StatusCode.ACCEPTED;
        }

        // patch is handled by change callback set on VirtualDevice
        ChangeCallback<VirtualDevice> changeCallback =
            base.getOnChangeCallback();
        try {
            if (changeCallback != null) {
                changeCallback.onChange(
                    new VirtualDeviceBase.ChangeEvent<VirtualDevice>(
                            this,
                            root
                    )
                );
            }

            List<DataItem<?>> dataItems = new ArrayList<DataItem<?>>();

            // If no exception, then assume the event was handled.
            // Now go update the values.
            for (NamedValue<?> namedValue = root; namedValue != null;
                    namedValue = namedValue.next()) {
                String attributeName = namedValue.getName();
                VirtualDeviceAttributeImpl attribute =
                    (VirtualDeviceAttributeImpl) getAttribute(attributeName);
                Object newValue = namedValue.getValue();

                // Call update, not set! Update sets the value but does not
                // cause a DataMessage to be sent. We want to send one
                // DataMessage with all the updated data items.
                //
                // There should be no issues with calling update here. All the
                // checking was done while building the event chain, and
                // newValue should be the converted data from the
                // RequestMessage.
                attribute.update(newValue);

                DeviceModelAttribute dma = attribute.getDeviceModelAttribute();
                switch (dma.getType()) {
                    case INTEGER:
                        dataItems.add(new DataItem<Object>(attributeName,
                            (Integer)newValue));
                        break;
                    case NUMBER:
                        dataItems.add(new DataItem<Object>(attributeName,
                            ((Number)newValue).doubleValue()));
                        break;
                    case BOOLEAN:
                        dataItems.add(new DataItem<Object>(attributeName,
                            (Boolean)newValue));
                        break;
                    case URI:
                        dataItems.add(new DataItem<Object>(attributeName,
                            ((oracle.iot.client.ExternalObject)newValue).getURI()));
                        break;
                    case STRING:
                        dataItems.add(new DataItem<Object>(attributeName,
                            (String)newValue));
                        break;
                    case DATETIME:
                        dataItems.add(new DataItem<Object>(attributeName,
                            (Long)newValue));
                        break;
                    default:
                        getLogger().severe("'"+
                            dma.getType() + "' not handled");
                }
            }

            DataMessage dataMessage = new DataMessage.Builder()
                    .format(getDeviceModel().getURN() + ":attributes")
                    .source(getEndpointId())
                    .dataItems(dataItems)
                    .build();

            MessageDispatcher messageDispatcher =
                MessageDispatcher.getMessageDispatcher(
                directlyConnectedDevice);
            messageDispatcher.queue(dataMessage);

            return StatusCode.ACCEPTED;

        } catch (Exception e) {
            // The onChange method may throw exceptions. Since the CL has no
            // knowledge of what these might be, Exception is caught here.
            getLogger().log(Level.SEVERE, e.getMessage(), e);

            return StatusCode.INTERNAL_SERVER_ERROR;
        }

    }

    // POST is for actions
    private StatusCode handlePost(RequestMessage requestMessage) {
        final String path = requestMessage.getURL();
        final String dmURN = "deviceModels/"+getDeviceModel().getURN()+"/actions";
        final String actionName =
                dmURN.regionMatches(0, path, 0, dmURN.length())
                        ? path.substring(dmURN.length()+1)
                        : path;
		final Callable callable;
		if (actionMap == null) {
			callable = null;
		} else {
			synchronized (actionMapLock) {
				callable = actionMap.get(actionName);
			}
		}
        if (callable != null) {
            try {
                DeviceModelAction action
                        = getDeviceModelAction(base.getDeviceModel(), actionName);
                Object data = action.getArgType() != null
                        ? getValue(action.getArgType(), requestMessage.getBody(), actionName)
                        : null;
                callable.call(this, data);
                return StatusCode.ACCEPTED;
            } catch (Exception e) {
                // The call method may throw exceptions. Since the CL has no
                // knowledge of what these might be, Exception is caught here.
                //
                // Possible exceptions from CL internals are JSONException,
                // IllegalArgumentException, NullPointerException, and
                // ClassCastException.
                //
                getLogger().log(Level.FINE, e.getMessage(), e);
                return  StatusCode.BAD_REQUEST;
            }
        }

        getLogger().log(Level.INFO, "No handler for: '" +
                requestMessage.getMethod().toUpperCase(Locale.ROOT) + " " + requestMessage.getURL());

        return StatusCode.NOT_FOUND;

    }

    // PUT is for attributes or for resources
    private StatusCode handlePut(RequestMessage requestMessage) {

        try {

            final String path = requestMessage.getURL();
            final String dmURN =
                    "deviceModels/"+getDeviceModel().getURN()+"/attributes";
            final String attributeName =
                    dmURN.regionMatches(0, path, 0, dmURN.length())
                            ? path.substring(dmURN.length()+1)
                            : path;

            final VirtualDeviceAttributeBase<VirtualDevice, Object>
                    virtualDeviceAttribute = getAttribute(attributeName);
            final DeviceModelAttribute deviceModelAttribute =
                    virtualDeviceAttribute.getDeviceModelAttribute();
            final DeviceModelAttribute.Type attributeType =
                    deviceModelAttribute.getType();

            final byte[] data = requestMessage.getBody();
            final Object newValue = getValue(attributeType,data,attributeName);
            final Object oldValue = virtualDeviceAttribute.get();
            // Do not update the value, or call onChange, if the value hasn't changed.
            if (oldValue != null ? oldValue.equals(newValue) : newValue == null) {
                return StatusCode.ACCEPTED;
            }

            final NamedValue<?> namedValue =
                    new VirtualDeviceBase.NamedValueImpl<Object>(attributeName, newValue);
            boolean attrOnChangeCalled = false;

            if (virtualDeviceAttribute.getOnChange() != null) {
                virtualDeviceAttribute.getOnChange().onChange(
                        new VirtualDeviceBase.ChangeEvent<VirtualDevice>(
                                this,
                                namedValue
                        )
                );

                attrOnChangeCalled = true;
            }

            if (base.getOnChangeCallback() != null) {
                base.getOnChangeCallback().onChange(
                        new VirtualDeviceBase.ChangeEvent<VirtualDevice>(
                                this,
                                namedValue
                        )
                );

                attrOnChangeCalled = true;
            }


            if (attrOnChangeCalled) {
                // Call set, not update! Update sets the value but does not
                // cause a DataMessage to be sent. Since the put is for a
                // single attribute only, we want the DataMessage to be sent.
                // This is different than handlePatch.
                virtualDeviceAttribute.set(newValue);
                return StatusCode.ACCEPTED;

            } else {

                getLogger().log(Level.INFO, "No handler for: '" +
                        requestMessage.getMethod().toUpperCase(Locale.ROOT) + " " + requestMessage.getURL());
                return StatusCode.NOT_FOUND;
            }


        } catch (Exception e) {
            // The call method may throw exceptions. Since the CL has no
            // knowledge of what these might be, Exception is caught here.
            //
            // Possible exceptions from CL internals are JSONException,
            // IllegalArgumentException, NullPointerException, and
            // ClassCastException.
            //
            getLogger().log(Level.FINE, e.getMessage(), e);
            return StatusCode.BAD_REQUEST;
        }

    }

	private final Object actionMapLock = new Object();
    private volatile Map<String, Callable<?>> actionMap;

    private static DeviceModelAction getDeviceModelAction(DeviceModelImpl deviceModel, String actionName) {
        if (deviceModel != null) {
            Map<String, DeviceModelAction> actions = deviceModel.getDeviceModelActions();
            if (!actions.isEmpty()) {
                DeviceModelAction act = actions.get(actionName);
                if (act != null)
                    return act;
                for (DeviceModelAction action : actions.values()) {
                    if (actionName.equals(action.getAlias())) {
                        return action;
                    }
                }
            }
        }
        return null;
    }

    @Override
    public void setCallable(String actionName, Callable<?> callable) {
        if (actionMap == null) {
			synchronized (actionMapLock) {
				if (actionMap == null) {
					actionMap = new HashMap<String, Callable<?>>();
				}
			}
        }

        Map<String, DeviceModelAction> actions = base.getDeviceModel().getDeviceModelActions();
        DeviceModelAction deviceModelAction = actions.get(actionName);
        if (deviceModelAction == null) {
            for(DeviceModelAction action : actions.values()) {
                if (actionName.equals(action.getAlias())) {
                    deviceModelAction = action;
                    break;
                }
            }
        }
        if (deviceModelAction == null) {
            throw new IllegalArgumentException("action not found in model");
        }

		synchronized (actionMapLock) {
			actionMap.put(deviceModelAction.getName(), callable);
			if (deviceModelAction.getAlias() != null) {
				actionMap.put(deviceModelAction.getAlias(), callable);
			}
		}
    }

    @Override
    public Alert createAlert(String format) {
        return new AlertImpl(this, format);
    }

    @Override
    public Data createData(String format) {
        return new DataImpl(this, format);
    }
    
    // called from VirtualDeviceAttributeImpl
    <T> void processOnChange(VirtualDeviceAttribute<VirtualDevice, T> virtualDeviceAttribute, Object newValue) {
        final DataMessage.Builder builder = new DataMessage.Builder();
        List<StorageObjectImpl> storageObjects = new ArrayList<StorageObjectImpl>();
        try {
            processOnChange(builder, (VirtualDeviceAttributeImpl)virtualDeviceAttribute, newValue, storageObjects);
        } catch(RuntimeException re) {
            getLogger().log(Level.SEVERE, re.getMessage(), re);
            return;
        }
        DataMessage dataMessage = builder.build();

        MessageDispatcherImpl messageDispatcher =
                (MessageDispatcherImpl)MessageDispatcher.getMessageDispatcher(directlyConnectedDevice);
        try {
            for (StorageObjectImpl so:storageObjects) {
                messageDispatcher.addStorageObjectDependency(so, dataMessage.getClientId());
            }
            messageDispatcher.queue(dataMessage);
        } catch(ArrayStoreException e) {
            // MessageDispatcher queue is full.
            Set<VirtualDeviceAttributeBase<VirtualDevice, Object>> attributes =
                    new HashSet<VirtualDeviceAttributeBase<VirtualDevice, Object>>(1);
            attributes.add((VirtualDeviceAttributeImpl)virtualDeviceAttribute);
            notifyException(attributes, e);
        } catch (Exception e) {
            getLogger().severe(e.getMessage());
        }
    }

    private void processOnChange(Map<VirtualDeviceAttributeBase<VirtualDevice, Object>, Object> updatedAttributes) {

        if (updatedAttributes.isEmpty()) return;

        final DataMessage.Builder builder = new DataMessage.Builder();
        List<StorageObjectImpl> storageObjects = new ArrayList<StorageObjectImpl>();

        for (Map.Entry<VirtualDeviceAttributeBase<VirtualDevice, Object>, Object> entry : updatedAttributes.entrySet()) {

            final VirtualDeviceAttributeBase<VirtualDevice, Object> attribute = entry.getKey();
            final Object newValue = entry.getValue();

            try {
                processOnChange(builder, attribute, newValue, storageObjects);
            } catch(RuntimeException re) {
                getLogger().log(Level.SEVERE, re.getMessage(), re);
                return;
            }
        }
        DataMessage dataMessage = builder.build();

        MessageDispatcherImpl messageDispatcher =
                (MessageDispatcherImpl)MessageDispatcher.getMessageDispatcher(directlyConnectedDevice);
        try {
            for (StorageObjectImpl so:storageObjects) {
                messageDispatcher.addStorageObjectDependency(so, dataMessage.getClientId());
            }
            messageDispatcher.queue(dataMessage);
        } catch(ArrayStoreException e) {
            // MessageDispatcher queue is full.
            notifyException(updatedAttributes.keySet(), e);
        } catch (Exception e) {
            getLogger().severe(e.getMessage());
        }
    }
    
    void handleStorageObjectStateChange(StorageObjectImpl so) {
        MessageDispatcherImpl messageDispatcherImpl =
                (MessageDispatcherImpl)MessageDispatcher.getMessageDispatcher(directlyConnectedDevice);
        messageDispatcherImpl.removeStorageObjectDependency(so);
    }

    @Override
    public String toString() {
        return base.toString();
    }

    private Object getValue(DeviceModelAttribute.Type attributeType,
            byte[] payload, String name) throws JSONException, IllegalArgumentException {
        String json;
        try {
             json = new String(payload, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new JSONException(e.getMessage());
        }

        JSONObject jsonObject = new JSONObject(json);
        Object jsonValue = jsonObject.opt("value");
        if (jsonValue == null) {
            throw new IllegalArgumentException("bad payload " +
                String.valueOf(jsonObject));
        }

        return getValue(attributeType, jsonValue, name);
    }

    private Object getValue(DeviceModelAttribute.Type attributeType,
            Object jsonValue, String name) throws IllegalArgumentException {

        if (jsonValue == null) {
            return null;
        }

        Object data = null;

        switch (attributeType) {
            case BOOLEAN:
                if (!(jsonValue instanceof Boolean)) {
                    throw new IllegalArgumentException("value is not BOOLEAN");
                }

                data = jsonValue;
                break;

            case INTEGER:
                if (!(jsonValue instanceof Number)) {
                    throw new IllegalArgumentException("value is not NUMBER");
                }

                data = ((Number)jsonValue).intValue();
                break;

            case NUMBER:
                if (!(jsonValue instanceof Number)) {
                    throw new IllegalArgumentException("value is not NUMBER");
                }

                data = jsonValue;
                break;

            case DATETIME:
                if (!(jsonValue instanceof Number)) {
                    throw new IllegalArgumentException("value is not NUMBER");
                }

                data = ((Number)jsonValue).longValue();
                break;

            case STRING:
                if (!(jsonValue instanceof String)) {
                    throw new IllegalArgumentException("value is not STRING");
                }

                data = jsonValue;
                break;
            case URI:
                if (!(jsonValue instanceof String)) {
                    throw new IllegalArgumentException("value is not STRING");
                }
                try {
                    if (StorageConnectionBase.isStorageCloudURI((String)jsonValue)) {
                        try {
                            com.oracle.iot.client.StorageObject delegate =
                                    directlyConnectedDevice.createStorageObject((String)jsonValue);
                            StorageObjectImpl storageObjectImpl = new StorageObjectImpl(directlyConnectedDevice, delegate);
                            storageObjectImpl.setSyncEventInfo(this, name);
                            data = storageObjectImpl;
                            break;
                        } catch (Exception e) {
                            //If SCS object in inaccessable, log it and create ExternalObject data instead.
                            getLogger().log(Level.WARNING, "Storage CS object access failed: " + e.getMessage());
                        }
                    }
                    data = new oracle.iot.client.ExternalObject((String)jsonValue);
                } catch (Exception e) {
                    throw new IllegalArgumentException("Cannot get value for attribute" + name, e);
                }
                break;

            default:
                throw new IllegalArgumentException("unexpected type '" +
                    attributeType + "'");
        }

        return data;
    }

    private static class ErrorCallbackBridge implements MessageDispatcher.ErrorCallback {

        private final Map<String, WeakReference<VirtualDeviceImpl>> deviceSet =
                new HashMap<String, WeakReference<VirtualDeviceImpl>> ();

        void add(VirtualDeviceImpl virtualDevice) {

            if (deviceSet.size() > 0) {
                // TODO: this will be a performance problem if the map is very large
                final Iterator<Map.Entry<String,WeakReference<VirtualDeviceImpl>>> iterator =
                        deviceSet.entrySet().iterator();
                while(iterator.hasNext()) {
                    final Map.Entry<String,WeakReference<VirtualDeviceImpl>> entry =
                            iterator.next();
                    final WeakReference<VirtualDeviceImpl> ref = entry.getValue();
                    if (ref.get() == null) iterator.remove();
                }
            }

            assert virtualDevice.getEndpointId() != null;
            deviceSet.put(
                    virtualDevice.getEndpointId(),
                    new WeakReference<VirtualDeviceImpl>(virtualDevice)
            );
        }

        @Override
        public void failed(List<Message> messages, Exception exception) {

            final Map<String,List<Message>> collatedMessages =
                    new HashMap<String, List<Message>>(messages.size());

            // Collate the messages by source
            for(Message message : messages) {
                final String source = message.getSource();
                List<Message> list = collatedMessages.get(source);
                if (list == null) {
                    list = new ArrayList<Message>();
                    collatedMessages.put(source, list);
                }
                list.add(message);
            }

            // Call the onError method of
            for(Map.Entry<String,List<Message>> entry : collatedMessages.entrySet()) {

                final String source = entry.getKey();
                final List<Message> list = entry.getValue();

                final WeakReference<VirtualDeviceImpl> ref = deviceSet.get(source);
                if (ref == null) {
                    new Exception("DEBUG source="+source+" deviceSet="+deviceSet).printStackTrace();
                }
                final VirtualDeviceImpl virtualDevice = (ref == null ? null : ref.get());
                if (virtualDevice != null) {
                    invokeErrorCallback(virtualDevice, list, exception);
                }
            }

        }

        private void invokeErrorCallback(VirtualDeviceImpl virtualDevice, List<Message> messages, Exception exception) {

            VirtualDeviceBase.NamedValueImpl<?> values = null;
            for (Message message : messages) {
                List<DataItem<?>> dataItems = null;
                if (message instanceof DataMessage) {
                    DataMessage dataMessage = (DataMessage) message;
                    dataItems = dataMessage.getDataItems();
//                TODO: alert needs error callback
//                } else if (message instanceof AlertMessage) {
//                    AlertMessage alertMessage = (AlertMessage) message;
//                    dataItems = alertMessage.getDataItems();
                }

                if (dataItems != null) {
                    VirtualDeviceBase.NamedValueImpl<?> last = null;
                    for (DataItem<?> dataItem : dataItems) {

                        VirtualDeviceBase.NamedValueImpl<?> namedValue =
                                new VirtualDeviceBase.NamedValueImpl<Object>(
                                        dataItem.getKey(),
                                        dataItem.getValue());
                        if (virtualDevice.attributeMap.containsKey(dataItem.getKey())) {
                            try {
                                VirtualDeviceAttributeBase<VirtualDevice, ?> attribute =
                                        virtualDevice.getAttribute(dataItem.getKey());
                                if (attribute.getOnError() != null) {
                                    attribute.getOnError().onError(
                                            new VirtualDeviceBase.ErrorEvent<VirtualDevice>(
                                                    virtualDevice,
                                                    namedValue,
                                                    exception.getMessage()
                                            )
                                    );
                                }
                            } catch (IllegalArgumentException e) {
                                getLogger().log(Level.FINE, e.getMessage(), e);
                            } catch (Exception e) {
                                // onError could throw an exception
                                getLogger().log(Level.FINE, e.getMessage(), e);
                            }
                        }

                        if (last != null) {
                            last.setNext(namedValue);
                            last = namedValue;
                        } else {
                            values = last = namedValue;
                        }
                    }
                }
            }

            if (virtualDevice.getErrorCallback() != null) {
                try {
                    VirtualDeviceBase.ErrorEvent<VirtualDevice> errorEvent =
                            new VirtualDeviceBase.ErrorEvent<VirtualDevice>(virtualDevice, values, exception.getMessage());
                        virtualDevice.getErrorCallback().onError(errorEvent);
                } catch (Exception e) {
                    // onError could throw an exception
                    getLogger().log(Level.FINE, e.getMessage(), e);
                }
            }
        }
    }


    private void processOnChange(final DataMessage.Builder builder, /*(VirtualDeviceAttributeImpl)*/VirtualDeviceAttributeBase<VirtualDevice, Object>attribute, 
            final Object newValue, List<StorageObjectImpl> storageObjectList) {
        final DeviceModelAttribute deviceModelAttribute =
            attribute.getDeviceModelAttribute();
        
        final String attributeName = deviceModelAttribute.getName();
        builder
            .format(base.getDeviceModel().getURN() + ":attributes")
            .source(base.getEndpointId());
        
        switch (deviceModelAttribute.getType()) {
        case INTEGER:
        case NUMBER:
            builder.dataItem(attributeName, ((Number) newValue).doubleValue());
            break;
        case STRING:
            builder.dataItem(attributeName, (String) newValue);
            break;
        case URI:
            if (newValue instanceof StorageObjectImpl) {
                StorageObjectImpl storageObjectImpl = (StorageObjectImpl)newValue;
                if ((storageObjectImpl.getSyncStatus() == StorageObject.SyncStatus.NOT_IN_SYNC) || 
                        (storageObjectImpl.getSyncStatus() == StorageObject.SyncStatus.SYNC_PENDING)) {
                    storageObjectList.add(storageObjectImpl);
                }
                storageObjectImpl.setSyncEventInfo(this, attributeName);
                storageObjectImpl.sync();
            }
            builder.dataItem(attributeName, ((oracle.iot.client.ExternalObject)newValue).getURI());
            break;
        case BOOLEAN:
            builder.dataItem(attributeName, ((Boolean) newValue).booleanValue());
            break;
        case DATETIME:
            if (newValue instanceof Date) {
                builder.dataItem(attributeName, ((Date) newValue).getTime());
            } else if (newValue instanceof Number) {
                builder.dataItem(attributeName, ((Number) newValue).longValue());
            }
            break;
        default:
            throw new RuntimeException("unknown attribute type " + deviceModelAttribute.getType());
        }
    }

    private static final ExecutorService errorEventDispatcher =
            Executors.newSingleThreadExecutor(new ThreadFactory() {
                @Override
                public Thread newThread(Runnable r) {
                    final SecurityManager s = System.getSecurityManager();
                    final ThreadGroup group = (s != null) ? s.getThreadGroup() :
                            Thread.currentThread().getThreadGroup();
                    Thread t = new Thread(group, r, "errorEventDispatchingThread", 0);

                    // this is opposite of what the Executors.DefaultThreadFactory does
                    if (!t.isDaemon())
                        t.setDaemon(true);
                    if (t.getPriority() != Thread.NORM_PRIORITY)
                        t.setPriority(Thread.NORM_PRIORITY);
                    return t;
                }
            });

    void notifyException(Set<VirtualDeviceAttributeBase<VirtualDevice, Object>> attributes, Exception e) {

        final AbstractVirtualDevice.ErrorCallback errorCallback = this.getErrorCallback();
        if (errorCallback == null) {
            return;
        }

        VirtualDeviceBase.NamedValueImpl<Object> head = null, tail = null;
        for (VirtualDeviceAttributeBase<VirtualDevice, Object> attribute : attributes) {
            String name = attribute.getDeviceModelAttribute().getName();
            Object value = attribute.get();
            VirtualDeviceBase.NamedValueImpl<Object> next = new VirtualDeviceBase.NamedValueImpl(name, value);
            if (head != null) {
                tail.setNext(next);
                tail = next;
            } else {
                head = tail = next;
            }
        }

        final ErrorEvent<VirtualDevice> errorEvent =
                new VirtualDeviceBase.ErrorEvent<VirtualDevice>(this, head, e.getMessage());
        final String msg = e.getMessage();

        errorEventDispatcher.submit(new Runnable() {
            @Override
            public void run() {
                try {
                    errorCallback.onError(errorEvent);
                } catch (Exception ignored) {
                    getLogger().info("onError threw: " + msg);
                }
            }
        });
    }

    private static final Logger LOGGER = Logger.getLogger("oracle.iot.client");
    private static Logger getLogger() { return LOGGER; }   
}
