/*
 * Copyright (c) 2015, 2017, Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.client.device.util;

import com.oracle.iot.client.device.DirectlyConnectedDevice;
import com.oracle.iot.client.impl.device.MessageDispatcherImpl;
import com.oracle.iot.client.message.Message;

import java.io.Closeable;
import java.util.WeakHashMap;
import java.util.List;
import java.util.Map;

/**
 * The MessageDispatcher queues messages for automatic dispatching to the
 * for the Oracle IoT Cloud Service.
 * The MessageDispatcher prioritizes message dispatching to ensure
 * high priority messages are dispatched ahead of lower priority messages.
 * There is one MessageDispatcher instance per client.
 */
public abstract class MessageDispatcher implements Closeable {

    // TODO: this leaks. MessageDispatcherImpl holds DirectlyConnectedDevice. Maybe move all this to impl.
    private final static Map<DirectlyConnectedDevice,MessageDispatcher> dispatcherMap =
        new WeakHashMap<DirectlyConnectedDevice,MessageDispatcher>();

    /**
     * Get the instance of a MessageDispatcher for the Client.
     * @param directlyConnectedDevice is a gateway or directly connected device
     * @return A MessageDispatcher for the Client.
     */
    public static MessageDispatcher getMessageDispatcher(DirectlyConnectedDevice directlyConnectedDevice) {
        MessageDispatcherImpl messageDispatcher = (MessageDispatcherImpl)dispatcherMap.get(directlyConnectedDevice);
        if (messageDispatcher == null || messageDispatcher.isClosed()) {
            messageDispatcher =  new MessageDispatcherImpl(directlyConnectedDevice);
            dispatcherMap.put(directlyConnectedDevice, messageDispatcher);
        }
        return messageDispatcher;
    }

    /**
     * Remove the instance of a MessageDispatcher for the Client.
     * @param directlyConnectedDevice is a gateway or directly connected device
     * @return A MessageDispatcher for the Client being removed from the Map, or
     * {@code null} if there's no MessageDispatcher instance referenced by DCD deviceId in the map.
     */
    static MessageDispatcher remove(DirectlyConnectedDevice directlyConnectedDevice) {
        return dispatcherMap.remove(directlyConnectedDevice);
    }

    /**
     * Get the RequestDispatcher that is used by this MessageDispatcher for
     * handling RequestMessages.
     * @return a RequestDispatcher
     * @see RequestDispatcher
     */
    public abstract RequestDispatcher getRequestDispatcher();

    /**
     * Add the message to the outgoing message queue if it is possible to
     * do so without violating capacity restrictions.
     * @param message The message to be queued
     * @throws ArrayStoreException if the message cannot be added to the queue
     * @throws IllegalArgumentException if the message is null
     */
    public abstract void queue(Message message);

    /**
     * A callback interface for successful delivery of messages.
     */
    public interface DeliveryCallback {
        /**
         * Notify that messages have been successfully dispatched. This callback
         * indicates that the messages have been delivered to, and accepted by,
         * the server and are no longer queued.
         * @param messages the messages that were delivered
         */
        void delivered(List<Message> messages);
    }

    /**
     * A callback interface for errors in delivery of messages.
     */
    public interface ErrorCallback {
        /**
         * Notify that an error occurred when attempting to deliver messages
         * to the server. This callback indicates that the messages have
         * not been delivered to the server and are no longer queued.
         * @param messages the messages that were not delivered
         * @param exception the exception that was raised when attempting
         *                  to deliver the messages.
         */
        void failed(List<Message> messages, Exception exception);
    }

    /**
     * Set a callback to be notified if there is an error in dispatching.
     * @param callback callback to invoke, if {@code null},
     *                 the existing callback will be removed
     */
    public abstract void setOnError(ErrorCallback callback);

    /**
     * Set a callback to be notified if message is successfully delivered.
     * @param callback callback to invoke, if {@code null},
     *                 the existing callback will be removed
     */
    public abstract void setOnDelivery(DeliveryCallback callback);
   
}
