/*
 * Copyright (c) 2015,2016 Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.sample;


/**
 * A simulated humidity sensor for use in the samples. The sensor has
 * humidity and maximum threshold attributes.
 */
public class HumiditySensor {

    // The min range for values
    private final int hmin = 0;

    // The hmin range for values
    private final int hmax = 100;

    // Used to generate fluctuation around the 'set point'
    private float x = 0f;
    private float amplitude;

    // setPoint is the value around which the the humidity value
    // will fluctate. The setPoint is set to 90% of the
    // difference between hmin and hmax;
    private float setPoint;

    private int maxThreshold;

    private String hardwareId;

    public HumiditySensor(String id) {
        hardwareId = id;
        maxThreshold = 80;
        setPoint = maxThreshold * .90f;
        amplitude = maxThreshold - setPoint + 0.5f;
    }

    public synchronized int getHumidity() {
        final double delta = amplitude * Math.sin(Math.toRadians(x));
        x += 30;
        return (int)Math.round(setPoint + delta);
    }

    public synchronized void setMaxThreshold(int threshold) {
        maxThreshold = threshold;
        setPoint = maxThreshold * .90f;
        amplitude = maxThreshold - setPoint + 0.5f;
    }

    public synchronized int getMaxThreshold() {
        return maxThreshold;
    }

    public String getManufacturer() {
        return "Sample";
    }

    public String getModelNumber() {
        return "MN-" + hardwareId;
    }

    public String getSerialNumber() {
        return "SN-" + hardwareId;
    }

    public String getHardwareId() {
        return hardwareId;
    }

}
