/*
 * Copyright (c) 2015,2016 Oracle and/or its affiliates.  All rights reserved.
 *
 * This software is dual-licensed to you under the MIT License (MIT) and 
 * the Universal Permissive License (UPL).  See the LICENSE file in the root
 * directory for license terms.  You may choose either license, or both.
 */

package com.oracle.iot.sample.ext;

import java.util.Date;

public class TemperatureSensor {
    public final static double START_TEMP = 20; 

    // for generating temperature fluctuation as a sine wave
    private int x;
    private double amplitude;

    private boolean power;

    private Date startTime;

    private double minTemp = Double.MAX_VALUE;

    private double maxTemp = Double.MIN_VALUE;

    private double setPoint;

    private int minThreshold = 0;

    private int maxThreshold = 65;

    private String hardwareId;

    public TemperatureSensor(String id) {
        hardwareId = id;
        reset();
        power(true);
    }

    public void power(boolean on) {
        if (on) {
            if (power) {
                return;
            }

            startTime = new Date();
            power = true;
            reset();
            return;
        }

        power = false;
    }

    public boolean isPoweredOn() {
        return power;
    }

    public void reset() {
        minTemp = Double.MAX_VALUE;
        maxTemp = Double.MIN_VALUE;
        setPoint = maxThreshold * .90f;
        amplitude = maxThreshold - setPoint + 1f;
    }

    /**
     * Get the current temperature value.
     * <p>
     * The simulated readings will fluctuate +-2 degrees.
     *
     * @return the temperature value
     */
    public double getTemp() {
        final double delta = amplitude * Math.sin(Math.toRadians(x));
        x += 30;
        // normal temp sensors don't have readings such as 21.890890309842234
        long l = (long)((setPoint + delta) * 100.0);
        double temp = (double)l / 100;

        if (temp < minTemp) {
            minTemp = temp;
        }

        if (temp > maxTemp) {
            maxTemp = temp;
        }

        return temp;
    }

    public String getUnit() {
        return "\u00B0C";
    }

    public double getMinTemp() {
        return minTemp;
    }

    public double getMaxTemp() {
        return maxTemp;
    }

    public int getMinThreshold() {
        return minThreshold;
    }

    public void setMinThreshold(int threshold) {
        minThreshold = threshold;
    }

    public int getMaxThreshold() {
        return maxThreshold;
    }
        
    public void setMaxThreshold(int threshold) {
        maxThreshold = threshold;
        setPoint = maxThreshold * .90f;
        amplitude = maxThreshold - setPoint + 1f;
    }

    public Date getStartTime() {
        return startTime;
    }

    public String getManufacturer() {
        return "Sample";
    }

    public String getModelNumber() {
        return "MN-" + hardwareId;
    }

    public String getSerialNumber() {
        return "SN-" + hardwareId;
    }

    public String getHardwareId() {
        return hardwareId;
    }
}
