/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the "License").  You may not use this file except
 * in compliance with the License.
 *
 * You can obtain a copy of the license at
 * glassfish/bootstrap/legal/CDDLv1.0.txt or
 * https://glassfish.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * HEADER in each file and include the License file at
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable,
 * add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your
 * own identifying information: Portions Copyright [yyyy]
 * [name of copyright owner]
 *
 * Copyright 2005 Sun Microsystems, Inc. All rights reserved.
 */
/*
 * Additions/modifications to this source file by Oracle USA, Inc. 2007, 2008, 2009
 */

// Mobile: renamed package
package oracle.adfnmc.el;

import java.util.Vector;

import oracle.adfnmc.util.Number;
import oracle.adfnmc.util.Utility;


/**
 * Defines property resolution behavior on instances of {@link java.util.List}.
 *
 * <p>
 * This resolver handles base objects of type <code>java.util.List</code>. It accepts any object
 * as a property and coerces that object into an integer index into the list. The resulting value is
 * the value in the list at that index.
 * </p>
 *
 * <p>
 * This resolver can be constructed in read-only mode, which means that {@link #isReadOnly} will
 * always return <code>true</code> and {@link #setValue} will always throw
 * <code>PropertyNotWritableException</code>.
 * </p>
 *
 * <p>
 * <code>ELResolver</code>s are combined together using {@link CompositeELResolver}s, to define
 * rich semantics for evaluating an expression. See the javadocs for {@link ELResolver} for details.
 * </p>
 *
 * @see CompositeELResolver
 * @see ELResolver
 * @see java.util.List
 * @since JSP 2.1
 */
// Mobile: changed signature
//public class ListELResolver extends ELResolver
public class ListELResolver
  implements ELResolver
{
  // Mobile: changed behavior
  // Mobile: unsupported type
  //static private Class<?> theUnmodifiableListClass = Collections.unmodifiableList(new ArrayList()).getClass();
  static private Class theUnmodifiableListClass = null;


  /**
   * Creates a new read/write <code>ListELResolver</code>.
   */
  public ListELResolver()
  {
    this.isReadOnly = false;
  }

  /**
   * Creates a new <code>ListELResolver</code> whose read-only status is determined by the given
   * parameter.
   *
   * @param isReadOnly
   *            <code>true</code> if this resolver cannot modify lists; <code>false</code>
   *            otherwise.
   */
  public ListELResolver(boolean isReadOnly)
  {
    this.isReadOnly = isReadOnly;
  }

  /**
   * If the base object is a list, returns the most general acceptable type for a value in this
   * list.
   *
   * <p>
   * If the base is a <code>List</code>, the <code>propertyResolved</code> property of the
   * <code>ELContext</code> object must be set to <code>true</code> by this resolver, before
   * returning. If this property is not <code>true</code> after this method is called, the
   * caller should ignore the return value.
   * </p>
   *
   * <p>
   * Assuming the base is a <code>List</code>, this method will always return
   * <code>Object.class</code>. This is because <code>List</code>s accept any object as an
   * element.
   * </p>
   *
   * @param context
   *            The context of this evaluation.
   * @param base
   *            The list to analyze. Only bases of type <code>List</code> are handled by this
   *            resolver.
   * @param property
   *            The index of the element in the list to return the acceptable type for. Will be
   *            coerced into an integer, but otherwise ignored by this resolver.
   * @return If the <code>propertyResolved</code> property of <code>ELContext</code> was set
   *         to <code>true</code>, then the most general acceptable type; otherwise undefined.
   * @throws PropertyNotFoundException
   *             if the given index is out of bounds for this list.
   * @throws NullPointerException
   *             if context is <code>null</code>
   * @throws ELException
   *             if an exception was thrown while performing the property or variable resolution.
   *             The thrown exception must be included as the cause property of this exception, if
   *             available.
   */
  public Class getType(ELContext context, Object base, Object property)
  {
    if (context == null)
    {
      throw new NullPointerException();
    }

    if (base instanceof Vector)
    {
      context.setPropertyResolved(true);
      Vector list = (Vector) base;
      int index = toInteger(property);
      if (index < 0 || index >= list.size())
      {
        throw new PropertyNotFoundException();
      }
      return Utility.OBJECT_CLASS;
    }

    return null;
  }

  /**
   * If the base object is a list, returns the value at the given index. The index is specified by
   * the <code>property</code> argument, and coerced into an integer. If the coercion could not
   * be performed, an <code>IllegalArgumentException</code> is thrown. If the index is out of
   * bounds, <code>null</code> is returned.
   *
   * <p>
   * If the base is a <code>List</code>, the <code>propertyResolved</code> property of the
   * <code>ELContext</code> object must be set to <code>true</code> by this resolver, before
   * returning. If this property is not <code>true</code> after this method is called, the
   * caller should ignore the return value.
   * </p>
   *
   * @param context
   *            The context of this evaluation.
   * @param base
   *            The list to be analyzed. Only bases of type <code>List</code> are handled by
   *            this resolver.
   * @param property
   *            The index of the value to be returned. Will be coerced into an integer.
   * @return If the <code>propertyResolved</code> property of <code>ELContext</code> was set
   *         to <code>true</code>, then the value at the given index or <code>null</code> if
   *         the index was out of bounds. Otherwise, undefined.
   * @throws IllegalArgumentException
   *             if the property could not be coerced into an integer.
   * @throws NullPointerException
   *             if context is <code>null</code>.
   * @throws ELException
   *             if an exception was thrown while performing the property or variable resolution.
   *             The thrown exception must be included as the cause property of this exception, if
   *             available.
   */
  public Object getValue(ELContext context, Object base, Object property)
  {
    if (context == null)
    {
      throw new NullPointerException();
    }

    if (base instanceof Vector)
    {
      context.setPropertyResolved(true);
      Vector list = (Vector) base;
      int index = toInteger(property);
      if (index < 0 || index >= list.size())
      {
        return null;
      }
      return list.elementAt(index);
    }

    return null;
  }

  /**
   * If the base object is a list, attempts to set the value at the given index with the given
   * value. The index is specified by the <code>property</code> argument, and coerced into an
   * integer. If the coercion could not be performed, an <code>IllegalArgumentException</code>
   * is thrown. If the index is out of bounds, a <code>PropertyNotFoundException</code> is
   * thrown.
   *
   * <p>
   * If the base is a <code>List</code>, the <code>propertyResolved</code> property of the
   * <code>ELContext</code> object must be set to <code>true</code> by this resolver, before
   * returning. If this property is not <code>true</code> after this method is called, the
   * caller can safely assume no value was set.
   * </p>
   *
   * <p>
   * If this resolver was constructed in read-only mode, this method will always throw
   * <code>PropertyNotWritableException</code>.
   * </p>
   *
   * <p>
   * If a <code>List</code> was created using {@link java.util.Collections#unmodifiableList},
   * this method must throw <code>PropertyNotWritableException</code>. Unfortunately, there is
   * no Collections API method to detect this. However, an implementation can create a prototype
   * unmodifiable <code>List</code> and query its runtime type to see if it matches the runtime
   * type of the base object as a workaround.
   * </p>
   *
   * @param context
   *            The context of this evaluation.
   * @param base
   *            The list to be modified. Only bases of type <code>List</code> are handled by
   *            this resolver.
   * @param property
   *            The index of the value to be set. Will be coerced into an integer.
   * @param val
   *            The value to be set at the given index.
   * @throws ClassCastException
   *             if the class of the specified element prevents it from being added to this list.
   * @throws NullPointerException
   *             if context is <code>null</code>, or if the value is <code>null</code> and
   *             this <code>List</code> does not support <code>null</code> elements.
   * @throws IllegalArgumentException
   *             if the property could not be coerced into an integer, or if some aspect of the
   *             specified element prevents it from being added to this list.
   * @throws PropertyNotWritableException
   *             if this resolver was constructed in read-only mode, or if the set operation is
   *             not supported by the underlying list.
   * @throws PropertyNotFoundException
   *             if the given index is out of bounds for this list.
   * @throws ELException
   *             if an exception was thrown while performing the property or variable resolution.
   *             The thrown exception must be included as the cause property of this exception, if
   *             available.
   */
  public void setValue(ELContext context, Object base, Object property, Object val)
  {
    if (context == null)
    {
      throw new NullPointerException();
    }

    if (base instanceof Vector)
    {
      context.setPropertyResolved(true);
      Vector list = (Vector) base;
      int index = toInteger(property);
      if (isReadOnly)
      {
        throw new PropertyNotWritableException();
      }
      try
      {
        list.setElementAt(val, index);
      }
      catch (IndexOutOfBoundsException ex)
      {
        throw new PropertyNotFoundException();
      }
      catch (ClassCastException ex)
      {
        throw ex;
      }
      catch (NullPointerException ex)
      {
        throw ex;
      }
      catch (IllegalArgumentException ex)
      {
        throw ex;
      }
    }
  }

  /**
   * If the base object is a list, returns whether a call to {@link #setValue} will always fail.
   *
   * <p>
   * If the base is a <code>List</code>, the <code>propertyResolved</code> property of the
   * <code>ELContext</code> object must be set to <code>true</code> by this resolver, before
   * returning. If this property is not <code>true</code> after this method is called, the
   * caller should ignore the return value.
   * </p>
   *
   * <p>
   * If this resolver was constructed in read-only mode, this method will always return
   * <code>true</code>.
   * </p>
   *
   * <p>
   * If a <code>List</code> was created using {@link java.util.Collections#unmodifiableList},
   * this method must return <code>true</code>. Unfortunately, there is no Collections API
   * method to detect this. However, an implementation can create a prototype unmodifiable
   * <code>List</code> and query its runtime type to see if it matches the runtime type of the
   * base object as a workaround.
   * </p>
   *
   * @param context
   *            The context of this evaluation.
   * @param base
   *            The list to analyze. Only bases of type <code>List</code> are handled by this
   *            resolver.
   * @param property
   *            The index of the element in the list to return the acceptable type for. Will be
   *            coerced into an integer, but otherwise ignored by this resolver.
   * @return If the <code>propertyResolved</code> property of <code>ELContext</code> was set
   *         to <code>true</code>, then <code>true</code> if calling the
   *         <code>setValue</code> method will always fail or <code>false</code> if it is
   *         possible that such a call may succeed; otherwise undefined.
   * @throws PropertyNotFoundException
   *             if the given index is out of bounds for this list.
   * @throws NullPointerException
   *             if context is <code>null</code>
   * @throws ELException
   *             if an exception was thrown while performing the property or variable resolution.
   *             The thrown exception must be included as the cause property of this exception, if
   *             available.
   */
  public boolean isReadOnly(ELContext context, Object base, Object property)
  {
    if (context == null)
    {
      throw new NullPointerException();
    }

    if (base instanceof Vector)
    {
      context.setPropertyResolved(true);
      Vector list = (Vector) base;
      int index = toInteger(property);
      if (index < 0 || index >= list.size())
      {
        throw new PropertyNotFoundException();
      }
      return list.getClass() == theUnmodifiableListClass || isReadOnly;
    }

    return false;
  }

  // Mobile: removed behavior
  // Mobile: unsupported language feature
  ///**
  // * Always returns <code>null</code>, since there is no reason to iterate through set set of
  // * all integers.
  // *
  // * <p>
  // * The {@link #getCommonPropertyType} method returns sufficient information about what
  // * properties this resolver accepts.
  // * </p>
  // *
  // * @param context
  // *            The context of this evaluation.
  // * @param base
  // *            The list. Only bases of type <code>List</code> are handled by this resolver.
  // * @return <code>null</code>.
  // */
  //public Iterator<FeatureDescriptor> getFeatureDescriptors(ELContext context, Object base)
  //{
  //  return null;
  //}

  // Mobile: removed behavior
  // Mobile: unsupported language feature
  ///**
  // * If the base object is a list, returns the most general type that this resolver accepts for
  // * the <code>property</code> argument. Otherwise, returns <code>null</code>.
  // *
  // * <p>
  // * Assuming the base is a <code>List</code>, this method will always return
  // * <code>Integer.class</code>. This is because <code>List</code>s accept integers as their
  // * index.
  // * </p>
  // *
  // * @param context
  // *            The context of this evaluation.
  // * @param base
  // *            The list to analyze. Only bases of type <code>List</code> are handled by this
  // *            resolver.
  // * @return <code>null</code> if base is not a <code>List</code>; otherwise
  // *         <code>Integer.class</code>.
  // */
  //public Class<?> getCommonPropertyType(ELContext context, Object base)
  //{
  //  if (base != null && base instanceof List)
  //  {
  //    return Integer.class;
  //  }
  //  return null;
  //}

  private int toInteger(Object p)
  {
    if (p instanceof Integer)
    {
      return ((Integer) p).intValue();
    }
    if (p instanceof Character)
    {
      return ((Character) p).charValue();
    }
    if (p instanceof Boolean)
    {
      return ((Boolean) p).booleanValue() ? 1: 0;
    }
    // Mobile: substituted type
    //if (p instanceof Number)
    //{
    //  return ((Number) p).intValue();
    //}
    if (Number.isObjectNumeric(p))
      return ((Number) p).intValue();
    if (p instanceof String)
    {
      return Integer.parseInt((String) p);
    }
    throw new IllegalArgumentException();
  }

  private boolean isReadOnly;
}
