/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Additions/modifications to this source file by Oracle USA, Inc. 2007, 2008, 2009
 */
package de.odysseus.el.tree.impl.ast;


import de.odysseus.el.misc.LocalMessages;
import de.odysseus.el.misc.TypeConversions;
import de.odysseus.el.tree.Bindings;
import de.odysseus.el.tree.Node;

import oracle.adfnmc.el.ELContext;
import oracle.adfnmc.el.ELException;
import oracle.adfnmc.el.MethodInfo;
import oracle.adfnmc.el.PropertyNotFoundException;
import oracle.adfnmc.util.MethodDispatch;


public abstract class AstProperty
  extends AstNode
{
  private final AstNode prefix;

  private final boolean lvalue;

  private final boolean strict; // allow null as property value?

  public AstProperty(AstNode prefix, boolean lvalue, boolean strict)
  {
    this.prefix = prefix;
    this.lvalue = lvalue;
    this.strict = strict;
  }

  protected abstract Object getProperty(Bindings bindings, ELContext context)
    throws ELException;

  public final Object eval(Bindings bindings, ELContext context)
  {
    Object base = prefix.eval(bindings, context);
    if (base == null)
    {
      return null;
    }
    Object property = getProperty(bindings, context);
    if (property == null && strict)
    {
      return null;
    }
    Object result = context.getELResolver().getValue(context, base, property);
    if (!context.isPropertyResolved())
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.property.notfound", property, base));
    }
    return result;
  }

  public final boolean isLiteralText()
  {
    return false;
  }

  public final boolean isLeftValue()
  {
    return lvalue;
  }

  public final Class getType(Bindings bindings, ELContext context)
  {
    if (!lvalue)
    {
      return null;
    }
    Object base = prefix.eval(bindings, context);
    if (base == null)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.base.null", prefix));
    }
    Object property = getProperty(bindings, context);
    if (property == null && strict)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.property.notfound", "null", base));
    }
    Class result = context.getELResolver().getType(context, base, property);
    if (!context.isPropertyResolved())
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.property.notfound", property, base));
    }
    return result;
  }

  public final boolean isReadOnly(Bindings bindings, ELContext context)
    throws ELException
  {
    if (!lvalue)
    {
      return true;
    }
    Object base = prefix.eval(bindings, context);
    if (base == null)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.base.null", prefix));
    }
    Object property = getProperty(bindings, context);
    if (property == null && strict)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.property.notfound", "null", base));
    }
    boolean result = context.getELResolver().isReadOnly(context, base, property);
    if (!context.isPropertyResolved())
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.property.notfound", property, base));
    }
    return result;
  }

  public final void setValue(Bindings bindings, ELContext context, Object value)
    throws ELException
  {
    if (!lvalue)
    {
      throw new ELException(LocalMessages.get("error.value.set.rvalue"));
    }
    Object base = prefix.eval(bindings, context);
    if (base == null)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.base.null", prefix));
    }
    Object property = getProperty(bindings, context);
    if (property == null && strict)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.property.notfound", "null", base));
    }
    context.getELResolver().setValue(context, base, property, value);
    if (!context.isPropertyResolved())
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.property.notfound", property, base));
    }
  }

  // Mobile: removed behavior
  // Mobile: unsupported type dependency
  //private Method findMethod(String name, Class clazz, Class returnType, Class[] paramTypes)
  //{
  //  try
  //  {
  //    method = clazz.getMethod(name, paramTypes);
  //  }
  //  catch (NoSuchMethodException e)
  //  {
  //    throw new MethodNotFoundException(LocalMessages.get("error.property.method.notfound", name, clazz));
  //  }
  //  if (returnType != null && !returnType.isAssignableFrom(method.getReturnType()))
  //  {
  //    throw new MethodNotFoundException(LocalMessages.get("error.property.method.notfound", name, clazz));
  //  }
  //
  //  return method;
  //}

  /**
   *
   * @param bindings
   * @param context
   * @param returnType
   * @param paramTypes
   * @return
   */
  public final MethodInfo getMethodInfo(Bindings bindings, ELContext context, Class returnType, Class[] paramTypes)
  {
    Object base = prefix.eval(bindings, context);
    if (base == null)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.base.null", prefix));
    }
    Object property = getProperty(bindings, context);
    if (property == null && strict)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.method.notfound", "null", base));
    }
    String name = TypeConversions.coerceToString(property);

    // Mobile: changed behavior
    // Mobile: unsupported type dependency
    // Mobile: unsupported language feature (reflection)
    //    
    //Method method = findMethod(name, base.getClass(), returnType, paramTypes);
    //return new MethodInfo(method.getName(), method.getReturnType(), paramTypes);

    // Until we add a general interface to inspect a method's signature, 
    // we can really only loop back the returnType and paramTypes passed in
    return new MethodInfo(name, returnType, paramTypes);
  }

  public final Object invoke(Bindings bindings, ELContext context, Class returnType, Class[] paramTypes,
                             Object[] paramValues)
  {
    Object base = prefix.eval(bindings, context);
    if (base == null)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.base.null", prefix));
    }
    Object property = getProperty(bindings, context);
    if (property == null && strict)
    {
      throw new PropertyNotFoundException(LocalMessages.get("error.property.method.notfound", "null", base));
    }
    String name = TypeConversions.coerceToString(property);
    
    // Mobile: changed behavior
    // Mobile: unsupported type dependency
    // Mobile: unsupported language feature (reflection)
    //
    //Method method = findMethod(name, base.getClass(), returnType, paramTypes);
    //
    //try
    //{
    //  return method.invoke(base, paramValues);
    //}
    //catch (IllegalAccessException e)
    //{
    //  throw new ELException(LocalMessages.get("error.property.method.access", name, base.getClass()));
    //}
    //catch (IllegalArgumentException e)
    //{
    //  throw new ELException(LocalMessages.get("error.property.method.invocation", name, base.getClass()), e);
    //}
    //catch (InvocationTargetException e)
    //{
    //  throw new ELException(LocalMessages.get("error.property.method.invocation", name, base.getClass()),
    //                        e.getCause());
    //}

    if (base instanceof MethodDispatch)
    {
      MethodDispatch md = (MethodDispatch)base;
      return md.invokeMethod(name, paramValues);
    }

    return null;
  }

  /**
   *
   * @param i
   * @return
   */
  // Mobile: changed signature
  //public AstNode getChild(int i)
  public Node getChild(int i)
  {
    return i == 0 ? prefix: null;
  }
}
