/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Additions/modifications to this source file by Oracle USA, Inc. 2007, 2008, 2009
 */
package de.odysseus.el.tree.impl.ast;


import de.odysseus.el.misc.BooleanOperations;
import de.odysseus.el.misc.NumberOperations;
import de.odysseus.el.tree.Bindings;
import de.odysseus.el.tree.Node;

import oracle.adfnmc.el.ELContext;
import oracle.adfnmc.el.event.generic.ValueChangeListener;


public final class AstBinary
  extends AstRightValue
{
  public interface Operator
  {
    public Object apply(Object o1, Object o2);
  }

  public static final Operator ADD = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return NumberOperations.add(o1, o2);
    }

    public String toString()
    {
      return "+";
    }
  };
  public static final Operator AND = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.and(o1, o2));
    }

    public String toString()
    {
      return "&&";
    }
  };
  public static final Operator DIV = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return NumberOperations.div(o1, o2);
    }

    public String toString()
    {
      return "/";
    }
  };
  public static final Operator EQ = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.eq(o1, o2));
    }

    public String toString()
    {
      return "==";
    }
  };
  public static final Operator GE = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.ge(o1, o2));
    }

    public String toString()
    {
      return ">=";
    }
  };
  public static final Operator GT = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.gt(o1, o2));
    }

    public String toString()
    {
      return ">";
    }
  };
  public static final Operator LE = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.le(o1, o2));
    }

    public String toString()
    {
      return "<=";
    }
  };
  public static final Operator LT = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.lt(o1, o2));
    }

    public String toString()
    {
      return "<";
    }
  };
  public static final Operator MOD = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return NumberOperations.mod(o1, o2);
    }

    public String toString()
    {
      return "%";
    }
  };
  public static final Operator MUL = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return NumberOperations.mul(o1, o2);
    }

    public String toString()
    {
      return "*";
    }
  };
  public static final Operator NE = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.ne(o1, o2));
    }

    public String toString()
    {
      return "!=";
    }
  };
  public static final Operator OR = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return BooleanOperations.box(BooleanOperations.or(o1, o2));
    }

    public String toString()
    {
      return "||";
    }
  };
  public static final Operator SUB = new Operator()
  {
    public Object apply(Object o1, Object o2)
    {
      return NumberOperations.sub(o1, o2);
    }

    public String toString()
    {
      return "-";
    }
  };

  private final Operator operator;
  private final AstNode left, right;

  public AstBinary(AstNode left, AstNode right, Operator operator)
  {
    this.left = left;
    this.right = right;
    this.operator = operator;
  }

  public Operator getOperator()
  {
    return operator;
  }

  public Object eval(Bindings bindings, ELContext context)
  {
    return operator.apply(left.eval(bindings, context), right.eval(bindings, context));
  }

  public String toString()
  {
    return "'" + operator.toString() + "'";
  }

  public void appendStructure(StringBuffer b, Bindings bindings)
  {
    left.appendStructure(b, bindings);
    b.append(' ');
    b.append(operator);
    b.append(' ');
    right.appendStructure(b, bindings);
  }

  public int getCardinality()
  {
    return 2;
  }

  /**
   *
   * @param i
   * @return
   */
  // Mobile: changed signature
  //public AstNode getChild(int i)
  public Node getChild(int i)
  {
    return i == 0 ? left: i == 1 ? right: null;
  }

  /**
   *
   * @param bindings
   * @param context
   * @param listener
   */
  // Mobile: added behavior
  public void addValueChangeListener(Bindings bindings, ELContext context, ValueChangeListener listener)
  {
    this.left.addValueChangeListener(bindings, context, listener);
    this.right.addValueChangeListener(bindings, context, listener);
  }

  /**
   *
   * @param bindings
   * @param context
   * @param listener
   */
  // Mobile: added behavior
  public void removeValueChangeListener(Bindings bindings, ELContext context, ValueChangeListener listener)
  {
    this.left.removeValueChangeListener(bindings, context, listener);
    this.right.removeValueChangeListener(bindings, context, listener);
  }
}
