/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Additions/modifications to this source file by Oracle USA, Inc. 2007, 2008, 2009
 */
package de.odysseus.el.tree.impl;


import de.odysseus.el.misc.LocalMessages;

import java.util.Hashtable;

import oracle.adfnmc.java.lang.CharacterHelper;

/**
 * Handcrafted scanner.
 *
 * @author Christoph Beck
 */
final class Scanner
{
  /**
   * Scan exception type
   */
  // Mobile: unsupported language feature
  //@SuppressWarnings("serial")
  static class ScanException
    extends Exception
  {
    ScanException(int position, String encountered, Object expected)
    {
      // Mobile: unsupported language feature - no autoboxing, have to do it explicitly
      //super(LocalMessages.get("error.scan", position, encountered, expected));
      super(LocalMessages.get("error.scan", new Integer(position), encountered, expected));
    }
  }

  /**
   * Symbol type
   */
  // Mobile: unsupported language feature
  // Mobile: substituted type
  //static enum Symbol
  //{
  //  EOF,
  //  PLUS("'+'"),
  //  MINUS("'-'"),
  //  MUL("'*'"),
  //  DIV("'/'|'div'"),
  //  MOD("'%'|'mod'"),
  //  LPAREN("'('"),
  //  RPAREN("')'"),
  //  IDENTIFIER,
  //  NOT("'!'|'not'"),
  //  AND("'&&'|'and'"),
  //  OR("'||'|'or'"),
  //  EMPTY("'empty'"),
  //  INSTANCEOF("'instanceof'"),
  //  INTEGER,
  //  FLOAT,
  //  TRUE("'true'"),
  //  FALSE("'false'"),
  //  STRING,
  //  NULL("'null'"),
  //  LE("'<='|'le'"),
  //  LT("'<'|'lt'"),
  //  GE("'>='|'ge'"),
  //  GT("'>'|'gt'"),
  //  EQ("'=='|'eq'"),
  //  NE("'!='|'ne'"),
  //  QUESTION("'?'"),
  //  COLON("':'"),
  //  TEXT,
  //  DOT("'.'"),
  //  LBRACK("'['"),
  //  RBRACK("']'"),
  //  COMMA("','"),
  //  START_EVAL_DEFERRED("'#{'"),
  //  START_EVAL_DYNAMIC("'${'"),
  //  END_EVAL("'}'");
  //
  //  private final String string;
  //
  //  private Symbol()
  //  {
  //    this(null);
  //  }
  //
  //  private Symbol(String string)
  //  {
  //    this.string = string;
  //  }
  //
  //  public String toString()
  //  {
  //    return string == null ? "<" + name() + ">": string;
  //  }
  //}
  static class Symbol
  {
    static final int EOF = 0;
    static final int PLUS = 1;
    static final int MINUS = 2;
    static final int MUL = 3;
    static final int DIV = 4;
    static final int MOD = 5;
    static final int LPAREN = 6;
    static final int RPAREN = 7;
    static final int IDENTIFIER = 8;
    static final int NOT = 9;
    static final int AND = 10;
    static final int OR = 11;
    static final int EMPTY = 12;
    static final int INSTANCEOF = 13;
    static final int INTEGER = 14;
    static final int FLOAT = 15;
    static final int TRUE = 16;
    static final int FALSE = 17;
    static final int STRING = 18;
    static final int NULL = 19;
    static final int LE = 20;
    static final int LT = 21;
    static final int GE = 22;
    static final int GT = 23;
    static final int EQ = 24;
    static final int NE = 25;
    static final int QUESTION = 26;
    static final int COLON = 27;
    static final int TEXT = 28;
    static final int DOT = 29;
    static final int LBRACK = 30;
    static final int RBRACK = 31;
    static final int COMMA = 32;
    static final int START_EVAL_DEFERRED = 33;
    static final int START_EVAL_DYNAMIC = 34;
    static final int END_EVAL = 35;

    private int _code;

    public Symbol(int code)
    {
      this._code = code;
    }

    public String toString()
    {
      switch (_code)
      {
        case EOF:
          break;

        case PLUS:
          return "+";

          //TODO flesh this out as needed...
      }
      return "";
    }
  }

  // Mobile: substituted type
  //private static final HashMap<String, Symbol> KEYMAP = new HashMap<String, Symbol>(16);
  private static final Hashtable KEYMAP = new Hashtable();

  static
  {
    // Mobile: unsupported language feature (no autoboxing)
    //KEYMAP.put("null", Symbol.NULL);
    //KEYMAP.put("true", Symbol.TRUE);
    //KEYMAP.put("false", Symbol.FALSE);
    //KEYMAP.put("empty", Symbol.EMPTY);
    //KEYMAP.put("div", Symbol.DIV);
    //KEYMAP.put("mod", Symbol.MOD);
    //KEYMAP.put("not", Symbol.NOT);
    //KEYMAP.put("and", Symbol.AND);
    //KEYMAP.put("or", Symbol.OR);
    //KEYMAP.put("le", Symbol.LE);
    //KEYMAP.put("lt", Symbol.LT);
    //KEYMAP.put("eq", Symbol.EQ);
    //KEYMAP.put("ne", Symbol.NE);
    //KEYMAP.put("ge", Symbol.GE);
    //KEYMAP.put("gt", Symbol.GT);
    //KEYMAP.put("instanceof", Symbol.INSTANCEOF);
    KEYMAP.put("null", new Integer(Symbol.NULL));
    KEYMAP.put("true", new Integer(Symbol.TRUE));
    KEYMAP.put("false", new Integer(Symbol.FALSE));
    KEYMAP.put("empty", new Integer(Symbol.EMPTY));
    KEYMAP.put("div", new Integer(Symbol.DIV));
    KEYMAP.put("mod", new Integer(Symbol.MOD));
    KEYMAP.put("not", new Integer(Symbol.NOT));
    KEYMAP.put("and", new Integer(Symbol.AND));
    KEYMAP.put("or", new Integer(Symbol.OR));
    KEYMAP.put("le", new Integer(Symbol.LE));
    KEYMAP.put("lt", new Integer(Symbol.LT));
    KEYMAP.put("eq", new Integer(Symbol.EQ));
    KEYMAP.put("ne", new Integer(Symbol.NE));
    KEYMAP.put("ge", new Integer(Symbol.GE));
    KEYMAP.put("gt", new Integer(Symbol.GT));
    KEYMAP.put("instanceof", new Integer(Symbol.INSTANCEOF));
  }

  private String image = null; // original image of the current token
  private String value = null; // unescaped string for text/string token
  private int position = 0; // start position of current token
  private boolean eval = false; // inside eval expression flag

  private final String input;
  private final StringBuffer builder = new StringBuffer();


  /**
   * Constructor.
   *
   * @param input
   *            expression string
   */
  Scanner(String input)
  {
    this.input = input;
  }

  /**
   * @return <code>true</code> iff the specified character is a digit
   */
  private boolean isDigit(char c)
  {
    return c >= '0' && c <= '9';
  }

  /**
   * text token
   */
  // Mobile: changed signature
  // Mobile: substituted type
  //private Symbol nextText() throws ScanException {
  private int nextText()
    throws ScanException
  {
    builder.setLength(0);
    int i = position;
    int l = input.length();
    boolean escaped = false;
    while (i < l)
    {
      char c = input.charAt(i);
      switch (c)
      {
        case '\\':
          if (escaped)
          {
            builder.append('\\');
          }
          else
          {
            escaped = true;
          }
          break;
        case '#':
        case '$':
          if (i + 1 < l && input.charAt(i + 1) == '{')
          {
            if (escaped)
            {
              builder.append(c);
            }
            else
            {
              value = builder.toString();
              image = input.substring(position, i);
              return Symbol.TEXT;
            }
          }
          else
          {
            if (escaped)
            {
              builder.append('\\');
            }
            builder.append(c);
          }
          escaped = false;
          break;
        default:
          if (escaped)
          {
            builder.append('\\');
          }
          builder.append(c);
          escaped = false;
      }
      i++;
    }
    if (escaped)
    {
      builder.append('\\');
    }
    value = builder.toString();
    image = input.substring(position, i);
    return Symbol.TEXT;
  }

  /**
   * string token
   */
  // Mobile: changed signature
  // Mobile: substituted type
  //private Symbol nextString() throws ScanException {
  private int nextString()
    throws ScanException
  {
    builder.setLength(0);
    char quote = input.charAt(position);
    int i = position + 1;
    int l = input.length();
    while (i < l)
    {
      char c = input.charAt(i++);
      if (c == '\\')
      {
        if (i == l)
        {
          throw new ScanException(position, "unterminated string", quote + " or \\");
        }
        else
        {
          c = input.charAt(i++);
          if (c == '\\' || c == quote)
          {
            builder.append(c);
          }
          else
          {
            throw new ScanException(position, "invalid escape sequence \\" + c, "\\" + quote + " or \\\\");
          }
        }
      }
      else if (c == quote)
      {
        value = builder.toString();
        image = input.substring(position, i);
        return Symbol.STRING;
      }
      else
      {
        builder.append(c);
      }
    }
    // Mobile: unsupported language feature (no autoboxing)
    //throw new ScanException(position, "unterminated string", quote);
    throw new ScanException(position, "unterminated string", new Character(quote));
  }

  /**
   * number token
   */
  // Mobile: changed signature
  // Mobile: substituted type
  //private Symbol nextNumber() throws ScanException {
  private int nextNumber()
    throws ScanException
  {
    int i = position;
    int l = input.length();
    while (i < l && isDigit(input.charAt(i)))
    {
      i++;
    }

    // Mobile: substituted type
    //Symbol symbol = Symbol.INTEGER;
    int symbol = Symbol.INTEGER;
    if (i < l && input.charAt(i) == '.')
    {
      i++;
      while (i < l && isDigit(input.charAt(i)))
      {
        i++;
      }
      symbol = Symbol.FLOAT;
    }
    if (i < l && (input.charAt(i) == 'e' || input.charAt(i) == 'E'))
    {
      int e = i;
      i++;
      if (i < l && (input.charAt(i) == '+' || input.charAt(i) == '-'))
      {
        i++;
      }
      if (i < l && isDigit(input.charAt(i)))
      {
        i++;
        while (i < l && isDigit(input.charAt(i)))
        {
          i++;
        }
        symbol = Symbol.FLOAT;
      }
      else
      {
        i = e;
      }
    }
    image = input.substring(position, i);
    return symbol;
  }

  /**
   * token inside an eval expression
   */
  // Mobile: changed signature
  // Mobile: substituted type
  //private Symbol nextEval() throws ScanException {
  private int nextEval()
    throws ScanException
  {
    char c = input.charAt(position);
    int p1 = position + 1;
    switch (c)
    {
      case '*':
        image = "*";
        return Symbol.MUL;
      case '/':
        image = "/";
        return Symbol.DIV;
      case '%':
        image = "%";
        return Symbol.MOD;
      case '+':
        image = "+";
        return Symbol.PLUS;
      case '-':
        image = "-";
        return Symbol.MINUS;
      case '?':
        image = "?";
        return Symbol.QUESTION;
      case ':':
        image = ":";
        return Symbol.COLON;
      case '[':
        image = "[";
        return Symbol.LBRACK;
      case ']':
        image = "]";
        return Symbol.RBRACK;
      case '(':
        image = "(";
        return Symbol.LPAREN;
      case ')':
        image = ")";
        return Symbol.RPAREN;
      case ',':
        image = ",";
        return Symbol.COMMA;
      case '.':
        if (p1 == input.length() || !isDigit(input.charAt(p1)))
        {
          image = ".";
          return Symbol.DOT;
        }
        break;
      case '=':
        if (p1 < input.length() && input.charAt(p1) == '=')
        {
          image = "==";
          return Symbol.EQ;
        }
        break;
      case '&':
        if (p1 < input.length() && input.charAt(p1) == '&')
        {
          image = "&&";
          return Symbol.AND;
        }
        break;
      case '|':
        if (p1 < input.length() && input.charAt(p1) == '|')
        {
          image = "||";
          return Symbol.OR;
        }
        break;
      case '!':
        if (p1 < input.length() && input.charAt(p1) == '=')
        {
          image = "!=";
          return Symbol.NE;
        }
        image = "!";
        return Symbol.NOT;
      case '<':
        if (p1 < input.length() && input.charAt(p1) == '=')
        {
          image = "<=";
          return Symbol.LE;
        }
        image = "<";
        return Symbol.LT;
      case '>':
        if (p1 < input.length() && input.charAt(p1) == '=')
        {
          image = "<=";
          return Symbol.GE;
        }
        image = ">";
        return Symbol.GT;
      case '"':
      case '\'':
        return nextString();
    }

    if (isDigit(c) || c == '.')
    {
      return nextNumber();
    }

    // Mobile: substituted type	
    //if (Character.isJavaIdentifierStart(c)) {
    if (CharacterHelper.isJavaIdentifierStart(c))
    {
      int i = p1;
      int l = input.length();

      // Mobile: substituted type
      //while (i < l && Character.isJavaIdentifierPart(input.charAt(i))) {
      while (i < l && CharacterHelper.isJavaIdentifierPart(input.charAt(i)))
      {
        i++;
      }
      image = input.substring(position, i);

      // Mobile: substituted type
      //return KEYMAP.containsKey(image) ? KEYMAP.get(image) : Symbol.IDENTIFIER;
      return KEYMAP.containsKey(image) ? ((Integer) KEYMAP.get(image)).intValue(): Symbol.IDENTIFIER;
    }

    throw new ScanException(position, "invalid character '" + c + "'", "expression token");
  }

  /**
   * expose current token image. If the current token is of type text or string, this method
   * returns the unescaped value instead of the original image.
   */
  String getImage()
  {
    return value == null ? image: value;
  }

  /**
   * current position
   */
  int getPosition()
  {
    return position;
  }

  /**
   * Scan next token. After calling this method, {@link #getImage()} and {@link #getPosition()}
   * can be used to retreive the token's image and input position.
   *
   * @return symbol describing the token type.
   */
  // Mobile: changed signature
  // Mobile: substituted type
  //Symbol next() throws ScanException {
  int next()
    throws ScanException
  {
    if (image != null)
    {
      position += image.length();
      image = null;
      value = null;
    }

    int length = input.length();
    if (eval)
    {
      // Mobile: substituted type
      //while (position < length && Character.isWhitespace(input.charAt(position))) {
      while ((position < length) && CharacterHelper.isWhitespace(input.charAt(position)))
      {
        position++;
      }
    }

    if (position == length)
    {
      return Symbol.EOF;
    }

    if (eval)
    {
      if (input.charAt(position) == '}')
      {
        image = "}";
        eval = false;
        return Symbol.END_EVAL;
      }
      return nextEval();
    }
    else
    {
      if (position + 1 < length && input.charAt(position + 1) == '{')
      {
        switch (input.charAt(position))
        {
          case '#':
            image = "#{";
            eval = true;
            return Symbol.START_EVAL_DEFERRED;
          case '$':
            image = "${";
            eval = true;
            return Symbol.START_EVAL_DYNAMIC;
        }
      }
      return nextText();
    }
  }

}
