/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Additions/modifications to this source file by Oracle USA, Inc. 2007, 2008, 2009
 */
package de.odysseus.el.tree.impl;


import de.odysseus.el.misc.LocalMessages;
import de.odysseus.el.tree.Tree;
import de.odysseus.el.tree.impl.Scanner.ScanException;
import de.odysseus.el.tree.impl.Scanner.Symbol;
import de.odysseus.el.tree.impl.ast.AstBinary;
import de.odysseus.el.tree.impl.ast.AstBoolean;
import de.odysseus.el.tree.impl.ast.AstBracket;
import de.odysseus.el.tree.impl.ast.AstChoice;
import de.odysseus.el.tree.impl.ast.AstComposite;
import de.odysseus.el.tree.impl.ast.AstDot;
import de.odysseus.el.tree.impl.ast.AstEval;
import de.odysseus.el.tree.impl.ast.AstFunction;
import de.odysseus.el.tree.impl.ast.AstIdentifier;
import de.odysseus.el.tree.impl.ast.AstMethod;
import de.odysseus.el.tree.impl.ast.AstNested;
import de.odysseus.el.tree.impl.ast.AstNode;
import de.odysseus.el.tree.impl.ast.AstNull;
import de.odysseus.el.tree.impl.ast.AstNumber;
import de.odysseus.el.tree.impl.ast.AstString;
import de.odysseus.el.tree.impl.ast.AstText;
import de.odysseus.el.tree.impl.ast.AstUnary;

import java.util.Vector;

import oracle.adfnmc.util.Number;


/**
 * Handcrafted top-down parser.
 *
 * @author Christoph Beck
 */
final class Parser
{
  /**
   * Parse exception type
   */
  // Mobile: unsupported language feature
  //@SuppressWarnings("serial")
  static class ParseException
    extends Exception
  {
    ParseException(int position, String encountered, String expected)
    {
      // Mobile: substituted type			
      //super(LocalMessages.get("error.parse", position, encountered, expected));
      super(LocalMessages.get("error.parse", new Integer(position), encountered, expected));
    }
  }

  /**
   * Token type (used to store lookahead)
   */
  private static final class LookaheadToken
  {
    // Mobile: substituted type		
    //final Symbol symbol;
    final int symbol;
    final String image;
    final int position;

    /**
     *
     * @param symbol
     * @param image
     * @param position
     */
    // Mobile: changed signature
    // Mobile: substituted type
    //LookaheadToken(Symbol symbol, String image, int position)
    LookaheadToken(int symbol, String image, int position)
    {
      this.symbol = symbol;
      this.image = image;
      this.position = position;
    }
  }


  private static final String EXPR_FIRST =
    Scanner.Symbol.IDENTIFIER + "|" + Scanner.Symbol.STRING + "|" + Scanner.Symbol.FLOAT + "|" +
    Scanner.Symbol.INTEGER + "|" + Scanner.Symbol.TRUE + "|" + Scanner.Symbol.FALSE + "|" + Scanner.Symbol.NULL + "|" +
    Scanner.Symbol.MINUS + "|" + Scanner.Symbol.NOT + "|" + Scanner.Symbol.EMPTY + "|" + Scanner.Symbol.LPAREN;

  private final Builder context;

  private final Scanner scanner;

  // Mobile: substituted type
  //private List<IdentifierNode> identifiers = Collections.emptyList();
  private Vector identifiers = new Vector();

  // Mobile: substituted type	
  //private List<FunctionNode> functions = Collections.emptyList();
  private Vector functions = new Vector();

  // Mobile: substituted type	
  //private List<LookaheadToken> lookahead = Collections.emptyList();
  private Vector lookahead = new Vector();

  // Mobile: substituted type
  //private Symbol symbol; // current token symbol
  private int symbol = 0; // current token symbol

  private String image = null; // current token text
  private int position = 0; // current token position


  public Parser(Builder context, String input)
  {
    this.context = context;
    this.scanner = new Scanner(input);
  }

  private Number parseInteger(String string)
    throws ParseException
  {
    try
    {
      return context.parseInteger(string);
    }
    catch (NumberFormatException e)
    {
      fail(Scanner.Symbol.INTEGER);
      return null;
    }
  }

  private Number parseFloat(String string)
    throws ParseException
  {
    try
    {
      return context.parseFloat(string);
    }
    catch (NumberFormatException e)
    {
      fail(Scanner.Symbol.FLOAT);
      return null;
    }
  }

  /**
   * throw exception
   */
  private void fail(String expected)
    throws ParseException
  {
    // Mobile: substituted type		
    //String encountered = image == null ? symbol.toString() : "'" + image + "'";
    String encountered = image == null ? new Symbol(symbol).toString(): "'" + image + "'";
    throw new ParseException(position, encountered, expected);
  }

  /**
   * throw exception
   */
  // Mobile: changed signature
  // Mobile: substituted type
  //private void fail(Symbol expected) throws ParseException
  private void fail(int expected)
    throws ParseException
  {
    fail((new Symbol(expected)).toString());
  }

  /**
   * get lookahead symbol.
   */
  // Mobile: changed signature
  // Mobile: substituted type	
  //private Symbol lookahead(int index) throws ScanException, ParseException
  private int lookahead(int index)
    throws ScanException, ParseException
  {
    if (lookahead.isEmpty())
    {
      // Mobile: substituted type			
      //lookahead = new LinkedList<LookaheadToken>();
      lookahead = new Vector();
    }
    while (index >= lookahead.size())
    {
      lookahead.addElement(new LookaheadToken(scanner.next(), scanner.getImage(), scanner.getPosition()));
    }
    return ((LookaheadToken) lookahead.elementAt(index)).symbol;
  }

  /**
   * consume current token (get next token)
   */
  private void consumeToken()
    throws ScanException, ParseException
  {
    if (lookahead.isEmpty())
    {
      symbol = scanner.next();
      image = scanner.getImage();
      position = scanner.getPosition();
    }
    else
    {
      LookaheadToken token = (LookaheadToken) lookahead.elementAt(0);
      symbol = token.symbol;
      image = token.image;
      position = token.position;
      lookahead.removeElementAt(0);
    }
  }

  /**
   * consume current token (get next token); throw exception if the current token doesn't match
   * the expected symbol.
   */
  // Mobile: changed signature
  // Mobile: substituted type	
  //private void consumeToken(Symbol expected) throws ScanException, ParseException
  private void consumeToken(int expected)
    throws ScanException, ParseException
  {
    if (symbol != expected)
    {
      fail(expected);
    }
    consumeToken();
  }

  /**
   * tree := text? ((dynamic text?)+ | (deferred text?)+)?
   */
  public Tree tree()
    throws ScanException, ParseException
  {
    consumeToken();
    AstNode t = text();
    if (symbol == Scanner.Symbol.EOF)
    {
      if (t == null)
      {
        t = new AstText("");
      }
      return new Tree(t, functions, identifiers, false);
    }
    AstEval e = eval();
    if (symbol == Scanner.Symbol.EOF && t == null)
    {
      return new Tree(e, functions, identifiers, e.isDeferred());
    }
    // Mobile: substituted type		
    //ArrayList<AstNode> list = new ArrayList<AstNode>();
    Vector list = new Vector();
    if (t != null)
    {
      list.addElement(t);
    }
    list.addElement(e);
    t = text();
    if (t != null)
    {
      list.addElement(t);
    }
    while (symbol != Scanner.Symbol.EOF)
    {
      if (e.isDeferred())
      {
        list.addElement(eval(true, true));
      }
      else
      {
        list.addElement(eval(true, false));
      }
      t = text();
      if (t != null)
      {
        list.addElement(t);
      }
    }

    return new Tree(new AstComposite(list), functions, identifiers, e.isDeferred());
  }

  /**
   * text := <TEXT>
   */
  private AstNode text()
    throws ScanException, ParseException
  {
    AstNode v = null;
    if (symbol == Scanner.Symbol.TEXT)
    {
      v = new AstText(image);
      consumeToken();
    }
    return v;
  }

  /**
   * eval := dynamic | deferred
   */
  private AstEval eval()
    throws ScanException, ParseException
  {
    AstEval e = eval(false, false);
    if (e == null)
    {
      e = eval(false, true);
      if (e == null)
      {
        fail(new Symbol(Scanner.Symbol.START_EVAL_DEFERRED) + "|" + new Symbol(Scanner.Symbol.START_EVAL_DYNAMIC));
      }
    }
    return e;
  }

  /**
   * dynmamic := <START_EVAL_DYNAMIC> expr <END_EVAL> deferred := <START_EVAL_DEFERRED> expr
   * <END_EVAL>
   */
  private AstEval eval(boolean required, boolean deferred)
    throws ScanException, ParseException
  {
    AstEval v = null;
    // Mobile: substituted type		
    //Symbol start_eval = deferred ? START_EVAL_DEFERRED : START_EVAL_DYNAMIC;
    int start_eval = deferred ? Scanner.Symbol.START_EVAL_DEFERRED: Scanner.Symbol.START_EVAL_DYNAMIC;
    if (symbol == start_eval)
    {
      consumeToken();
      v = new AstEval(expr(true), deferred);
      consumeToken(Scanner.Symbol.END_EVAL);
    }
    else if (required)
    {
      fail(start_eval);
    }
    return v;
  }

  /**
   * expr := or (<QUESTION> expr <COLON> expr)?
   */
  private AstNode expr(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = or(required);
    if (v == null)
    {
      return null;
    }
    if (symbol == Scanner.Symbol.QUESTION)
    {
      consumeToken();
      AstNode a = expr(true);
      consumeToken(Scanner.Symbol.COLON);
      AstNode b = expr(true);
      v = new AstChoice(v, a, b);
    }
    return v;
  }

  /**
   * or := and (<OR> and)*
   */
  private AstNode or(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = and(required);
    if (v == null)
    {
      return null;
    }
    while (symbol == Scanner.Symbol.OR)
    {
      consumeToken();
      v = new AstBinary(v, and(true), AstBinary.OR);
    }
    return v;
  }

  /**
   * and := eq (<AND> eq)*
   */
  private AstNode and(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = eq(required);
    if (v == null)
    {
      return null;
    }
    while (symbol == Scanner.Symbol.AND)
    {
      consumeToken();
      v = new AstBinary(v, eq(true), AstBinary.AND);
    }
    return v;
  }

  /**
   * eq := cmp (<EQ> cmp | <NE> cmp)*
   */
  private AstNode eq(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = cmp(required);
    if (v == null)
    {
      return null;
    }
    while (true)
    {
      switch (symbol)
      {
        case Scanner.Symbol.EQ:
          consumeToken();
          v = new AstBinary(v, cmp(true), AstBinary.EQ);
          break;
        case Scanner.Symbol.NE:
          consumeToken();
          v = new AstBinary(v, cmp(true), AstBinary.NE);
          break;
        default:
          return v;
      }
    }
  }

  /**
   * cmp := add (<LT> add | <LE> add | <GE> add | <GT> add)*
   */
  private AstNode cmp(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = add(required);
    if (v == null)
    {
      return null;
    }
    while (true)
    {
      switch (symbol)
      {
        case Scanner.Symbol.LT:
          consumeToken();
          v = new AstBinary(v, add(true), AstBinary.LT);
          break;
        case Scanner.Symbol.LE:
          consumeToken();
          v = new AstBinary(v, add(true), AstBinary.LE);
          break;
        case Scanner.Symbol.GE:
          consumeToken();
          v = new AstBinary(v, add(true), AstBinary.GE);
          break;
        case Scanner.Symbol.GT:
          consumeToken();
          v = new AstBinary(v, add(true), AstBinary.GT);
          break;
        default:
          return v;
      }
    }
  }

  /**
   * add := add (<PLUS> mul | <MINUS> mul)*
   */
  private AstNode add(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = mul(required);
    if (v == null)
    {
      return null;
    }
    while (true)
    {
      switch (symbol)
      {
        case Scanner.Symbol.PLUS:
          consumeToken();
          v = new AstBinary(v, mul(true), AstBinary.ADD);
          break;
        case Scanner.Symbol.MINUS:
          consumeToken();
          v = new AstBinary(v, mul(true), AstBinary.SUB);
          break;
        default:
          return v;
      }
    }
  }

  /**
   * mul := unary (<MUL> unary | <DIV> unary | <MOD> unary)*
   */
  private AstNode mul(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = unary(required);
    if (v == null)
    {
      return null;
    }
    while (true)
    {
      switch (symbol)
      {
        case Scanner.Symbol.MUL:
          consumeToken();
          v = new AstBinary(v, unary(true), AstBinary.MUL);
          break;
        case Scanner.Symbol.DIV:
          consumeToken();
          v = new AstBinary(v, unary(true), AstBinary.DIV);
          break;
        case Scanner.Symbol.MOD:
          consumeToken();
          v = new AstBinary(v, unary(true), AstBinary.MOD);
          break;
        default:
          return v;
      }
    }
  }

  /**
   * unary := <NOT> unary | <MINUS> unary | <EMPTY> unary | value
   */
  private AstNode unary(boolean required)
    throws ScanException, ParseException
  {
    AstNode v = null;
    switch (symbol)
    {
      case Scanner.Symbol.NOT:
        consumeToken();
        v = new AstUnary(unary(true), AstUnary.NOT);
        break;
      case Scanner.Symbol.MINUS:
        consumeToken();
        v = new AstUnary(unary(true), AstUnary.NEG);
        break;
      case Scanner.Symbol.EMPTY:
        consumeToken();
        v = new AstUnary(unary(true), AstUnary.EMPTY);
        break;
      default:
        v = value();
    }
    if (v == null && required)
    {
      fail(EXPR_FIRST);
    }
    return v;
  }

  /**
   * value := (nonliteral | literal) (<DOT> <IDENTIFIER> | <LBRACK> expr <RBRACK>)*
   */
  private AstNode value()
    throws ScanException, ParseException
  {
    boolean lvalue = true;
    AstNode v = nonliteral();
    if (v == null)
    {
      v = literal();
      if (v == null)
      {
        return null;
      }
      lvalue = false;
    }
    while (true)
    {
      switch (symbol)
      {
        case Scanner.Symbol.DOT:
          consumeToken();
          String name = image;
          consumeToken(Scanner.Symbol.IDENTIFIER);
          if (symbol == Scanner.Symbol.LPAREN && context.isEnabled(Builder.Feature.METHOD_INVOCATIONS))
          {
            consumeToken();
            v = new AstMethod(v, name, list());
            consumeToken(Scanner.Symbol.RPAREN);
          }
          else
          {
            v = new AstDot(v, name, lvalue);
          }
          break;
        case Scanner.Symbol.LBRACK:
          consumeToken();
          AstNode property = expr(true);
          boolean strict = !context.isEnabled(Builder.Feature.NULL_PROPERTIES);
          v = new AstBracket(v, property, lvalue, strict);
          consumeToken(Scanner.Symbol.RBRACK);
          break;
        default:
          return v;
      }
    }
  }

  /**
   * nonliteral := <IDENTIFIER> | function | <LPAREN> expr <RPAREN> function := (<IDENTIFIER>
   * <COLON>)? <IDENTIFIER> <LPAREN> list? <RPAREN>
   */
  private AstNode nonliteral()
    throws ScanException, ParseException
  {
    AstNode v = null;
    switch (symbol)
    {
      case Scanner.Symbol.IDENTIFIER:
        String name = image;
        consumeToken();
        if (symbol == Scanner.Symbol.COLON && lookahead(0) == Scanner.Symbol.IDENTIFIER &&
            lookahead(1) == Scanner.Symbol.LPAREN)
        { // ns:f(...)
          consumeToken();
          name += ":" + image;
          consumeToken();
        }
        if (symbol == Scanner.Symbol.LPAREN)
        { // function
          consumeToken();
          // Mobile: substituted type					
          //List<AstNode> args = list();
          Vector args = list();
          consumeToken(Scanner.Symbol.RPAREN);
          if (functions.isEmpty())
          {
            // Mobile: substituted type						
            //functions = new ArrayList<FunctionNode>(4);
            functions = new Vector();
          }
          AstFunction function = new AstFunction(name, functions.size(), args);
          functions.addElement(function);
          v = function;
        }
        else
        { // identifier
          if (identifiers.isEmpty())
          {
            // Mobile: substituted type						
            //identifiers = new ArrayList<IdentifierNode>(4);
            identifiers = new Vector();
          }
          AstIdentifier identifier = new AstIdentifier(name, identifiers.size());
          identifiers.addElement(identifier);
          v = identifier;
        }
        break;
      case Scanner.Symbol.LPAREN:
        consumeToken();
        v = expr(true);
        consumeToken(Scanner.Symbol.RPAREN);
        v = new AstNested(v);
        break;
    }
    return v;
  }

  /**
   * list := expr (<COMMA> expr)*
   */
  // Mobile: substituted type
  //private List<AstNode> list() throws ScanException, ParseException {
  private Vector list()
    throws ScanException, ParseException
  {
    // Mobile: substituted type		
    //List<AstNode> l = null;
    Vector l = null;
    AstNode v = expr(false);
    if (v != null)
    {
      // Mobile: substituted type			
      //l = new ArrayList<AstNode>();
      l = new Vector();
      l.addElement(v);
      while (symbol == Scanner.Symbol.COMMA)
      {
        consumeToken();
        l.addElement(expr(true));
      }
    }
    return l;
  }

  /**
   * literal := <TRUE> | <FALSE> | <STRING> | <INTEGER> | <FLOAT> | <NULL>
   */
  private AstNode literal()
    throws ScanException, ParseException
  {
    AstNode v = null;
    switch (symbol)
    {
      case Scanner.Symbol.TRUE:
        v = new AstBoolean(true);
        consumeToken();
        break;
      case Scanner.Symbol.FALSE:
        v = new AstBoolean(false);
        consumeToken();
        break;
      case Scanner.Symbol.STRING:
        v = new AstString(image);
        consumeToken();
        break;
      case Scanner.Symbol.INTEGER:
        v = new AstNumber(parseInteger(image));
        consumeToken();
        break;
      case Scanner.Symbol.FLOAT:
        v = new AstNumber(parseFloat(image));
        consumeToken();
        break;
      case Scanner.Symbol.NULL:
        v = new AstNull();
        consumeToken();
        break;
    }
    return v;
  }
}
