/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Additions/modifications to this source file by Oracle USA, Inc. 2007, 2008, 2009
 */
package de.odysseus.el.misc;


import oracle.adfnmc.el.ELException;
import oracle.adfnmc.el.TypeCoercionException;
import oracle.adfnmc.util.Number;
import oracle.adfnmc.util.Utility;


/**
 * Type Conversions as described in EL 2.1 specification (section 1.17).
 */
// Mobile: added type
public class TypeConversionsImpl
{
  public static final Class CLASS_INSTANCE = TypeConversionsImpl.class;

  // Mobile: substituted type
  //private static final Long LONG_ZERO = Long.valueOf(0L);
  private static final Number LONG_ZERO = new Number(0L);


  // Mobile: removed behavior
  // Mobile: unsupported type
  //private static final HashMap<,> WRAPPER_TYPES = new HashMap<,>();

  // Mobile: removed behavior
  //static {
  //  WRAPPER_TYPES.put(Boolean.TYPE, Boolean.class);
  //  WRAPPER_TYPES.put(Character.TYPE, Character.class);
  //  WRAPPER_TYPES.put(Byte.TYPE, Byte.class);
  //  WRAPPER_TYPES.put(Short.TYPE, Short.class);
  //  WRAPPER_TYPES.put(Integer.TYPE, Integer.class);
  //  WRAPPER_TYPES.put(Long.TYPE, Long.class);
  //  WRAPPER_TYPES.put(Float.TYPE, Float.class);
  //  WRAPPER_TYPES.put(Double.TYPE, Double.class);
  //}

  public static final Boolean coerceToBoolean(Object value)
  {
    if (value == null || "".equals(value))
    {
      return Boolean.FALSE;
    }

    if (value instanceof Boolean)
    {
      return (Boolean) value;
    }

    if (value instanceof String)
    {
      // Mobile: unsupported method
      //return Boolean.valueOf((String)value);
      String strValue = (String) value;
      if (strValue.equalsIgnoreCase("true"))
      {
        return Boolean.TRUE;
      }

      if (strValue.equalsIgnoreCase("false"))
      {
        return Boolean.FALSE;
      }

      // now try to parse an int
      try
      {
        int intValue = Integer.parseInt(strValue);
        return (intValue == 0) ? Boolean.FALSE: Boolean.TRUE;
      }
      catch (NumberFormatException e)
      {
        // do nothing
        ;
      }
    }

    // Mobile: added behavior - added this boolean conversion
    //check EL spec rules on Boolean conversions from numeric values
    if (value instanceof Number)
    {
      byte val = ((Number) value).byteValue();
      return (val == 0) ? Boolean.FALSE: Boolean.TRUE;
    }

    // Mobile: substituted type
    //throw new ELException(LocalMessages.get("error.coerce.type", value.getClass(), Boolean.class));
    throw new TypeCoercionException(LocalMessages.get("error.coerce.type", value.getClass(), Utility.BOOLEAN_CLASS));
  }

  public static final Character coerceToCharacter(Object value)
  {
    if (value == null || "".equals(value))
    {
      // Mobile: unsupported method
      //return Character.valueOf((char)0);
      return new Character((char) 0);
    }
    if (value instanceof Character)
    {
      return (Character) value;
    }
    if (value instanceof Number)
    {
      return new Character((char) ((Number) value).shortValue());
    }
    if (value instanceof String)
    {
      return new Character(((String) value).charAt(0));
    }

    // Mobile: substituted type
    //throw new ELException(LocalMessages.get("error.coerce.type", value.getClass(), Character.class));
    throw new TypeCoercionException(LocalMessages.get("error.coerce.type", value.getClass(), Utility.CHARACTER_CLASS));
  }

  /**
   *
   * @param value
   * @param type
   * @return
   */
  // Mobile: changed signature
  // Mobile: unsupported language feature
  //private static final Number coerceStringToNumber(String value, Class<? extends Number> type)
  private static final Object coerceStringToNumber(String value, Class type)
  {
    try
    {
      // Mobile: unsupported type
      //if (type == BigDecimal.class)
      //{
      //  return new BigDecimal(value);
      //}

      // Mobile: unsupported type
      //if (type == BigInteger.class)
      //{
      //  return new BigInteger(value);
      //}

      // Mobile: substituted type
      if (type == Utility.BYTE_CLASS)
      {
        // Mobile: unsupported method
        //return Byte.valueOf(value);
        return Number.box(Byte.parseByte(value));
      }
      // Mobile: substituted type
      if (type == Utility.SHORT_CLASS)
      {
        // Mobile: unsupported method
        //return Short.valueOf(value);
        return Number.box(Short.parseShort(value));
      }
      // Mobile: substituted type
      if (type == Utility.INTEGER_CLASS)
      {
        return Integer.valueOf(value);
      }
      // Mobile: substituted type
      if (type == Utility.LONG_CLASS)
      {
        // Mobile: unsupported method
        //return Long.valueOf(value);
        return Number.box(Long.parseLong(value));
      }
      // Mobile: substituted type
      if (type == Utility.FLOAT_CLASS)
      {
        return Float.valueOf(value);
      }
      // Mobile: substituted type
      if (type == Utility.DOUBLE_CLASS)
      {
        return Double.valueOf(value);
      }
    }
    catch (NumberFormatException e)
    {
      throw new ELException(LocalMessages.get("error.coerce.value", value, type));
    }

    //throw new ELException(LocalMessages.get("error.coerce.type", ClassObjectCache.getStringClass(), type));
    throw new TypeCoercionException(LocalMessages.get("error.coerce.type", Utility.STRING_CLASS, type));
  }

  /**
   *
   * @param value
   * @param type
   * @return
   */
  // Mobile: changed signature
  // Mobile: unsupported language feature
  //private static final Number coerceNumberToNumber(Number value, Class<? extends Number> type)
  private static final Object coerceNumberToNumber(Number value, Class type)
  {
    // Mobile: substituted type
    //if (type.isInstance(value))
    if (type.isInstance(value.getInnerType()))
    {
      return value;
    }

    // Mobile: unsupported type		
    //if (type == BigInteger.class)
    //{
    //  if (value instanceof BigDecimal)
    //  {
    //    return ((BigDecimal) value).toBigInteger();
    //  }
    //  return BigInteger.valueOf((value).longValue());
    //}

    // Mobile: unsupported type
    //if (type == BigDecimal.class)
    //{
    //  if (value instanceof BigInteger)
    //  {
    //    return new BigDecimal((BigInteger) value);
    //  }
    //  return new BigDecimal(value.doubleValue());
    //}

    // Mobile: substituted type
    if (type == Utility.BYTE_CLASS)
    {
      // Mobile: unsupported method
      //return Byte.valueOf(value.byteValue());
      return new Byte(value.byteValue());
    }
    // Mobile: substituted type
    if (type == Utility.SHORT_CLASS)
    {
      // Mobile: unsupported method
      //return Short.valueOf(value.shortValue());
      return new Short(value.shortValue());
    }
    // Mobile: substituted type
    if (type == Utility.INTEGER_CLASS)
    {
      // Mobile: unsupported method
      //return Integer.valueOf(value.intValue());
      return new Integer(value.intValue());
    }
    // Mobile: substituted type
    if (type == Utility.LONG_CLASS)
    {
      // Mobile: unsupported method
      //return Long.valueOf(value.longValue());
      return new Long(value.longValue());
    }
    // Mobile: substituted type
    if (type == Utility.FLOAT_CLASS)
    {
      // Mobile: unsupported method
      //return Float.valueOf(value.floatValue());
      return new Float(value.floatValue());
    }
    // Mobile: substituted type
    if (type == Utility.DOUBLE_CLASS)
    {
      // Mobile: unsupported method
      //return Double.valueOf(value.doubleValue());
      return new Double(value.doubleValue());
    }

    //throw new ELException(LocalMessages.get("error.coerce.type", value.getClass(), type));
    throw new TypeCoercionException(LocalMessages.get("error.coerce.type", value.getClass(), type));
  }

  /**
   *
   * @param value
   * @param type
   * @return
   */
  // Mobile: changed signature
  // Mobile: unsupported language feature
  //@SuppressWarnings("unchecked")
  //public static final <T extends Number> T coerceToNumber(Object value, Class type)
  public static final Object coerceToNumber(Object value, Class type)
  {
    if (value == null || "".equals(value))
    {
      // Mobile: unsupported language feature
      //return (T)coerceNumberToNumber(LONG_ZERO, type);
      return coerceNumberToNumber(LONG_ZERO, type);
    }

    if (value instanceof Character)
    {
      // Mobile: unsupported language feature
      // Mobile: unsupported method
      //return (T)coerceNumberToNumber(Short.valueOf((short)((Character)value).charValue()), type);
      return coerceNumberToNumber(new Number((short) ((Character) value).charValue()), type);
    }

    // Mobile: removed behavior
    //if (value instanceof Number)
    //{
    //	return (T)coerceNumberToNumber((Number)value, type);
    //}

    // Mobile: substituted type
    if (Number.isObjectNumeric(value))
    {
      return coerceNumberToNumber(new Number(value), type);
    }

    if (value instanceof String)
    {
      // Mobile: unsupported language feature
      //return (T)coerceStringToNumber((String)value, type);
      return coerceStringToNumber((String) value, type);
    }

    //throw new ELException(LocalMessages.get("error.coerce.type", value.getClass(), type));
    throw new TypeCoercionException(LocalMessages.get("error.coerce.type", value.getClass(), type));
  }

  public static final String coerceToString(Object value)
  {
    if (value == null)
    {
      return "";
    }
    if (value instanceof String)
    {
      return (String) value;
    }

    // Mobile: unsupported type
    //if (value instanceof Enum)
    //{
    //  return ((Enum)value).name();
    //}

    return value.toString();
  }

  // Mobile: removed behavior
  // Mobile: unsupported language feature
  // Mobile: unsupported type dependency
  //@SuppressWarnings("unchecked")
  //public static final <T extends Enum<T>> T coerceToEnum(Object value, Class type)
  //{
  //  if (value == null || "".equals(value))
  //  {
  //    return null;
  //  }
  //  if (type.isInstance(value))
  //  {
  //    return (T) value;
  //  }
  //  if (value instanceof String)
  //  {
  //    try
  //    {
  //      return Enum.valueOf(type, (String) value);
  //    }
  //    catch (IllegalArgumentException e)
  //    {
  //      throw new ELException(LocalMessages.get("error.coerce.value", value, type));
  //    }
  //  }
  //  throw new ELException(LocalMessages.get("error.coerce.type", value.getClass(), type));
  //}

  // Mobile: removed behavior
  // Mobile: unsupported type dependency
  //private static final Object coerceStringToType(String value, Class type)
  //{
  //  PropertyEditor editor = PropertyEditorManager.findEditor(type);
  //  if (editor == null)
  //  {
  //    if ("".equals(value))
  //    {
  //      return null;
  //    }
  //    throw new ELException(LocalMessages.get("error.coerce.type", ClassObjectCache.getStringClass(), type));
  //  }
  //  else
  //  {
  //    if ("".equals(value))
  //    {
  //      try
  //      {
  //        editor.setAsText(value);
  //      }
  //      catch (IllegalArgumentException e)
  //      {
  //        return null;
  //      }
  //    }
  //    else
  //    {
  //      try
  //      {
  //        editor.setAsText(value);
  //      }
  //      catch (IllegalArgumentException e)
  //      {
  //        throw new ELException(LocalMessages.get("error.coerce.value", value, type));
  //      }
  //    }
  //    return editor.getValue();
  //  }
  //}

  /**
   *
   * @param value
   * @param type
   * @return
   */
  // Mobile: unsupported language feature
  //@SuppressWarnings("unchecked")
  public static final Object coerceToType(Object value, Class type)
  {
    // Mobile: substituted type
    if (type == Utility.STRING_CLASS)
    {
      return coerceToString(value);
    }

    // Mobile: unsupported method
    //if (type.isPrimitive())
    //{
    //  type = WRAPPER_TYPES.get(type);
    //}

    // Mobile: unsupported method
    // Mobile: substituted type
    //if (Number.class.isAssignableFrom(type))
    if (Number.isTypeNumeric(type))
    {
      // Mobile: unsupported language feature
      //return coerceToNumber(value, (Class<? extends Number>)type);
      return coerceToNumber(value, type);
    }

    // Mobile: substituted type
    if (type == Utility.CHARACTER_CLASS)
    {
      return coerceToCharacter(value);
    }

    // Mobile: substituted type
    if (type == Utility.BOOLEAN_CLASS)
    {
      return coerceToBoolean(value);
    }

    // Mobile: unsupported language feature
    //if (Enum.class.isAssignableFrom(type))
    //{
    //  return coerceToEnum(value, (Class<? extends Enum>)type);
    //}

    if (value == null)
    {
      return null;
    }

    if (type.isInstance(value))
    {
      return value;
    }

    if (value instanceof String)
    {
      // Mobile: unsupported type dependency
      //return coerceStringToType((String)value, type);
    }

    //throw new ELException(LocalMessages.get("error.coerce.type", value.getClass(), type));
    throw new TypeCoercionException(LocalMessages.get("error.coerce.type", value.getClass(), type));
  }

  /**
   *
   * @param type
   * @return
   */
  public static final boolean isPrimitive(Class type)
  {
    throw new UnsupportedOperationException("isPrimitive");
  }
}
