/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Additions/modifications to this source file by Oracle USA, Inc. 2007, 2008, 2009
 */
package de.odysseus.el;


import de.odysseus.el.misc.LocalMessages;
import de.odysseus.el.tree.Bindings;
import de.odysseus.el.tree.ExpressionNode;
import de.odysseus.el.tree.NodePrinter;
import de.odysseus.el.tree.Tree;
import de.odysseus.el.tree.TreeBuilder;
import de.odysseus.el.tree.TreeStore;

import java.util.Hashtable;

import oracle.adfnmc.el.ELContext;
import oracle.adfnmc.el.ELException;
import oracle.adfnmc.el.ExpressionFactory;
import oracle.adfnmc.el.FunctionMapper;
import oracle.adfnmc.el.ValueExpression;
import oracle.adfnmc.el.VariableMapper;
import oracle.adfnmc.el.event.privateImpl.ValueChangeDelegate;
import oracle.adfnmc.java.io.PrintWriter;


/**
 * A value expression is ready to be evaluated (by calling either {@link #getType(ELContext)},
 * {@link #getValue(ELContext)}, {@link #isReadOnly(ELContext)} or
 * {@link #setValue(ELContext, Object)}.
 *
 * Instances of this class are usually created using an {@link ExpressionFactoryImpl}.
 *
 * @author Christoph Beck
 */
// Mobile: changed signature
//public final class TreeValueExpression extends oracle.adfnmc.el.ValueExpression
public final class TreeValueExpression
  implements ValueExpression, oracle.adfnmc.el.event.expression.ValueChangeSource
{
  // Mobile: unsupported language feature
  //private static final long serialVersionUID = 1L;

  private final TreeBuilder builder;
  private final Bindings bindings;
  private final String expr;
  private Class type;
  private final boolean deferred;
  private boolean valueIsSecret = false;
  private transient ExpressionNode node;
  private String structure;

  /**
   * Create a new value expression.
   *
   * @param store
   *            used to get the parse tree from.
   * @param functions
   *            the function mapper used to bind functions
   * @param variables
   *            the variable mapper used to bind variables
   * @param expr
   *            the expression string
   * @param type
   *            the expected type (may be <code>null</code>)
   */
  public TreeValueExpression(TreeStore store, FunctionMapper functions, VariableMapper variables, String expr,
                             Class type)
  {
    super();

    Tree tree = store.get(expr);
    this.builder = store.getBuilder();
    this.bindings = tree.bind(functions, variables);
    this.expr = expr;
    this.type = type;
    this.node = tree.getRoot();
    this.deferred = tree.isDeferred();

    if (type == null)
    {
      throw new NullPointerException(LocalMessages.get("error.value.notype"));
    }
  }

  private String getStructuralId()
  {
    if (structure == null)
    {
      structure = node.getStructuralId(bindings);
    }
    return structure;
  }

  private Hashtable delegates = new Hashtable();

  /**
   *
   * @param listener
   */
  // Mobile: added behavior
  public void addValueChangeListener(oracle.adfnmc.el.event.expression.ValueChangeListener listener)
  {
    ValueChangeDelegate delegate = (ValueChangeDelegate) this.delegates.get(listener);
    if (delegate == null)
    {
      delegate = new ValueChangeDelegate(this, listener);
      this.delegates.put(listener, delegate);
    }
    this.node.addValueChangeListener(this.bindings, ELContext.getInstance(), delegate);
  }

  /**
   *
   * @param listener
   */
  // Mobile: added behavior
  public void removeValueChangeListener(oracle.adfnmc.el.event.expression.ValueChangeListener listener)
  {
    ValueChangeDelegate delegate = (ValueChangeDelegate) this.delegates.get(listener);
    if (delegate != null)
    {
      this.node.removeValueChangeListener(this.bindings, ELContext.getInstance(), delegate);
      this.delegates.remove(listener);
    }
  }

  public Class getExpectedType()
  {
    return type;
  }

  public void setExpectedType(Class type)
  {
    this.type = type;
  }

  public String getExpressionString()
  {
    return expr;
  }

  /**
   * Evaluates the expression as an lvalue and answers the result type.
   *
   * @param context
   *            used to resolve properties (<code>base.property</code> and
   *            <code>base[property]</code>) and to determine the result from the last
   *            base/property pair
   * @return lvalue evaluation type or <code>null</code> for rvalue expressions
   * @throws ELException
   *             if evaluation fails (e.g. property not found, type conversion failed, ...)
   */
  public Class getType(ELContext context)
    throws ELException
  {
    return node.getType(bindings, context);
  }

  public Object getValue()
    throws ELException
  {
    return this.getValue(ELContext.getInstance());
  }

  /**
   * Evaluates the expression as an rvalue and answers the result.
   *
   * @param context
   *            used to resolve properties (<code>base.property</code> and
   *            <code>base[property]</code>) and to determine the result from the last
   *            base/property pair
   * @return rvalue evaluation result
   * @throws ELException
   *             if evaluation fails (e.g. property not found, type conversion failed, ...)
   */
  public Object getValue(ELContext context)
    throws ELException
  {
    Object value = node.getValue(bindings, context, type);

    return value;
  }

  public boolean isReadOnly()
  {
    return this.isReadOnly(ELContext.getInstance());
  }

  /**
   * Evaluates the expression as an lvalue and determines if {@link #setValue(ELContext, Object)}
   * will always fail.
   *
   * @param context
   *            used to resolve properties (<code>base.property</code> and
   *            <code>base[property]</code>) and to determine the result from the last
   *            base/property pair
   * @return <code>true</code> if {@link #setValue(ELContext, Object)} always fails.
   * @throws ELException
   *             if evaluation fails (e.g. property not found, type conversion failed, ...)
   */
  public boolean isReadOnly(ELContext context)
    throws ELException
  {
    return node.isReadOnly(bindings, context);
  }

  public void setValue(Object value)
  {
    this.setValue(ELContext.getInstance(), value);
  }

  /**
   * Evaluates the expression as an lvalue and assigns the given value.
   *
   * @param context
   *            used to resolve properties (<code>base.property</code> and
   *            <code>base[property]</code>) and to perform the assignment to the last
   *            base/property pair
   * @throws ELException
   *             if evaluation fails (e.g. property not found, type conversion failed, assignment
   *             failed...)
   */
  public void setValue(ELContext context, Object value)
    throws ELException
  {
    node.setValue(bindings, context, value);
  }

  /**
   * @return <code>true</code> if this is a literal text expression
   */
  public boolean isLiteralText()
  {
    return node.isLiteralText();
  }

  /**
   * Answer <code>true</code> if this could be used as an lvalue. This is the case for eval
   * expressions consisting of a simple identifier or a nonliteral prefix, followed by a sequence
   * of property operators (<code>.</code> or <code>[]</code>)
   */
  public boolean isLeftValue()
  {
    return node.isLeftValue();
  }

  /**
   * Answer <code>true</code> if this is a deferred expression (containing sub-expressions
   * starting with <code>#{</code>)
   */
  public boolean isDeferred()
  {
    return deferred;
  }

  /**
   * Returns true if the value resolved by this expression is secret and should not be displayed or logged
   */
  public boolean isValueSecret()
  {
    return this.valueIsSecret;
  }

  /**
   * Set to true if the value resolved by this expression is secret and should not be displayed or logged
   */
  public void setValueSecret(boolean valueIsSecret)
  {
    this.valueIsSecret = valueIsSecret;
  }

  /**
   * Expressions are compared using the concept of a <em>structural id</em>: variable and
   * function names are anonymized such that two expressions with same tree structure will also
   * have the same structural id and vice versa. Two value expressions are equal if
   * <ol>
   * <li>their structural id's are equal</li>
   * <li>their bindings are equal</li>
   * <li>their expected types are equal</li>
   * </ol>
   */
  public boolean equals(Object obj)
  {
    if (obj != null && obj.getClass() == getClass())
    {
      TreeValueExpression other = (TreeValueExpression) obj;
      if (!builder.equals(other.builder))
      {
        return false;
      }
      if (type != other.type)
      {
        return false;
      }
      return getStructuralId().equals(other.getStructuralId()) && bindings.equals(other.bindings);
    }
    return false;
  }

  public int hashCode()
  {
    return getStructuralId().hashCode();
  }

  public String toString()
  {
    return "TreeValueExpression(" + expr + ")";
  }

  /**
   * Print the parse tree.
   *
   * @param writer
   */
  public void dump(PrintWriter writer)
  {
    NodePrinter.dump(writer, node);
  }

  // Mobile: removed behavior
  // Mobile: unsupported type dependency
  //private void readObject(ObjectInputStream in)
  //  throws IOException, ClassNotFoundException
  //{
  //  in.defaultReadObject();
  //  try
  //  {
  //    node = builder.build(expr).getRoot();
  //  }
  //  catch (ELException e)
  //  {
  //    throw new IOException(e.getMessage());
  //  }
  //}
}
