<?xml version="1.0" encoding="windows-1252" ?>
<!-- 
  This XSL document transforms the preferences contained in adfmf_application.xml and 
  adfmf_feature.xml from ADFmf definitions to corresponding Android definitions.  This XSL must 
  first be transformed by the "SetIncludes.xsl" before it can be used.  The "SetIncludes.xsl" sets 
  the location of the XSL file that this file includes - namely "TransformUtilities.xsl" (see the 
  <xsl:include> element at the root of
  this XSL). 
   
-->
<xsl:stylesheet version="2.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:adfmf="http://xmlns.oracle.com/adf/mf"
                xmlns:android="http://schemas.android.com/apk/res/android"
                xmlns:ora="http://www.oracle.com/XSL/Transform/java" 
                xmlns:xlf="urn:oasis:names:tc:xliff:document:1.1"
                exclude-result-prefixes="adfmf xlf">
  <!-- 
    NOTE:  the exclude-result-prefixes attribute above must be specified in order to prevent
           the adfmf namespace of http://xmlns.oracle.com/adfmf from appearing in the
           output file(s).
    -->
  <!-- 
        This href is not valid as-is - the value being specified here is to let the reader know 
        that this XSL is to include TransformUtilities.xsl.  The true value is set at deployment 
        time by a tranformation done on this XSL using the SetIncludes.xsl.
   -->
  <xsl:include href="TransformUtilities.xsl"/>
  
  <!-- 
    Root folder where the Android preferences.xml, strings.xml and arrays.xml should be created 
  -->
  <xsl:param name="XSL_PARAM_targetDirectoryPath" 
             required="yes" 
             as="xsl:string"/>
  
  <!-- Application package name.-->
  <xsl:param name="XSL_PARAM_applicationPackage" 
             required="yes" 
             as="xsl:string"/>
  
  <!--
  This parameter is an unordered sequence of zero or more "LocalData" elements,
  as specified in the "as" attribute of the parameter declaration.

  Each LocaleData element identifies a locale.  The locale is specified as two different
  representations corresponding to the following two attributes.  Both attributes must
  both be populated.
  
    oracleLocale   <- Standard (canonical) representation of a locale extracted from
                      an XLIFF file name suffix. Each oracleLocale attribute is
                      used to construct the name of a localized XLIFF file.
                      For the definition of a valid Oracle locale,
                      see Java method "isValidOracleLocale" in class
                      "oracle.adfmf.common.OracleLocale".
    
    androidLocale  <- Corresponding Android representation of a locale specified
                      in attribute oracleLocale. Each androidLocale value is used
                      to construct a localized directory name for Android string
                      resources. Each locale value corresponds to a custom Android
                      localization pattern.  The Android locale pattern is described
                      in table entry "Language and region" in section
                      "Providing Alternate Resources" in the Android localization
                      document at 
                      "http://developer.android.com/guide/topics/resources/providing-resources.html#
                      AlternativeResources".

  The following table shows example attribute values.

    oracleLocale    androidLocale
    ************    ***********************
    en              en
    pt              pt
    pt_PT           pt-rPT
    fr              fr
    fr_FR           fr-rFR

   The oracle and Android locale values are different only when the Oracle locale
   includes a country code appended to the language code.
  -->
  <xsl:param name="XSL_PARAM_LocaleDataElements" 
             required="yes" 
             as="xsl:element(LocaleData)*"/>
  
  <!-- A XML document containing the list of FAR JAR URL paths.  Used when processing features -->
  <xsl:param name="XSL_PARAM_FeatureArchiveUrlNodeSet" 
             required="yes" 
             as="document-node()"/>
  
  <!-- A relative path to the location of adfmf-feature.xml in a FAR JAR file. -->
  <xsl:param name="XSL_PARAM_FeatureXmlRelativePathInFAR" 
             required="yes" 
             as="xsl:string"/>
  
  <!-- A path to the location of the XLIFF files -->
  <xsl:param name="XSL_PARAM_applicationXliffBasePath" 
             required="yes" 
             as="xsl:string"/>
  
  <xsl:output method="xml" indent="yes" name="xml"/>
  
  <!-- The namespace alias required in the Android preferences.xml file in the root element. -->
  <xsl:variable name="g_nsPrefix" select="'adfmf'"/>
  
  <!-- The namespace URI required in the Android preferences.xml file in the root element. -->
  <xsl:variable name="g_nsNamespace"
                select="concat ('http://schemas.android.com/apk/res/', 
                                $XSL_PARAM_applicationPackage)"/>

  <!-- Name of the Android preference 'title' attribute -->                
  <xsl:variable name="g_androidTitleAttributeName" select="'title'"/>
  
  <!-- Name of the Android preference 'defaultValue' attribute -->
  <xsl:variable name="g_androidDefaultAttributeName" select="'defaultValue'"/>
  
  <!-- Name of the ADFmf preference 'default' attribute -->
  <xsl:variable name="g_adfmfDefaultAttributeName" select="'default'"/>
  
  <!-- Name of the ADFmf preference 'label' attribute -->
  <xsl:variable name="g_adfmfLabelAttributeName" select="'label'"/>
  
  <!-- Name of the ADFmf preference 'id' attribute -->
  <xsl:variable name="g_adfmfIdAttributeName" select="'id'"/>
  
  <!-- Name of the ADFmf preference 'name' attribute -->
  <xsl:variable name="g_adfmfNameAttributeName" select="'name'"/>
  
  <!-- Name of the ADFmf preference 'value' attribute -->
  <xsl:variable name="g_adfmfValueAttributeName" select="'value'"/>

  <!-- Used to construct a Android string identifer.  Separates a
       preference identifier value from the Android attribute -->
  <xsl:variable name="g_androidAttributeValueSeparator" select="'___'"/>

  <!-- . character -->  
  <xsl:variable name="g_periodChar" select="'.'"/>

  <!-- _ character -->  
  <xsl:variable name="g_underscoreChar" select="'_'"/>
  
  <!-- empty string -->
  <xsl:variable name="g_emptyString" select="''"/>
  
  <!-- - character used as the separator for locale specific directory names -->
  <xsl:variable name="g_androidLocaleSeparator" select="'-'"/>
  
  <!-- Non-escaped and escaped definitions for apostrophe (') -->
  <xsl:variable name="g_APOSTROPHE">
    <xsl:text>'</xsl:text>
  </xsl:variable>
  
  <xsl:variable name="g_ESCAPED_APOSTROPHE">
    <xsl:text>\'</xsl:text>
  </xsl:variable>

  <!-- Non-escaped and escaped definitions for double quotes (") -->
  <xsl:variable name="g_DOUBLE_QUOTE">
    <xsl:text>"</xsl:text>
  </xsl:variable>
  
  <xsl:variable name="g_ESCAPED_DOUBLE_QUOTE">
    <xsl:text>\"</xsl:text>
  </xsl:variable>

  <!-- Non-escaped and escaped definitions for backslash (\) -->
  <xsl:variable name="g_BACK_SLASH">
    <xsl:text>\</xsl:text>
  </xsl:variable>
  
  <xsl:variable name="g_ESCAPED_BACK_SLASH">
    <xsl:text>\\</xsl:text>
  </xsl:variable>
  
  <!-- Maps non-escaped strings to corresponding escaped strings.  Android
       uses "\" to escape XML values.  For example, a ADF Mobile boolean
       preference definition of:
       <adfmf:preferenceBoolean id="foo" label="Don't Load Images\Icons"/>
       must be translated in strings.xml as:
       <string name="foo">Don\'t Load Images\\Icons</string>
       in order for the preference to appear as "Don't Load Images\Icons"
       at runtime.
  -->
  <xsl:variable name="g_ESCAPE_CHARACTER_MAP">
    <android:entries>
      <android:entry>
        <android:nonescaped><xsl:value-of select="$g_BACK_SLASH"/></android:nonescaped>
        <android:escaped><xsl:value-of select="$g_ESCAPED_BACK_SLASH"/></android:escaped>
      </android:entry>
      <android:entry>
        <android:nonescaped><xsl:value-of select="$g_APOSTROPHE"/></android:nonescaped>
        <android:escaped><xsl:value-of select="$g_ESCAPED_APOSTROPHE"/></android:escaped>
      </android:entry>
      <android:entry>
        <android:nonescaped><xsl:value-of select="$g_DOUBLE_QUOTE"/></android:nonescaped>
        <android:escaped><xsl:value-of select="$g_ESCAPED_DOUBLE_QUOTE"/></android:escaped>
      </android:entry>
    </android:entries>
  </xsl:variable>


  <!-- *********************************************************************************************
    This function generates a string id for a Android preference value.
    param:  p_attribute
    param:  p_bIsResourceBundleRef
    param:  p_bIsPreferenceAttribute
    param:  p_resourceStringPrefix
    return: A string id if the given attribute
            * is a resource bundle reference
            * is a preference attribute and 
              ** is the 'default' or 'name' attribute of a element 
            Empty string otherwise.
   -->
  <xsl:function name="android:getStringId" as="xsl:string">
      <xsl:param name="p_attribute" 
                 as="attribute()" 
                 required="yes"/>
                 
      <xsl:param name="p_bIsResourceBundleRef" 
                 as="xsl:boolean" 
                 required="yes"/>
                 
      <xsl:param name="p_bIsPreferenceAttribute" 
                 as="xsl:boolean" 
                 required="yes"/>
                 
      <xsl:param name="p_resourceStringPrefix" 
                 as="xsl:boolean" 
                 required="yes"/>

    <xsl:variable name="stringId">

      <xsl:choose>

        <xsl:when test="$p_bIsResourceBundleRef = true()">
          <!-- The given attribute is a reference to a string in a xliff file, which
               may or may not be a ADFmf preference attribute.  Generate a string id
               of the form [PreferenceId].[LoadBundleVariableName].[String_Id] where:
               [PreferenceId]:      is a fully qualified dot-delimited list of id's up to the
                                    element containing the xliff reference.
               [LoadBundleVarName]: is the 'var' attribute value of the <loadBundle> element
               [String_Id]:         is the xliff string id w/o the #{} (or #{''}) delimiters
               NOTE:  Even though the xliff string may not be associated with a preference element
                      a fully qualified preference id is used in order to ensure the returned
                      string id is unique.
          -->
          <xsl:sequence select="concat (adfmf:generatePreferenceId ($p_attribute), 
                                        $g_periodChar, 
                                        adfmf:getLoadBundleVar($p_attribute),
                                        $g_periodChar,
                                        adfmf:getStringResourceId($p_attribute))"/>
        </xsl:when>

        <xsl:when test="$p_bIsPreferenceAttribute = true()">
          <!-- the given attribute is part of a preference element whose value is either hard-coded 
               or is a bundle reference. Generate a string id only when the given attribute:
               1) is the 'default' attribute
               2) is the 'label' attribute
               3) is the id attribute and the preference element has a label 
               attribute that is not a resource bundle reference (if it is,
               it will be processed when the label attribute is processed).
          -->
          <xsl:choose>
            <xsl:when test="name($p_attribute) = $g_adfmfDefaultAttributeName">
            <!-- default attribute -->
                <xsl:sequence select="concat (adfmf:generatePreferenceId ($p_attribute), 
                                              $g_androidAttributeValueSeparator, 
                                              $g_androidDefaultAttributeName)"/>
            </xsl:when>

            <xsl:when test="name($p_attribute) = $g_adfmfLabelAttributeName">
            <!-- label attribute-->
                <xsl:sequence select="concat (adfmf:generatePreferenceId ($p_attribute), 
                                              $g_androidAttributeValueSeparator, 
                                              $g_androidTitleAttributeName)"/>
            </xsl:when>

            <!-- id attribute -->
            <xsl:otherwise>
              <xsl:sequence select="$g_emptyString"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>

        <xsl:otherwise>
          <!-- the given attribute doesn't need to be translated to a resource string -->
          <xsl:sequence select="$g_emptyString"/>
        </xsl:otherwise>
      </xsl:choose>
   </xsl:variable>

   <xsl:value-of select="$stringId"/>
   
  </xsl:function>

  <!-- *********************************************************************************************
      This template drives the creation of three files:
      * preferences.xml - contains Android preferences derived from the source ADFmf application
                          XML.  Also contains feferences to arrays and localizable strings contained
                          in the arrays.xml and strings.xml file.
      * arrays.xml - contains Android list preference elements.  Each ADFmf preferenceList is
                     represented as two Android <string-array> elements (one for enumerating keys
                     and one for enumerating values) in this file.
      * strings.xml - contains Android strings. 
      
      If locale tokens are provided to the XSL, then localized versions of these three files
      get created in locale-specific folders.
   -->
  <xsl:template match="adfmf:application">
      <xsl:variable name="rootxml" select="."/>
      <!-- Create file preferences.xml -->
     <xsl:call-template name="android:createPreferencesXmlFile">
      <xsl:with-param name="p_appElement" select="$rootxml"/>
    </xsl:call-template>
      
      <!-- Create file strings.xml -->
      <xsl:call-template name="android:createStringsXmlFile">
      <xsl:with-param name="p_appElement" select="$rootxml"/>
    </xsl:call-template>

      <!-- Create file arrays.xml -->
      <xsl:call-template name="android:createArraysXmlFile">
      <xsl:with-param name="p_appElement" select="$rootxml"/>
    </xsl:call-template>
   </xsl:template>
   
  <!-- *********************************************************************************************
      Writes the content of a single ADFmf preferenceGroup element as a collection
      of Android preferences elements to the output.
      
      p_prefGroup - the preference group or page whose elements are to be written.
      p_featureId - if the given prefGroup is defined in a feature preference, this is the id of the 
                    feature. Otherwise, it is null/empty.
   -->
  <xsl:template name="android:writePreferenceElements">
      <xsl:param name="p_prefGroup" 
                 as="element (adfmf:preferenceGroup) or element (adfmf:preferencePage)"
                 required="yes"/>
                 
      <xsl:param name="p_featureId" 
                 as="xsl:string" 
                 required="no"/>

      <xsl:call-template name="android:writeKeyAndTitlePreferenceAttributes">
        <xsl:with-param name="p_prefNode" select="$p_prefGroup"/>
        <xsl:with-param name="p_featureId" select="$p_featureId"/>
      </xsl:call-template>

      <xsl:for-each select="$p_prefGroup/adfmf:*">

         <xsl:if test=".[self::adfmf:preferencePage]">

            <xsl:call-template name="android:writePreferencePageElements">
              <xsl:with-param name="p_prefPage" select="."/>
              <xsl:with-param name="p_featureId" select="$p_featureId"/>
            </xsl:call-template>

         </xsl:if>
      
         <xsl:if test=".[self::adfmf:preferenceGroup]">

            <xsl:call-template name="android:writePreferenceCategoryElement">
              <xsl:with-param name="p_prefGroup" select="."/>
              <xsl:with-param name="p_featureId" select="$p_featureId"/>
            </xsl:call-template>

         </xsl:if>

         <xsl:if test=".[self::adfmf:preferenceText]">

            <xsl:call-template name="android:writeAdfMFPreferenceText">
              <xsl:with-param name="p_prefText" select="."/>
              <xsl:with-param name="p_featureId" select="$p_featureId"/>
            </xsl:call-template>

         </xsl:if>

         <xsl:if test=".[self::adfmf:preferenceBoolean]">

            <xsl:call-template name="android:writeAdfMfPreferenceBoolean">
              <xsl:with-param name="p_prefBool" select="."/>
              <xsl:with-param name="p_featureId" select="$p_featureId"/>
            </xsl:call-template>

         </xsl:if>

         <xsl:if test=".[self::adfmf:preferenceNumber]">

            <xsl:call-template name="android:writeAdfMFPreferenceText">
              <xsl:with-param name="p_prefText" select="."/>
              <xsl:with-param name="p_featureId" select="$p_featureId"/>
            </xsl:call-template>

         </xsl:if>

         <xsl:if test=".[self::adfmf:preferenceList]">

            <xsl:call-template name="android:writeAdfMFPreferenceList">
              <xsl:with-param name="p_prefList" select="."/>
              <xsl:with-param name="p_featureId" select="$p_featureId"/>
            </xsl:call-template>

         </xsl:if>

      </xsl:for-each>
   </xsl:template>
   
  <!-- *********************************************************************************************
      Writes the content of a single ADFmf preferencePage element as a collection
      of Android preferences elements to the output.
      
      p_prefPage - the preference page whose elements are to be written.
      p_featureId - if the given prefGroup is defined in a feature preference, this is the id of the
      feature. Otherwise, it is null/empty.
   -->
  <xsl:template name="android:writePreferencePageElements">
    <xsl:param name="p_prefPage" 
               as="element(adfmf:preferencePage)" 
               required="yes"/>
               
    <xsl:param name="p_featureId" 
               select="$p_featureId" 
               as="xsl:string" 
               required="yes"/>
    
    <xsl:call-template name="android:writePreferenceScreenElement">
      <xsl:with-param name="p_prefPage" select="$p_prefPage"/>
      <xsl:with-param name="p_featureId" select="$p_featureId"/>
    </xsl:call-template>
    
   </xsl:template>
   
  <!-- *********************************************************************************************
      Writes the content of the ADFmf preferencePage child elements as a collection
      of Android preferences elements to the output.
      
      p_prefPage - the preference page whose elements are to be written.
      p_featureId - if the given prefGroup is defined in a feature preference, this is the id of the
                    feature. Otherwise, it is null/empty.
   -->
  <xsl:template name="android:writePreferenceScreenElement">
      <xsl:param name="p_prefPage"
                 as="element (adfmf:preferencePage)" 
                 required="yes"/>
                 
      <xsl:param name="p_featureId" 
                 as="xsl:string" 
                 required="no"/>
      
      <PreferenceScreen>
        <xsl:call-template name="android:writePreferenceElements">
          <xsl:with-param name="p_prefGroup" select="$p_prefPage"/>
          <xsl:with-param name="p_featureId" select="$p_featureId"/>
        </xsl:call-template>

     </PreferenceScreen>
   </xsl:template>
   
  <!-- *********************************************************************************************
        Writes the content of a ADFmf preferenceGroup element as a Android
        PreferenceCategory element to the output.
        
        p_prefGroup - the preference group or page whose elements are to be written.
        p_featureId - if the given prefGroup is defined in a feature preference, this is the id of 
                      the feature.  Otherwise, it is null/empty.
   -->
  <xsl:template name="android:writePreferenceCategoryElement">
      <xsl:param name="p_prefGroup" 
                 as="element (adfmf:preferenceGroup)" 
                 required="yes"/>
                 
      <xsl:param name="p_featureId" 
                 as="xsl:string" 
                 required="no"/>
      
      <PreferenceCategory>

        <xsl:call-template name="android:writePreferenceElements">
          <xsl:with-param name="p_prefGroup" select="$p_prefGroup"/>
          <xsl:with-param name="p_featureId" select="$p_featureId"/>
        </xsl:call-template>

      </PreferenceCategory>
   </xsl:template>
   
   
  <!-- *********************************************************************************************
        Writes the content of a ADFmf preferenceText element as a Android AdfMFPreferenceText 
        element to the output.

        p_prefText - to be written as a Android template AdfMFPreferenceText
        p_featureId - if the given prefGroup is defined in a feature preference, this is the id of 
                      the feature. Otherwise, it is null/empty.
   -->
  <xsl:template name="android:writeAdfMFPreferenceText">
      <xsl:param name="p_prefText" 
                 as="element (adfmf:preferenceText)" 
                 required="yes"/>
                 
      <xsl:param name="p_featureId" 
                 as="xsl:string" 
                 required="no"/>
      
      <oracle.adfmf.preferences.AdfMFPreferenceText>
      
        <xsl:call-template name="android:writeKeyAndTitlePreferenceAttributes">
          <xsl:with-param name="p_prefNode" select="$p_prefText"/>
          <xsl:with-param name="p_featureId" select="$p_featureId"/>
        </xsl:call-template>

        <xsl:attribute name="android:defaultValue">
          <xsl:call-template name="android:getTextDefaultValue">
            <xsl:with-param name="p_xmlValue" select="$p_prefText/@default"/>
            <xsl:with-param name="p_id" select="@id"/>
            <xsl:with-param name="p_featureId" select="$p_featureId"/>
          </xsl:call-template>
        </xsl:attribute>
        
        <xsl:if test="@secret">
          <xsl:attribute name="android:password">
            <xsl:value-of select="$p_prefText/@secret"/>
          </xsl:attribute>
        </xsl:if>
    </oracle.adfmf.preferences.AdfMFPreferenceText>
  </xsl:template>
  <!-- *********************************************************************************************
        Writes the content of a ADFmf preferenceBoolean element as a Android 
        AdfMFPreferenceBoolean element to the output.

        p_prefBool  - to be written as a Android template AdfMFPreferenceBoolean
        p_featureId - if the given prefGroup is defined in a feature preference, this is the id of 
                      the feature.  Otherwise, it is null/empty.
   -->
  <xsl:template name="android:writeAdfMfPreferenceBoolean">
      <xsl:param name="p_prefBool" 
                 as="element(adfmf:preferenceBoolean)" 
                 required="yes"/>
                 
      <xsl:param name="p_featureId" 
                 as="xsl:string" 
                 required="no"/>

    <oracle.adfmf.preferences.AdfMFPreferenceBoolean>
        <xsl:call-template name="android:writeKeyAndTitlePreferenceAttributes">
          <xsl:with-param name="p_prefNode" select="$p_prefBool"/>
          <xsl:with-param name="p_featureId" select="$p_featureId"/>
        </xsl:call-template>
      
      <xsl:attribute name="android:defaultValue">
        <xsl:call-template name="android:getBooleanDefaultValue">
          <xsl:with-param name="p_xmlValue" select="$p_prefBool/@default"/>
          <xsl:with-param name="p_id" select="@id"/>
        </xsl:call-template>
      </xsl:attribute>
    </oracle.adfmf.preferences.AdfMFPreferenceBoolean>
  </xsl:template>
  <!-- *********************************************************************************************
        Writes the content of a ADFmf AdfMFPreferenceList element as a Android AdfMFPreferenceList
        element

        p_prefList - to be written as a Android template AdfMFPreferenceList
        p_featureId - if the given prefGroup is defined in a feature preference, this is the id of 
                      the feature.  Otherwise, it is null/empty.
   -->
  <xsl:template name="android:writeAdfMFPreferenceList">
      <xsl:param name="p_prefList" 
                 as="element(adfmf:preferenceList)" 
                 required="yes"/>
                 
      <xsl:param name="p_featureId" 
                 as="xsl:string" 
                 required="no"/>

      <oracle.adfmf.preferences.AdfMFPreferenceList>

      <xsl:call-template name="android:writeKeyAndTitlePreferenceAttributes">
        <xsl:with-param name="p_prefNode" select="$p_prefList"/>
        <xsl:with-param name="p_featureId" select="$p_featureId"/>
      </xsl:call-template>
      
      <xsl:attribute name="android:defaultValue">
        <xsl:call-template name="android:getListDefaultValue">
          <xsl:with-param name="p_xmlValue" select="$p_prefList/@default"/>
          <xsl:with-param name="p_prefList" select="$p_prefList"/>
          <xsl:with-param name="p_id" select="@id"/>
          <xsl:with-param name="p_featureId" select="$p_featureId"/>
        </xsl:call-template>
      </xsl:attribute>
      
      <xsl:attribute name="android:entries">
        <xsl:call-template name="android:entriesAttributeValueName">
          <xsl:with-param name="p_arrayId" select="@id"/>
        </xsl:call-template>
      </xsl:attribute>
      
      <xsl:attribute name="android:entryValues">
        <xsl:call-template name="android:entryValuesAttributeValueName">
          <xsl:with-param name="p_arrayId" select="@id"/>
        </xsl:call-template>
      </xsl:attribute>
    </oracle.adfmf.preferences.AdfMFPreferenceList>
  </xsl:template>
  
  <!-- *********************************************************************************************
      Creates a value to be used as the XML attribute value of a <android:entries> XML element.
      The generated value is of the form "<arrayId>_entries" where <arrayId> is the id of an array.
      The value is used in both preferences.xml and arrays.xml:
      * in preferences.xml the value is a reference to a resource in arrays.xml and has the form
        "@array/<id>" where <id> is the array identifier generated by this template.
      * in arrays.xml the value is used as the value of the 'name' attribute of a <string-array>
        element as in (<id> below is the array identifier generated by this template):
         <string-array name='<id>'>
            <item>...</item>
         </string-array>
         
         param:  p_arrayId - the id value of a preferenceList.
   -->
  <xsl:template name="android:generateEntriesId">
      <xsl:param name="p_arrayId" 
                 as="xsl:string" 
                 required="yes"/>
      
      <!-- The fully qualified dot-path of a ADFmf preference list id -->
      <xsl:variable name="preferenceId" select="adfmf:generatePreferenceId ($p_arrayId)"/>
      
      <!-- The Android attribute name whose value refers to the entries in a list -->
      <xsl:variable name="ENTRIES_ATTRIBUTE" select="'entries'"/>

      <!-- TODO:  Determine why AAPT throws an error if the identifiers for the list preferences are
           "." characters.  Working around the problem by using "_" instead of "." for the
           <arrayId>_entries value -->
           
      <!-- Build an id of the form <preferenceId>___entries (replacing '.' with '_') -->
      <xsl:value-of select="translate (concat ($preferenceId, 
                                               $g_androidAttributeValueSeparator, 
                                               $ENTRIES_ATTRIBUTE),
                                       $g_periodChar, 
                                       $g_underscoreChar)"/>
   </xsl:template>
   
  <!-- *********************************************************************************************
      Creates a value to be used as the XML attribute value of a <android:entries> XML element
      that refers to a value contained in arrays.xml.  The constructed value is of the form:
      "@array/<arrayId>_entries" where <arrayId> is the id of an array.

         param: p_arrayId - - the id of the array.
   -->
  <xsl:template name="android:entriesAttributeValueName">
      <xsl:param name="p_arrayId" 
                 as="xsl:string" 
                 required="yes"/>
                 
      <xsl:variable name="entriesId">
         <xsl:call-template name="android:generateEntriesId">
        <xsl:with-param name="p_arrayId" select="$p_arrayId"/>
      </xsl:call-template>
      </xsl:variable>

      <!-- Contains the prefix of a Android array reference of the form '@array/<endtriesId>' -->
      <xsl:variable name="ARRAY_REFERENCE_PREFIX" select="'@array/'"/>
      
      <!-- create a identifier for the list  -->
      <xsl:value-of select="concat ($ARRAY_REFERENCE_PREFIX, $entriesId)"/>
   </xsl:template>
  <!-- *********************************************************************************************
      Creates a value to be used as the XML attribute value of a <android:entryValues> XML element.
      The generated value is of the form "<arrayId>___entry_values" where <arrayId> is the id of an 
      array.  The value is used in both preferences.xml and arrays.xml:
      * in preferences.xml the value is a reference to a resource in arrays.xml and has the form
        "@array/<id>" where <id> is the array identifier generated by this template.
      * in arrays.xml the value is used as the value of the 'name' attribute of a <string-array>
        element as in (<id> below is the array identifier generated by this template):
         <string-array name='<id>'>
            <item>...</item>
         </string-array>

         param:  p_arrayId - the id value of a preferenceList.
   -->
  <xsl:template name="android:generateEntryValuesId">
      <xsl:param name="p_arrayId" 
                 as="xsl:string" 
                 required="yes"/>
      
      <!-- The fully qualified dot-path of a ADFmf preference list id -->
      <xsl:variable name="preferenceId" select="adfmf:generatePreferenceId ($p_arrayId)"/>

      <!-- The Android attribute name whose value refers to the entry values in a list -->
      <xsl:variable name="ENTRY_VALUES_ATTRIBUTE" select="'entry_values'"/>


      <!-- TODO:  Determine why AAPT throws an error if the identifiers for the list preferences are
           "." characters.  Working around the problem by using "_" instead of "." for the
           <arrayId>___entryValues value -->
      <!-- Build an id of the form <preferenceId>___entryValues (replacing '.' with '_') -->
      <xsl:value-of select="translate (concat ($preferenceId, 
                                               $g_androidAttributeValueSeparator, 
                                               $ENTRY_VALUES_ATTRIBUTE),
                                       $g_periodChar, 
                                       $g_underscoreChar)"/>
   </xsl:template>
  <!-- *********************************************************************************************
      Creates a value to be used as the XML attribute value of a <android:entryValues> XML element
      that refers to a value contained in arrays.xml.  The constructed value is of the form:
      "@array/<arrayId>_entry_values" where <arrayId> is the id of an array.
         preferenceGroupId - the preferenceGroupId if the preference group containing a 
         preferenceList

      param:  p_arrayId - the id of the array.
   -->
  <xsl:template name="android:entryValuesAttributeValueName">
      <xsl:param name="p_arrayId" 
                 as="xsl:string" 
                 required="yes"/>
      
      <xsl:variable name="entryValuesId">
        <xsl:call-template name="android:generateEntryValuesId">
          <xsl:with-param name="p_arrayId" select="$p_arrayId"/>
        </xsl:call-template>
      </xsl:variable>

      <!-- Contains the prefix of a Android array reference of the form '@array/<entryValuesId>' -->
      <xsl:variable name="ARRAY_REFERENCE_PREFIX" select="'@array/'"/>
      
      <!-- create a identifier for the list ensuring replace any spaces with an _ -->
      <xsl:value-of select="concat($ARRAY_REFERENCE_PREFIX, $entryValuesId)"/>
   </xsl:template>
   
  <!-- *********************************************************************************************
        Writes the content of a ADFmf preferenceGroup element as a Android
        * PreferenceScreen element when the depth of the preferenceGroup is a even number.
        * PreferenceCategory element when the depth of the preference group is a odd number.
        
         param:  p_prefGroup - the preference group id of preference group to write.
         param:  p_featureId - name of the feature id if the given preferenceGroup is defined in a 
                               feature..
   -->
  <xsl:template name="android:writePreferenceGroup">
    <xsl:param name="p_prefGroup" 
               as="element (adfmf:preferenceGroup)" 
               required="yes"/>
               
    <xsl:param name="p_featureId" 
               as="xsl:string" 
               required="no"/>
    
    <xsl:call-template name="android:writePreferenceCategoryElement">
      <xsl:with-param name="p_prefGroup" select="."/>
      <xsl:with-param name="p_featureId" select="$p_featureId"/>
    </xsl:call-template>
  </xsl:template>
  <!-- *********************************************************************************************
        Creates a preferences.xml file located in the directory specified as a parameter
        to the XSL and populates it with Android preferences if and only if the ADF Mobile
        application contains preferences.
        
        p_appElement - a adfmf:application containing the application preferences.
   -->
  <xsl:template name="android:createPreferencesXmlFile">
    <xsl:param name="p_appElement" 
               as="element (adfmf:application)" 
               required="yes"/>
    <!-- Folder name to contain the preferences.xml file. -->
    <xsl:variable name="PREFERENCE_XML_FOLDER_NAME" select="'xml'"/>
    
    <xsl:variable name="PREFERENCES_XML" select="'preferences.xml'"/>

    <xsl:variable name="p_filePath" 
                  select="concat ($XSL_PARAM_targetDirectoryPath, 
                                  $PREFERENCE_XML_FOLDER_NAME,
                                  $g_FILE_PATH_SEPARATOR
                                  $PREFERENCES_XML)"/>
                  
    <xsl:if test="adfmf:applicationHasPreferenceElements ($p_appElement, 
                                                          $XSL_PARAM_FeatureArchiveUrlNodeSet) = 
                                                          true()">
    
      <xsl:result-document href="{$p_filePath}" indent="yes">

       <!-- Android preferences XML always has a <PreferenceScreen> root element. -->
       <PreferenceScreen>
          <xsl:namespace name="{$g_nsPrefix}">
            <xsl:value-of select="$g_nsNamespace"/>
          </xsl:namespace>
        
          <!-- write the application level resources -->
          <xsl:call-template name="android:writeApplicationPreferences">
            <xsl:with-param name="p_appElement" select="$p_appElement"/>
          </xsl:call-template>
        
          <!-- write the feature preferences -->
          <xsl:call-template name="android:writeFeaturePreferences"/>
        </PreferenceScreen>
      </xsl:result-document>
    </xsl:if>
  </xsl:template>
  <!-- *********************************************************************************************
        Create a arrays.xml file located in the directory specified as a parameter
        to the XSL and populates it with Android list preferences.  Each ADFmf
        preferenceList element maps to two Android XML elements.  One of the elements
        enumerates the keys of the preference list and the other element enumerates the
        values of the preference list.
        
        p_appElement - a adfmf:application containing the application preferences.
   -->
  <xsl:template name="android:createArraysXmlFile">
      <xsl:param name="p_appElement" 
                 as="element (adfmf:application)" 
                 required="yes"/>

      <xsl:variable name="ARRAYS_XML" select="'arrays.xml'"/>
      <!-- Create the default arrays.xml file -->
      <xsl:call-template name="android:createLocalizedArraysXml">
        <xsl:with-param name="p_fileName" select="$ARRAYS_XML"/>
        <xsl:with-param name="p_localeData"/>
        <xsl:with-param name="p_appElement" select="$p_appElement"/>
      </xsl:call-template>

      <!-- Create a arrays.xml file for each locale -->
      <xsl:for-each select="$XSL_PARAM_LocaleDataElements">
         <xsl:call-template name="android:createLocalizedArraysXml">
          <xsl:with-param name="p_fileName" select="$ARRAYS_XML"/>
          <xsl:with-param name="p_localeData" select="."/>
          <xsl:with-param name="p_appElement" select="$p_appElement"/>
        </xsl:call-template>
      </xsl:for-each>
   </xsl:template>

  <!-- *********************************************************************************************
    return: A path to the specified Android resource file.
   -->
  <xsl:function name="android:getResourceFilePath" as="xsl:string">
    <!-- Resource file name -->
    <xsl:param name="p_fileName" 
               as="xsl:string" 
               required="yes"/>
    
    <!-- This parameter is optional. If the caller specifies it, then the resource file is for a
         particular language, or language and country if the country code is also specified.
    -->
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required="no"/>

    <!-- Name of the folder to contain a default version of strings.xml and arrays.xml -->
    <xsl:variable name="DEFAULT_FOLDER_NAME" select="'values'"/>
    
    <!-- Prefix name of the folder to contain a localized version of strings.xml and arrays.xml.
         Each folder name is of the form values-<languageCode>-[r<countryCode>] where:
         <languageCode> is a 2 character language code and
         [r<countryCode>] is optional and <countryCode> is a country code
    -->
    <xsl:variable name="LOCALIZED_VALUES_FOLDER_NAME_PREFIX" 
                  select="concat ($DEFAULT_FOLDER_NAME, $g_androidLocaleSeparator)"/>
    
    <xsl:variable name="filePath">
      <xsl:choose>
        <xsl:when test="$p_localeData/@androidLocale != ''">
          <!-- $p_localeData/@oracleLocale must also have a non-empty value since androidLocale is 
               non-empty. -->
          <xsl:value-of select="concat ($XSL_PARAM_targetDirectoryPath, 
                                        $LOCALIZED_VALUES_FOLDER_NAME_PREFIX, 
                                        $p_localeData/@androidLocale, 
                                        $g_FILE_PATH_SEPARATOR, 
                                        $p_fileName)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="concat ($XSL_PARAM_targetDirectoryPath, 
                                        $DEFAULT_FOLDER_NAME,
                                        $g_FILE_PATH_SEPARATOR, 
                                        $p_fileName)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:value-of select="$filePath"/>
  </xsl:function>

  <!-- *********************************************************************************************
    This template creates a arrays.xml in a directory whose name indicates it is a localized 
    arrays.xml.
    
    param: p_fileName name of the arrays.xml file
    param: p_localeData an optional LocaleData element.
    param: p_appElement a adfmf:application element
  -->
  <xsl:template name="android:createLocalizedArraysXml">
    <xsl:param name="p_fileName" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required="no"/>
               
    <xsl:param name="p_appElement" 
               as="element (adfmf:application)" 
               required="yes"/>
    
    <xsl:variable name="filePath"
                  select="android:getResourceFilePath($p_fileName, $p_localeData)"/>
    
    <xsl:result-document href="{$filePath}" indent="yes">
      <!-- Android arrays XML always has a <resources> root element. -->
      <resources>

        <!-- write the lists from the adfmf-application.xml file -->
        <xsl:call-template name="android:writeLists">
          <xsl:with-param name="p_appOrFeatureElement" select="$p_appElement"/>
          <xsl:with-param name="p_localeData" select="$p_localeData"/>
          <xsl:with-param name="p_baseXliffPath" select="$XSL_PARAM_applicationXliffBasePath"/>
        </xsl:call-template>
      
        <!-- write the strings for each locale and feature in the adfmf-feature.xml file -->
        <!-- Loop through the FAR JAR URLs and load each adfmf-feature.xml document -->
        <xsl:for-each select="$XSL_PARAM_FeatureArchiveUrlNodeSet"> 
        
          <xsl:variable name="featureArchiveUrl" select="." as="xsl:string"/>      
          <xsl:variable name="featureDoc" select="adfmf:loadFeatureDocument ($featureArchiveUrl)"/>
          <xsl:variable name="featuresElement" select="$featureDoc/adfmf:features"/>

          <!-- Loop through the list of featureReference's contained in adfmf-application.xml -->
          <xsl:for-each select="$gFeatureReferenceSet">

            <!-- featureId from featureReference element -->
            <xsl:variable name="featureId" select="@id"/>

            <!-- Loop through each of the features in the current feature document and,
                 if the feature document has the featureId, write the lists.
            -->
            <xsl:for-each select="$featureDoc/adfmf:features">

              <xsl:variable name="featureElem" select="key('featureKey', $featureId)"/>
              
              <xsl:if test="$featureElem">
                <xsl:call-template name="android:writeLists">
                  <xsl:with-param name="p_appOrFeatureElement" select="$featureElem"/>
                  <xsl:with-param name="p_localeData" select="$p_localeData"/>
                  <xsl:with-param name="p_baseXliffPath" select="$featureArchiveUrl"/>
                </xsl:call-template>
              </xsl:if>              
            </xsl:for-each>
          </xsl:for-each>
        </xsl:for-each>
      </resources>  
    </xsl:result-document>
  </xsl:template>

  <!-- *********************************************************************************************
    This template writes each preferenceList element to arrays.xml 
    
    param: p_appOrFeatureElement - a adfmf:application or adfmf:features element.
    param: p_localeData    - an optional LocaleData element.
    param: p_baseXliffPath - absolute URL path to the directory containing the base .xlif file.
  -->
  <xsl:template name="android:writeLists">
    <xsl:param name="p_appOrFeatureElement" 
               as="element (adfmf:application) or element (adfmf:feature)" 
               required="yes"/>
               
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required="no"/>
               
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string"
               required="no"/>
    
    <xsl:for-each select="$p_appOrFeatureElement//adfmf:preferenceList">
      
      <!-- Write out the 'name' values of each preferenceValue node as a Android <string-array> -->
      <xsl:variable name="prefList" select="."/>

      <xsl:call-template name="android:writeStringArrayElement">
        <xsl:with-param name="p_appOrFeatureElement" select="$p_appOrFeatureElement"/>
        <xsl:with-param name="p_prefList" select="$prefList"/>
        <xsl:with-param name="p_preferenceValueAttributeName" select="$g_adfmfNameAttributeName"/>
        <xsl:with-param name="p_localeData" select="$p_localeData"/>
        <xsl:with-param name="p_baseXliffPath" select="$p_baseXliffPath"/>
      </xsl:call-template>
      
      <!-- Write out the 'value' values of each preferenceValue node as a Android <string-array> -->
      <xsl:call-template name="android:writeStringArrayElement">
        <xsl:with-param name="p_appOrFeatureElement" select="$p_appOrFeatureElement"/>
        <xsl:with-param name="p_prefList" select="$prefList"/>
        <xsl:with-param name="p_preferenceValueAttributeName" select="$g_adfmfValueAttributeName"/>
        <xsl:with-param name="p_localeData" select="$p_localeData"/>
        <xsl:with-param name="p_baseXliffPath" select="$p_baseXliffPath"/>
      </xsl:call-template>

    </xsl:for-each>
  </xsl:template>
  <!-- *********************************************************************************************
      Generates a reference to a string resource.  The reference is used in preferences.xml and
      the strings.xml contains the actual string.
      
      param:  p_stringId the android string id of the string.
   -->
  <xsl:template name="android:generateStringIdReference">
      <xsl:param name="p_stringId" 
                 as="xsl:string" 
                 required="yes"/>
      
      <!-- Contains the prefix of a Android string reference of the form '@string/<stringId>' -->
      <xsl:variable name="STRING_REFERENCE_PREFIX" select="'@string/'"/>

      <xsl:value-of select="concat ($STRING_REFERENCE_PREFIX, $p_stringId)"/>
   </xsl:template>

  <!-- *********************************************************************************************
        Create a strings.xml file located in the directory specified as a parameter
        to the XSL and populates it with Android list preferences.  
        
        The file contains the string identifiers and values for all the ADFmf preferences having a 
        label attribute.
        
        param:  p_appElement - a adfmf:application element.
   -->
  <xsl:template name="android:createStringsXmlFile">
    <xsl:param name="p_appElement" 
               as="element (adfmf:application)" 
               required="yes"/>
    
    <xsl:variable name="STRINGS_XML" select="'strings.xml'"/>
    
    <!-- Create the default strings.xml file -->
    <xsl:call-template name="android:createLocalizedStringsXml">
      <xsl:with-param name="p_fileName" select="$STRINGS_XML"/>
      <xsl:with-param name="p_appElement" select="$p_appElement"/>
      <xsl:with-param name="p_localeData"/>
      <xsl:with-param name="p_baseXliffPath" select="$XSL_PARAM_applicationXliffBasePath"/>
    </xsl:call-template>

      <!-- Create a strings.xml file for each locale -->
      <xsl:for-each select="$XSL_PARAM_LocaleDataElements">
        <xsl:call-template name="android:createLocalizedStringsXml">
          <xsl:with-param name="p_fileName" select="$STRINGS_XML"/>
          <xsl:with-param name="p_appElement" select="$p_appElement"/>
          <xsl:with-param name="p_localeData" select="."/>
          <xsl:with-param name="p_baseXliffPath" select="$XSL_PARAM_applicationXliffBasePath"/>
        </xsl:call-template>
      </xsl:for-each>
   </xsl:template>
   
  <!-- *********************************************************************************************
    This template creates a strings.xml in a directory whose name indicates it is a localized 
    strings.xml.
    
    param: p_fileName - name of the strings.xml file
    param: p_appElement - a adfmf:application element
    param: p_localeData    - an optional LocaleData element.
    param: p_baseXliffPath - absolute URL path to the directory containing the base .xlif file.
  -->
   
  <xsl:template name="android:createLocalizedStringsXml">
    <xsl:param name="p_fileName" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_appElement" 
               as="element (adfmf:application)"/>
               
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required="no"/>
               
    <xsl:param name="p_baseXliffPath"
               as="xsl:string" 
               required="yes"/>

    <xsl:variable name="filePath"
                  select="android:getResourceFilePath($p_fileName, $p_localeData)"/>

    <xsl:result-document href="{$filePath}" indent="yes">
       <!-- Android strings XML always has a <resources> root element. -->
       <resources>
  
        <xsl:call-template name="android:writeApplicationName">
          <xsl:with-param name="p_appElement" select="$p_appElement"/>
          <xsl:with-param name="p_localeData" select = "$p_localeData"/>
          <xsl:with-param name="p_baseXliffPath" select="$p_baseXliffPath"/>
        </xsl:call-template>
  
        <!-- write the strings from the adfmf-application.xml file -->
        <xsl:call-template name="android:writeStrings">
          <xsl:with-param name="p_appOrFeatureElement" select="$p_appElement"/>
          <xsl:with-param name="p_localeData" select="$p_localeData"/>
          <xsl:with-Param name="p_featureId"/>
          <xsl:with-param name="p_baseXliffPath" select="$p_baseXliffPath"/>
        </xsl:call-template>

        <!-- write the strings for each feature the adfmf-feature.xml file -->
        
        <!-- Loop through the FAR JAR URLs and load each adfmf-feature.xml document -->
        <xsl:for-each select="$XSL_PARAM_FeatureArchiveUrlNodeSet">
  
          <xsl:variable name="featureArchiveUrl" select="." as="xsl:string"/>      
          <xsl:variable name="featureDoc" select="adfmf:loadFeatureDocument ($featureArchiveUrl)"/>
          <xsl:variable name="featuresElement" select="$featureDoc/adfmf:features"/>
          <!-- Loop through the list of featureReference's contained in adfmf-application.xml -->
          <xsl:for-each select="$gFeatureReferenceSet">
            
            <!-- featureId from featureReference element -->
            <xsl:variable name="featureId" select="@id"/>
            
            <!-- Loop through each of the features in the current feature document and,
                 if the feature document has the featureId, write the strings.
            -->
            <xsl:for-each select="$featureDoc/adfmf:features">
            
              <xsl:variable name="featureElem" 
                            select="key('featureKey', $featureId)" 
                            as="element(adfmf:feature)"/>
                            
              <!-- Verify the FAR has the feature identified by featureId -->
              <xsl:if test="$featureElem">
                <xsl:call-template name="android:writeStrings">
                  <xsl:with-param name="p_appOrFeatureElement" select="$featureElem"/>
                  <xsl:with-param name="p_localeData" select="$p_localeData"/>
                  <xsl:with-param name="p_featureId" select="$featureId"/>
                  <xsl:with-param name="p_baseXliffPath" select="$featureArchiveUrl"/>
                </xsl:call-template>
              </xsl:if>
            </xsl:for-each> <!-- end for-each feature -->
          </xsl:for-each> <!-- end for-each feature reference -->
        </xsl:for-each> <!-- end for-each FAR -->
      </resources>  
    </xsl:result-document>
  </xsl:template>

  <!-- *********************************************************************************************
    This template writes the application name to strings.xml.  The value for the "name" attribute 
    must be "app_name" as it is written to AndroidManifest.xml at deploy time.  This value is 
    rendered under the application icon.

    param: p_appElement - a adfmf:application element
    param: p_localeData    - an optional LocaleData element.
    param: p_baseXliffPath - absolute URL path to the directory containing the base .xlif file.
  -->
  <xsl:template name="android:writeApplicationName">
    <xsl:param name="p_appElement" 
               as="element (adfmf:application)" 
               required="yes"/>
               
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required ="no"/>
                 
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string" 
               required="yes"/>
    
    <xsl:variable name="bIsResourceBundleRef" 
                  select="adfmf:hasStringResourceReference ($p_appElement/@name)"/>

    <xsl:variable name="appNameValue">
      <xsl:choose>
  
        <xsl:when test="$bIsResourceBundleRef = false()">
          <xsl:value-of select="$p_appElement/@name"/>
        </xsl:when>
  
        <xsl:otherwise>
          <xsl:variable name="xliffValue" 
                        select="adfmf:getStringFromXliffFile ($p_appElement, 
                                                              $p_baseXliffPath,
                                                              $p_appElement/@name, 
                                                              $p_localeData/@oracleLocale)"/>

          <xsl:value-of select="$xliffValue"/>
          
        </xsl:otherwise>
  
      </xsl:choose>
    </xsl:variable>
    
    <xsl:if test="$appNameValue != ''">
      <string>
        <xsl:attribute name="name">app_name</xsl:attribute>
        <xsl:value-of select="$appNameValue"/>
      </string>
    </xsl:if>
  </xsl:template>
  
  <!-- *********************************************************************************************
    This template writes a <string> element to strings.xml if p_stringId and p_stringValue are both 
    not empty
    param: p_appOrFeatureElement - a adfmf:application or adfmf:feature element.
    param: p_stringId - the android string id (may be empty)
    param: p_stringValue - the value (may be empty)
    param: p_bIsResourceBundleRef - whether or not the string value is resource bundle reference.
    param: p_localeData - an optional LocaleData element.
    param: p_baseXliffPath - absolute URL path to the directory containing the base .xlif file.
  -->
  <xsl:template name="android:writeStringElement">
    <xsl:param name="p_appOrFeatureElement" 
               as="element (adfmf:application) or element (adfmf:feature)" 
               required="yes"/>
               
    <xsl:param name="p_stringId" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_stringValue" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_bIsResourceBundleRef" 
               as="xsl:boolean" 
               required="yes"/>
               
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required="no"/>
               
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string" 
               required="yes"/>

    <!-- Attribute name for a Android <string> element in strings.xml -->
    <xsl:variable name="ANDROID_STRING_NAME_ATTRIBUTE" select="'name'"/>

    <xsl:if test="string-length ($p_stringId) > 0 and string-length ($p_stringValue) > 0">
      <xsl:choose>

        <xsl:when test="$p_bIsResourceBundleRef = true()">
          <xsl:variable name="xliffValue" 
                        select="adfmf:getStringFromXliffFile ($p_appOrFeatureElement, 
                                                              $p_baseXliffPath,
                                                              $p_stringValue, 
                                                              $p_localeData/@oracleLocale)"/>

          <xsl:if test="string-length ($xliffValue) > 0">
            <string>
              <xsl:attribute name="{$ANDROID_STRING_NAME_ATTRIBUTE}">
                <xsl:value-of select="$p_stringId"/>
              </xsl:attribute>
              
              <!-- Escape any special characters in the value and write it.  -->
              <xsl:call-template name="android:escapeCharactersInString">
                <xsl:with-param name="p_sourceString" select="$xliffValue"/>
                <xsl:with-param name="p_entryList" 
                                select="$g_ESCAPE_CHARACTER_MAP/android:entries/android:entry"/>
              </xsl:call-template>
              
            </string>
          </xsl:if>
        </xsl:when>
  
        <xsl:otherwise>
          <string>
            <xsl:attribute name="{$ANDROID_STRING_NAME_ATTRIBUTE}">
              <xsl:value-of select="$p_stringId"/>
            </xsl:attribute>
            
            <!-- Escape any special characters in the value and write it.  -->
            <xsl:call-template name="android:escapeCharactersInString">
              <xsl:with-param name="p_sourceString" select="$p_stringValue"/>
              <xsl:with-param name="p_entryList" 
                              select="$g_ESCAPE_CHARACTER_MAP/android:entries/android:entry"/>
            </xsl:call-template>
          </string>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:if>
   </xsl:template>
   
  <!--
      Writes out all the strings to strings.xml
        
      param: p_appOrFeatureElement a adfmf:application or adfmf:feature element
      param: p_localeData - an optional LocaleData element.
      param: p_featureId a feature id if the string belongs to a feature.
      param: p_baseXliffPath - absolute URL path to the directory containing the base .xlif file.
   -->
  <xsl:template name="android:writeStrings">
    <xsl:param name="p_appOrFeatureElement" 
               as="element (adfmf:application) or element (adfmf:feature)" 
               required="yes"/>
               
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required="no"/>
               
    <xsl:param name="p_featureId" 
               as="xsl:string" 
               required="no"/>
               
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string" 
               required="yes"/>

    <xsl:for-each select="$p_appOrFeatureElement/descendant-or-self::node()/@*">

      <xsl:variable name="currAttribute" as="attribute()" select="."/>
      <xsl:variable name="loadBundleVarValue" select="adfmf:getLoadBundleVar ($currAttribute)"/>
      <xsl:variable name="resourceStringPrefix"
                    select="adfmf:createResourceStringPrefix ($loadBundleVarValue, $p_featureId)"/>
      
      <xsl:variable name="bIsResourceBundleRef" 
                    as="xsl:boolean" 
                    select="adfmf:hasStringResourceReference ($currAttribute)"/>
                    
      <xsl:variable name="bIsPreferenceAttribute" 
                    as="xsl:boolean" 
                    select="adfmf:isPreferenceAttribute ($currAttribute)"/>
                    
      <xsl:variable name="stringId" 
                    select="android:getStringId ($currAttribute, 
                                                 $bIsResourceBundleRef, 
                                                 $bIsPreferenceAttribute, 
                                                 $resourceStringPrefix)"/>

        <xsl:variable name="stringValue" as="xsl:string">
          <xsl:choose>
            <!-- if the attribute is a preferenceValue attribute, then do not write the
                 string to strings.xml.  The strings for preferenceValue are written to
                 arrays.xml directly and don't need to be in strings.xml.
            -->
            <xsl:when test="ancestor::adfmf:preferenceValue">
                  <xsl:value-of select="$g_emptyString"/>
            </xsl:when>
            <!-- if the attribute is the 'default' attribute:
                  * skip it if it is a preferenceList attribute having a string resource bundle 
                    reference because the localized list elements are written out when they are 
                    encountered.  If the default is not localized, then we have write a reference to
                    the default value
                  * skip it if it is not a preferenceList or preferenceText default as these are the
                    only preferences that have default values that should be in strings.xml.
            -->
            <xsl:when test="name($currAttribute) = $g_adfmfDefaultAttributeName">
                <xsl:choose>
                  <xsl:when test="ancestor::adfmf:preferenceList and
                                  $bIsResourceBundleRef=true()">
                    <xsl:value-of select="$g_emptyString"/>
                  </xsl:when>
                  <xsl:when test="not(ancestor::adfmf:preferenceList) and
                                  not(ancestor::adfmf:preferenceText)">
                      <xsl:value-of select="$g_emptyString"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="$currAttribute"/>
                  </xsl:otherwise>
                  
                </xsl:choose>
            </xsl:when>
            <xsl:when test="name($currAttribute) = $g_adfmfIdAttributeName and ../@label">
              <xsl:value-of select="../@label"/>
            </xsl:when>
            
            <xsl:otherwise>
              <xsl:value-of select="$currAttribute"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        
        <xsl:call-template name="android:writeStringElement">
          <xsl:with-param name="p_appOrFeatureElement" select="$p_appOrFeatureElement"/>
          <xsl:with-param name="p_stringId" select="$stringId"/>
          <xsl:with-param name="p_stringValue" select="$stringValue"/>
          <xsl:with-param name="p_bIsResourceBundleRef" select="$bIsResourceBundleRef"/>
          <xsl:with-param name="p_localeData" select="$p_localeData"/>
          <xsl:with-param name="p_baseXliffPath" select="$p_baseXliffPath"/>
      </xsl:call-template>
    </xsl:for-each>
   </xsl:template>
  <!-- *********************************************************************************************
         This template writes a ADFmf preferenceGroup as a Android PreferenceScreen or 
         PreferenceCategory.
         
         param:  p_appElement a adfmf:application element whose preferences are to be written to 
                 preferences.xml.
         
         If a given ADFmf attribute value is a resource bundle reference, the constructed 
         preferences.xml will contain a android-style reference to a localizable string 
         contained in strings.xml.  
         
         This template does not:
         1) transform any resource bundle references that are part of non-preference constructs.  
         2) construct strings.xml file(s) containing the localized strings.
   -->
  <xsl:template name="android:writeApplicationPreferences">
    <xsl:param name="p_appElement" 
               as="element (adfmf:application)" 
               required="yes"/>

    <xsl:for-each select="$p_appElement/adfmf:preferences/adfmf:*">
      <xsl:choose>
        <xsl:when test=".[self::adfmf:preferenceGroup]">
          <!-- Transform the current preference group elements to Android counterparts.-->
          <xsl:call-template name="android:writePreferenceGroup">
            <xsl:with-param name="p_prefGroup" select="."/>
            <xsl:with-param name="p_featureId"/>
          </xsl:call-template>
        </xsl:when>
      </xsl:choose>
    </xsl:for-each>
   </xsl:template>

  <!-- *********************************************************************************************
      Writes a ADFmf preferenceGroup contained in adfmf-feature.xml as a Android PreferenceScreen or 
      PreferenceCategory.
   -->
  <xsl:template name="android:writeFeaturePreferences">

    <!-- Loop through the FAR JAR URLs and load each adfmf-feature.xml document -->
    <xsl:for-each select="$XSL_PARAM_FeatureArchiveUrlNodeSet">
    
      <xsl:variable name="featureArchiveUrl" select="." as="xsl:string"/>      
      <xsl:variable name="featureDoc" select="adfmf:loadFeatureDocument ($featureArchiveUrl)"/>
      <xsl:variable name="featuresElement" select="$featureDoc/adfmf:features"/>
      
      <!-- Loop through the list of featureReference's contained in adfmf-application.xml -->
      <xsl:for-each select="$gFeatureReferenceSet">

        <!-- featureId from featureReference element -->
        <xsl:variable name="featureId" select="@id"/>
   
        <!-- Loop through each of the features in the current feature document and,
             if the feature document has the featureId, write the feature preferences.
        -->
        <xsl:for-each select="$featureDoc/adfmf:features">
        
          <xsl:variable name="featureElem" select="key('featureKey', $featureId)"/>
          
          <xsl:if test="$featureElem">
             <!-- Loop through each preference group and write it -->
             <xsl:for-each select="$featureElem/adfmf:preferences/adfmf:preferenceGroup">
                <xsl:choose>
                  <xsl:when test=".[self::adfmf:preferenceGroup]">
                    <!-- Transform the current preference group elements to Android counterparts.-->
                    <xsl:call-template name="android:writePreferenceGroup">
                      <xsl:with-param name="p_prefGroup" select="."/>
                      <xsl:with-param name="p_featureId" select="$featureId"/>
                    </xsl:call-template>
                  </xsl:when>
                </xsl:choose>
            </xsl:for-each>
          </xsl:if>
        </xsl:for-each>
      </xsl:for-each>
    </xsl:for-each>
   </xsl:template>
  <!-- *********************************************************************************************
      Builds the default value for a text preference.

      This template specifies a value of an empty string if the given xmlValue is not defined.  
      Otherwise, iT specifies the given xmlValue for the android:defaultValue value.
      
      p_xmlValue        - a value from the adfmf_application.xml or adfmf_feature.xml containing the
                          value of the "default" attribute of the ADFmf preferenceText.  May be 
                          null/empty.
      p_id              - the id of the preferenceText
      p_featureId - the name of the feature if the given p_xmlValue belongs to a feature.
   -->
  <xsl:template name="android:getTextDefaultValue">
    <xsl:param name="p_xmlValue" 
               as="attribute()" 
               required="no"/>
               
    <xsl:param name="p_id" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_featureId" 
               as="xsl:string"
               required="yes"/>

    <xsl:variable name="bIsResourceBundleRef" as="xsl:boolean"
                  select="adfmf:hasStringResourceReference ($p_xmlValue)"/>

      <xsl:choose>
        <xsl:when test="$p_xmlValue != $g_emptyString">
          <!-- 
            Specify string reference to a string in strings.xml only if the value is associated with
            a  preferenceText preference.
            This template is invoked for getting a default value for a preferenceNumber which should
            not be a string reference to string in strings.xml
          -->
          <xsl:choose>
            <xsl:when test="$p_xmlValue/ancestor::adfmf:preferenceText">
                  <xsl:variable name="loadBundleVarValue" 
                                select="adfmf:getLoadBundleVar ($p_xmlValue)"/>
                  <xsl:variable name="resourceStringPrefix"
                    select="adfmf:createResourceStringPrefix ($loadBundleVarValue, $p_featureId)"/>

              <xsl:variable name="stringId"
                            select="android:getStringId ($p_xmlValue, 
                                                         $bIsResourceBundleRef, 
                                                         true(), 
                                                         $resourceStringPrefix)"/>
        
              <xsl:variable name="stringIdReference">
                <xsl:call-template name="android:generateStringIdReference">
                  <xsl:with-param name="p_stringId" select="$stringId"/>
                </xsl:call-template>
              </xsl:variable>
    
              <xsl:value-of select="$stringIdReference"/>
            </xsl:when>
            
            <xsl:otherwise>
              <xsl:value-of select="$p_xmlValue"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$g_emptyString"/>
        </xsl:otherwise>
      </xsl:choose>  
   </xsl:template>
  <!-- *********************************************************************************************
      Builds the default value for a boolean preference.  
      
      This template specifies a value of "false" if the given xmlValue is not defined.  Otherwise,
      is specifies the given xmlValue for the android:defaultValue value.
      
      p_xmlValue        - a value from the adfmf_application.xml or adfmf_feature.xml containing the
                          value of the "default" attribute of the ADFmf preferenceBoolean.  May be 
                          null/empty.
      p_id              - the id of the preferenceBoolean
   -->
  <xsl:template name="android:getBooleanDefaultValue">
    <xsl:param name="p_xmlValue" 
               as="xsl:string" 
               required="no"/>
               
    <xsl:param name="p_id"
               as="xsl:string" 
               required="yes"/>

    <xsl:choose>
      <xsl:when test="$p_xmlValue">
        <xsl:value-of select="$p_xmlValue"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="'false'"/>
      </xsl:otherwise>
    </xsl:choose>
   </xsl:template>
  <!-- *********************************************************************************************
      Builds the default value for a list preference.  The adfmf:preferenceList element defines the 
      "default" attribute to be one of the "value" members of the preferenceList's child 
      adfmf:preferenceValue  elements.  
      
      This template iterates the given preferenceList to find the value and specify the 
      corresponding "name" attribute.  If one is not found, an empty string is specified.
      
      p_xmlValue        - a value from the adfmf_application.xml or adfmf_feature.xml containing the
                          value of the "default" attribute of the ADFmf preferenceList.  May be 
                          null/empty.
      p_prefList        - a adfmf:preferenceList element
      p_id              - the id of the preferenceText
      p_featureId - the name of the feature if the given p_xmlValue belongs to a feature.
      
   -->
  <xsl:template name="android:getListDefaultValue">
    <xsl:param name="p_xmlValue" 
               as="attribute" 
               required="no"/>
    
    <xsl:param name="p_prefList" 
               as="element (adfmf:preferenceList)" 
               required="yes"/>
               
    <xsl:param name="p_id" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_featureId" 
               as="xsl:string" 
               required="yes"/>
      
    <xsl:variable name="defaultValue">
      <xsl:choose>
  
        <xsl:when test="string-length ($p_prefList/@default) > 0">
          <xsl:variable name="bIsResourceBundleRef" as="xsl:boolean"
                        select="adfmf:hasStringResourceReference ($p_prefList/@default)"/>
    
          <xsl:variable name="loadBundleVarValue" select="adfmf:getLoadBundleVar ($p_xmlValue)"/>
          <xsl:variable name="resourceStringPrefix"
                    select="adfmf:createResourceStringPrefix ($loadBundleVarValue, $p_featureId)"/>

          <xsl:variable name="stringId"
                        select="android:getStringId ($p_prefList/@default, 
                                                     $bIsResourceBundleRef, 
                                                     true(), 
                                                     $resourceStringPrefix)"/>

          <xsl:variable name="stringIdReference">
            <xsl:call-template name="android:generateStringIdReference">
              <xsl:with-param name="p_stringId" select="$stringId"/>
            </xsl:call-template>
          </xsl:variable>

        <xsl:value-of select="$stringIdReference"/>
      </xsl:when>
  
      <xsl:otherwise>
        <xsl:value-of select="$g_emptyString"/>
      </xsl:otherwise>
    
      </xsl:choose>
      
    </xsl:variable>    
    <xsl:value-of select="$defaultValue"/>
   </xsl:template>
   
     <!-- ******************************************************************************************
      This template writes the android:key and android:title attributes to preferences.xml
      
      p_prefNode - a preference node
      p_featureId - the name of the feature if the given p_prefNode belongs to a feature.
   -->
   <xsl:template name="android:writeKeyAndTitlePreferenceAttributes">
      <xsl:param name="p_prefNode" 
                 as="element (adfmf:preferenceText) or 
                     element (adfmf:preferenceBoolean) or
                     element (adfmf:preferenceNumber) or
                     element (adfmf:preferenceList) or
                     element (adfmf:preferencePage) or
                     element (adfmf:preferenceGroup)" 
                     required="yes"/>
                     
      <xsl:param name="p_featureId" 
                 as="xsl:string" 
                 required="no"/>

        <xsl:attribute name="android:key">
          <xsl:value-of select="adfmf:generatePreferenceId ($p_prefNode)"/>
        </xsl:attribute>

        <xsl:variable name="bIsResourceBundleRef" as="xsl:boolean"
                      select="adfmf:hasStringResourceReference ($p_prefNode/@label)"/>

        <!-- The preference node 'label' attribute value maps to the android:title
             attribute.  Output the title only if a label exists. -->
        <xsl:if test="$p_prefNode/@label and string-length ($p_prefNode/@label) > 0">
          <xsl:variable name="loadBundleVarValue" 
                        select="adfmf:getLoadBundleVar ($p_prefNode/@label)"/>
                        
          <xsl:variable name="resourceStringPrefix"
                        select="adfmf:createResourceStringPrefix ($loadBundleVarValue, 
                                                                  $p_featureId)"/>
          <xsl:variable name="stringId"
                        select="android:getStringId ($p_prefNode/@label, 
                                                     $bIsResourceBundleRef, 
                                                     true(), 
                                                     $resourceStringPrefix)"/>
      
          <xsl:variable name="stringRef">
            <xsl:call-template name="android:generateStringIdReference">
              <xsl:with-param name="p_stringId" select="$stringId"/>
            </xsl:call-template>
          </xsl:variable>
      
          <xsl:attribute name="android:title">
            <xsl:value-of select="$stringRef"/>
          </xsl:attribute>
        </xsl:if>
   </xsl:template>

  <!-- *********************************************************************************************
      This template writes a android <string-array> element to a arrays.xml file for the given
      prefList preference list.  Each adfmf:preferenceList is represented in arrays.xml as two 
      Android <string-array> elements where one <string-array> element contains the entries in the 
      list and the other <string-array> element contains the entry values.  
      
      Note that:
      * the adfmf:preferenceValue/@name maps to a <string-array> element whose name ends with, 
        "___entries", and the elements in this list may be localized.
        
      * the adfmf:preferenceValue/@value maps to a <string-array> eleement whose name ends with, 
        "___entry_values", and the elements in this list should not be localized.
      
      param: p_appOrFeatureElement  - a adfmf:application or adfmf:feature element
  
      param: p_prefList - a adfmf:preferenceList whose adfmf:preferenceValue elements are to be 
                          written as a android <string-array> element.
      
      param: p_preferenceValueAttributeName - the name of the adfmf preferenceValue attribute to use 
                                              when writing a adfmf  preferenceValue.  Can be 'name' 
                                              or 'value'. 
      
      param: p_localeData an optional LocaleData element.

      param: p_baseXliffPath - absolute URL path to the directory containing the base .xlif file.
      
   -->
  <xsl:template name="android:writeStringArrayElement">
    <xsl:param name="p_appOrFeatureElement" 
               as="element (adfmf:application) or element (adfmf:feature)" 
               required="yes"/>
               
    <xsl:param name="p_prefList" 
               as="element(adfmf:listPreference)" 
               required="yes"/>
               
    <xsl:param name="p_preferenceValueAttributeName" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_localeData" 
               as="element(LocaleData)" 
               required="no"/>
               
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string" 
               required="yes"/>

      <string-array>

        <xsl:variable name="nameAttributeValue">
          <xsl:choose>
            <xsl:when test="$p_preferenceValueAttributeName = $g_adfmfNameAttributeName">
              <xsl:call-template name="android:generateEntriesId">
                <xsl:with-param name="p_arrayId" select="$p_prefList/@id"/>
              </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="android:generateEntryValuesId">
                <xsl:with-param name="p_arrayId" select="$p_prefList/@id"/>
              </xsl:call-template>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>

        <xsl:attribute name="{$g_adfmfNameAttributeName}">
          <xsl:value-of select="$nameAttributeValue"/>
        </xsl:attribute>
    
        <xsl:for-each select="$p_prefList/adfmf:preferenceValue">
          <item>
            <xsl:variable name="itemAttribute">
              <xsl:choose>

                <xsl:when test="$p_preferenceValueAttributeName = 'name'">
                  <xsl:value-of select="@name"/>
                </xsl:when>

                <xsl:otherwise>
                  <xsl:value-of select="@value"/>
                </xsl:otherwise>
                
              </xsl:choose>
            </xsl:variable>
            
            <xsl:variable name="bIsResourceBundleRef" as="xsl:boolean"
                          select="adfmf:hasStringResourceReference ($itemAttribute)"/>
            
            <xsl:choose>
              <xsl:when test="$bIsResourceBundleRef = true()">

                <xsl:variable name="xliffValue" 
                              select="adfmf:getStringFromXliffFile ($p_appOrFeatureElement, 
                                                                    $p_baseXliffPath,
                                                                    $itemAttribute, 
                                                                    $p_localeData/@oracleLocale)"/>
                
                <xsl:if test="$xliffValue != ''">
                  <!-- Escape any special characters in the value and write it. -->
                  <xsl:call-template name="android:escapeCharactersInString">
                    <xsl:with-param name="p_sourceString" select="$xliffValue"/>
                    <xsl:with-param name="p_entryList" 
                                    select="$g_ESCAPE_CHARACTER_MAP/android:entries/android:entry"/>
                  </xsl:call-template>
                </xsl:if>
              </xsl:when>
              <xsl:otherwise>
                  <!-- Escape any special characters in the value and write it. -->
                  <xsl:call-template name="android:escapeCharactersInString">
                    <xsl:with-param name="p_sourceString" select="$itemAttribute"/>
                    <xsl:with-param name="p_entryList" 
                                    select="$g_ESCAPE_CHARACTER_MAP/android:entries/android:entry"/>
                  </xsl:call-template>
              </xsl:otherwise>
            </xsl:choose>
          </item>
        </xsl:for-each>
      </string-array>
  </xsl:template>
  
  <!-- *********************************************************************************************
    This template searches the given p_sourceString for characters (represented
    as strings) that need to be escaped and replaces the characters with the
    escaped version.
    
    param: p_sourceString - a string to be searched for sub-strings that need to be escaped
    param: p_entryList - a 'list' of android:entry elements mapping non-escaped 
                         character (as a string) to it's escaped equivalent.  See
                         definition of global variable g_ESCAPE_CHARACTER_MAP for
                         more information on how the values are mapped.
  -->
  <xsl:template name="android:escapeCharactersInString">
    <xsl:param name="p_sourceString" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_entryList" 
               as="element (android:entry)" 
               required="yes"/>

    <xsl:choose>

      <xsl:when test="$p_entryList">
      
        <!-- Using the first android:entry in the list, replace the 
             non-escaped string with the escaped string and save it
             in a variable.  -->
        <xsl:variable name="newString">
          <xsl:call-template name="adfmf:string-replace-all">
            <xsl:with-param name="p_sourceString" select="$p_sourceString"/>
            <xsl:with-param name="p_searchString" select="$p_entryList[1]/android:nonescaped"/>
            <xsl:with-param name="p_replaceString" select="$p_entryList[1]/android:escaped"/>
          </xsl:call-template>
        </xsl:variable>

        <!-- recursively call the template passing:
             1) the android:entry element(s) after the first android:entry in the list 
             2) the new string created above
        -->
        <xsl:call-template name="android:escapeCharactersInString">
          <xsl:with-param name="p_sourceString" select="$newString"/>
          <xsl:with-param name="p_entryList" select="$p_entryList[position () > 1]"/> 
        </xsl:call-template>
      </xsl:when>
        
      <xsl:otherwise>
        <!-- Traversed the entire list.  Output the current source string. -->
        <xsl:value-of select="$p_sourceString"/>
      </xsl:otherwise>

    </xsl:choose>
  </xsl:template>
   
</xsl:stylesheet>
