/*
 * Copyright 2003 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

/*
 * @(#)DataSource.java	1.7 03/10/21
 */

package javax.sql;

import java.sql.Connection;
import java.sql.SQLException;

/**
 *
 * <p>A factory for connections to the physical data source that this
 * <code>DataSource</code> object represents.  A replacement for the
 * <code>DriverManager</code> facility, a <code>DataSource</code> object
 * is the preferred means of getting a connection.
 * <P>
 * The <code>DataSource</code> interface is implemented by a driver vendor.
 * <P>
 * A <code>DataSource</code> object has properties that can be modified
 * when necessary.  For example, if the data source is moved to a different
 * server, the property for the server can be changed.  The benefit is that
 * because the data source's properties can be changed, any code accessing
 * that data source does not need to be changed.
 * <P>
 * An instance of a <code>DataSource</code> object can be used in a stand
 * alone program to create <code>Connection</code> objects. In the following
 * example an instance of <code>DataSource</code>, in this case,
 * <code>VendorDataSource</code> is used to create a <code>Connection</code>
 * to a database on the machine <code>bookserver</code> which is listening at
 * port <code>12345</code>:
 * <PRE>
 *   VendorDataSource vds = new VendorDataSource();
 *   vds.setServerName("bookserver");
 *   vds.setPortNumber(12345);
 *   Connection conn = vds.getConnection("Bob", "passwd");
 * </PRE>
 * A list of standard properties is provided in seciton 9.3.1 of the JDBC
 * 3.0 specification. Consult your vendors documentation for a list the
 * supported properties for you <code>DataSource</code>.
 *
 * Further documentation on dynamically setting <code>DataSource</code>
 * properties is provided <a href="../../datasource-properties.html">here</a>.
 *
 * @see java.sql.Connection
 * @since 1.4
 */
public interface DataSource {
    
    /**
     * <p>Attempts to establish a connection with the data source that
     * this <code>DataSource</code> object represents.
     *
     * @return  a connection to the data source
     * @exception SQLException if a database access error occurs
     */
    Connection getConnection() throws SQLException;
    
    /**
     * <p>Attempts to establish a connection with the data source that
     * this <code>DataSource</code> object represents.
     *
     * @param username the database user on whose behalf the connection is 
     *  being made
     * @param password the user's password
     * @return  a connection to the data source
     * @exception SQLException if a database access error occurs
     */
    Connection getConnection(String username, String password) throws SQLException;
    
    /**
     * <p>Retrieves the log writer for this <code>DataSource</code>
     * object.
     *
     * <p>The log writer is a character output stream to which all logging
     * and tracing messages for this data source will be
     * printed.  This includes messages printed by the methods of this
     * object, messages printed by methods of other objects manufactured
     * by this object, and so on.  Messages printed to a data source
     * specific log writer are not printed to the log writer associated
     * with the <code>java.sql.Drivermanager</code> class.  When a
     * <code>DataSource</code> object is
     * created, the log writer is initially null; in other words, the
     * default is for logging to be disabled.
     *
     * @return the log writer for this data source or null if
     *        logging is disabled
     * @exception SQLException if a database access error occurs  
     * @see #setLogWriter
     */
    java.io.PrintWriter getLogWriter() throws SQLException;
    
    /**
     * <p>Sets the log writer for this <code>DataSource</code>
     * object to the given <code>java.io.PrintWriter</code> object.
     *
     * <p>The log writer is a character output stream to which all logging
     * and tracing messages for this data source will be
     * printed.  This includes messages printed by the methods of this
     * object, messages printed by methods of other objects manufactured
     * by this object, and so on.  Messages printed to a data source-
     * specific log writer are not printed to the log writer associated
     * with the <code>java.sql.Drivermanager</code> class. When a 
     * <code>DataSource</code> object is created the log writer is
     * initially null; in other words, the default is for logging to be
     * disabled.
     *
     * @param out the new log writer; to disable logging, set to null
     * @exception SQLException if a database access error occurs  
     * @see #getLogWriter
     */
    void setLogWriter(java.io.PrintWriter out) throws SQLException;
    
    /**
     * <p>Sets the maximum time in seconds that this data source will wait
     * while attempting to connect to a database.  A value of zero
     * specifies that the timeout is the default system timeout 
     * if there is one; otherwise, it specifies that there is no timeout.
     * When a <code>DataSource</code> object is created, the login timeout is
     * initially zero.
     *
     * @param seconds the data source login time limit
     * @exception SQLException if a database access error occurs.
     * @see #getLoginTimeout
     */
    void setLoginTimeout(int seconds) throws SQLException;
    
    /**
     * Gets the maximum time in seconds that this data source can wait
     * while attempting to connect to a database.  A value of zero
     * means that the timeout is the default system timeout 
     * if there is one; otherwise, it means that there is no timeout.
     * When a <code>DataSource</code> object is created, the login timeout is
     * initially zero.
     *
     * @return the data source login time limit
     * @exception SQLException if a database access error occurs.
     * @see #setLoginTimeout
     */
    int getLoginTimeout() throws SQLException;
}
