<?xml version="1.0" encoding="windows-1252" ?>
<!--
This XSL document contains utility type XSL templates, functions, and global variables that can be 
shared by the iOS and Android deployment.
--> 
<xsl:stylesheet version="2.0" 
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:adfmf="http://xmlns.oracle.com/adf/mf"
                xmlns:xlf="urn:oasis:names:tc:xliff:document:1.1"
                xmlns:FileUtils=
                      "http://www.oracle.com/XSL/Transform/java/oracle.adfmf.common.util.FileUtils">
                
  <xsl:variable name="g_XLIFF_FILE_EXTENSION" 
                select="'.xlf'" 
                as="xsl:string"/>
                
  <xsl:variable name="g_LANGUAGE_CODE_SEPARATOR" 
                select="'_'" 
                as="xsl:string"/>
                
  <xsl:variable name="g_TARGET_STRING_RESOURCE_FILE_NAME" 
                select="'Localizable.strings'" 
                as="xsl:string"/>
                
  <xsl:variable name="g_STRING_RESOURCE_DELIMITER_START" 
                select="'#{'" 
                as="xsl:string"/>
                
  <xsl:variable name="g_STRING_RESOURCE_DELIMITER_END" 
                select="'}'" 
                as="xsl:string"/>
                
  <xsl:variable name="g_STRING_RESOURCE_DELIMITOR_MIDDLE" 
                select="'.'" 
                as="xsl:string"/>
                
  <xsl:variable name="g_FILE_PATH_SEPARATOR" 
                select="'/'" 
                as="xsl:string"/>
  
  <!--
  Dot separator character to use between tokens.
  TODO: This constant is copied (without change) from iOS XSL file "PreferenceTransform.xsl"
        and also is duplicated in iOS XSL file "TransformStringResources.xsl".  So the iOS
        should be refactored to use this constant if/when it is updated to use this 
        TransformUtilities.xsl file.
  -->
  <xsl:variable name="g_TOKEN_DELIMITOR_DOT" select="'.'" as="xsl:string"/>

  <!--
  Constant string literal that represents the prefix [' to a decorated
  string ID.
  TODO: This constant is copied (without change) from iOS XSL file "PreferenceTransform.xsl"
        and also is duplicated in iOS XSL file "TransformStringResources.xsl".  So the iOS
        should be refactored to use this constant if/when it is updated to use this 
        TransformUtilities.xsl file.
  -->
  <xsl:variable name="g_PREFIX_TO_STRING_ID_THAT_CONTAINS_DOTS"
                as="xsl:string">
    <!--
    Initializing a string with an embedded apostrophe can be complicated when using
    using "xsl:variable/@select". Therefore, we use xsl:text to simplify how the string
    is specified.  This technique of using xsl:text to simplify a string definition having
    embedded quotes is taken from page 533 of book "XSLT 2.0 and XPath 2.0",
    4th edition, Programmer's Reference, by Michael Kay.
    -->
    <xsl:text>['</xsl:text>
  </xsl:variable>

  <!--
  Constant string literal that represents the suffix '] to a decorated
  string ID.
  TODO: This constant is copied (without change) from iOS XSL file "PreferenceTransform.xsl"
        and also is duplicated in iOS XSL file "TransformStringResources.xsl".  So the iOS
        should be refactored to use this constant if/when it is updated to use this 
        TransformUtilities.xsl file.
  -->          
  <xsl:variable name="g_SUFFIX_TO_STRING_ID_THAT_CONTAINS_DOTS"
                as="xsl:string">
    <!--
    Initializing a string with an embedded apostrophe can be complicated when using
    using "xsl:variable/@select". Therefore, we use xsl:text to simplify how the string
    is specified.  This technique of using xsl:text to simplify a string definition having
    embedded quotes is taken from page 533 of book "XSLT 2.0 and XPath 2.0",
    4th edition, Programmer's Reference, by Michael Kay.
    -->
    <xsl:text>']</xsl:text>
  </xsl:variable>
  <!--
  Prefix token that must appear first for any iOS preference element
  or iOS string resource that originates from adfmf:application.
  This constant is duplicated in file "TransformStringResources.xsl".
  -->
  <xsl:variable name="g_APPLICATION_CONTEXT_TOKEN" 
                select="'application'" 
                as="xsl:string"/>
  
  <!--
  Prefix token that must appear first for any iOS preference element
  or iOS string resource that originates from adfmf:feature.
  This constant is duplicated in file "TransformStringResources.xsl".
  -->
  <xsl:variable name="g_FEATURE_CONTEXT_TOKEN" 
                select="'feature'" 
                as="xsl:string"/>
  
  <!-- Application token that must preceded each resource string identifier that comes from 
       "adfmf-application.xml". 
  -->
  <xsl:variable name="g_APPLICATION_PREFIX" 
                select="'application'" 
                as="xsl:string"/>

  <!-- Feature token that must preceded each resource string identifier that comes from 
       "adfmf-application.xml". 
  -->
  <xsl:variable name="g_FEATURE_PREFIX" select="'feature'" as="xsl:string"/>

  <!-- Use this key to find a particular trans-unit with a specified @id attribute value -->
  <xsl:key name="xliffStringKey" 
           match="xlf:trans-unit" 
           use="@id"/>

  <!-- Use this key to fina a particular feature with a specified @id attribute value -->
  <xsl:key name="featureKey" 
           match="adfmf:feature" 
           use="@id"/>
  
   <!-- Get the set of adfmf:featureReference elements, in document order, from file 
        "adfmf-application.xml".  We know that adfmf:featureReference elements are direct 
        descendents of the outtermost element. Therefore, we use prefix "/*/" to ensure efficiency 
        because using select value "//adfmf:featureReference" might result in a search of the entire
        document, depending on the xslt parser.
   -->
  <xsl:variable name="gFeatureReferenceSet" select="/*/adfmf:featureReference"/>

  <!-- *********************************************************************************************
    This function returns true if the given attribute start with
    #{ and ends with }
    
    param:  p_attribute - a XML attribute to test if it has a string resource reference
    return: true if the given attribute starts with #{ and ends with }
            false otherwise.
  -->
  <xsl:function name="adfmf:hasStringResourceReference" as="xsl:boolean">
    <xsl:param name="p_attribute" 
               as="attribute()" 
               required="yes"/>  
    <xsl:choose>
      <xsl:when test="starts-with($p_attribute, $g_STRING_RESOURCE_DELIMITER_START) and 
                      ends-with($p_attribute, $g_STRING_RESOURCE_DELIMITER_END)">
        <xsl:sequence select="true()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:sequence select="false()"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>

  <!-- *********************************************************************************************
    This function loads the adfmf-feature.xml specified by the given URL as a 
    xsl:document-node
    param:  p_featureArchiveUrl a path to the location of a FAR JAR file.  Must
                                be a jar style path.
    return: a xsl:document-node() representation of the adfmf-feature.xml file.
  -->
  <xsl:function name="adfmf:loadFeatureDocument" as="document-node()">
    <xsl:param name="p_featureArchiveUrl" 
               as="xsl:string" 
               required="yes"/>

    <xsl:variable name="featureXmlUrl" 
                  select="concat($p_featureArchiveUrl, $XSL_PARAM_FeatureXmlRelativePathInFAR)"/>
    
    <xsl:if test="FileUtils:doesUrlResourceExist($p_featureArchiveUrl) = true()"> 
      <xsl:sequence select="document($featureXmlUrl)"/>
    </xsl:if>
  </xsl:function>

  <!-- *********************************************************************************************
  Return:  Returns a string that represents a fully qualified preference id.
  TODO: This function is copied (without change) from iOS XSL file "PreferenceTransform.xsl"
        So the iOS XSL should be refactored to use this method if/when it is updated to use this 
        TransformUtilities.xsl file.
  
  -->
  <xsl:function name="adfmf:generatePreferenceId" as="xsl:string">
    <!-- This parameter specifies the element whose identifier will be generated. -->
    <xsl:param name="elementToIdentify" 
               as="element(adfmf:*)"/>

    <!-- Get the first token to use for the identifier. -->
    <xsl:variable name="prefixToken" as="xsl:string">
      <xsl:choose>
        <xsl:when test="$elementToIdentify/ancestor::adfmf:application">
          <!-- The element is from adfmf-application.xml -->
          <xsl:sequence select="$g_APPLICATION_CONTEXT_TOKEN"/>
        </xsl:when>
        <xsl:otherwise>
          <!-- The element is from adfmf-feature.xml -->
          <xsl:sequence select="$g_FEATURE_CONTEXT_TOKEN"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- Get the set of elements that are used to build the fully
         qualified identifier. It includes all necessary parent preference
         elements as well as the target element itself.  It's OK that
         element adfmf-preferences is included in the set because it doesn't
         have an "id" attribute and so it won't be included in the final
         identifier string that is built.
    -->
    <xsl:variable name="parentPreferenceElementsAndSelf" as="node-set">
      <!-- The result of this is to produce a document that contains the elements we want. -->
      <xsl:choose>
        <xsl:when test="$elementToIdentify/ancestor::adfmf:application">
          <!--
          Get all ancestor-or-self elements in document order except for element adfmf:application.
          BUGBUG:
          There appears to be a bug in the XSLT processor because this call to ancestor-or-self
          returns the nodes in document order, as we desire. However, ancestor-or-self should return
          the nodes in reverse order. The "reverse" function must be applied after the xslt 
          processor fixes its bug.  Example shows function "reverse" is expected to be applied.
          
            reverse($elementToIdentify/ancestor-or-self::adfmf:*[not(self::adfmf:application)])
          -->
          <xsl:sequence 
              select="$elementToIdentify/ancestor-or-self::adfmf:*[not(self::adfmf:application)]"/>
        </xsl:when>
        <xsl:otherwise>
          <!--
          Get all ancestor-or-self elements in document order. The call to ancestor-or-self 
          correctly returns nodes in reverse document order. So the call to reverse puts them in 
          document order.
          -->
          <xsl:sequence select="reverse($elementToIdentify/ancestor-or-self::adfmf:*)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- Get the sequence of "id" attributes under the document node.
         These will be the tokens we use for building the identifer. -->
    <xsl:variable name="identifierTokens"
                  select="$parentPreferenceElementsAndSelf/child::adfmf:*/@id"/>

    <!-- Put the prefix element at the top of the sequence of identifier token elements -->
    <xsl:variable name="completeTokenSequence" select="($prefixToken, $identifierTokens)"/>
    
    <xsl:value-of select="string-join($completeTokenSequence,
                                      $g_TOKEN_DELIMITOR_DOT)"/>
  </xsl:function>

  <!-- *********************************************************************************************
       This function loads a localized XLIFF file if it exists.  If the
       file doesn't exist then an empty sequence is returned.
       Returns an XML document.
       param: p_oracleLocale - the XLIFF file locale suffix, which may be empty, or may specify just
                               a language code alone, or a language code followed by underscore and
                               a country code. For the definition of a valid Oracle locale,
                               see Java method "isValidOracleLocale" in class 
                               "oracle.adfmf.common.OracleLocale".
       param: p_baseXliffPath - a path to the base location of the XLIFF files.
       param: p_relativeXliffPathAndBaseFileName  - a relative path derived from loadBundle.baseName
       attribute.  This value contains the base name of the XLIFF file.
       
       return:  
       if p_oracleLocale is not null/empty:  a XLIFF document read from the location 
          $p_baseXliffPath/$p_relativeXliffPath_$p_oracleLocale.xlif
       if p_oracleLocale is not null/empty:  a XLIFF document read from the location 
          $p_baseXliffPath/$p_relativeXliffPath.xlif
  -->
  <xsl:function name="adfmf:getXliffDocument" as="document-node()">

    <xsl:param name="p_oracleLocale" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_relativeXliffPathAndBaseFileName" 
               as="xsl:string" 
               required="yes"/>

    <!-- Calculate XLIFF file URL for current language -->
    <xsl:variable name="localizedXliffFileName">
      <xsl:choose>
        <xsl:when test="$p_oracleLocale != ''">
        <xsl:value-of select="string-join(($p_relativeXliffPathAndBaseFileName, 
                                           $g_LANGUAGE_CODE_SEPARATOR, 
                                           $p_oracleLocale, 
                                           $g_XLIFF_FILE_EXTENSION), 
                                          '')"/>
        </xsl:when>
        <xsl:otherwise>
        <xsl:value-of select="string-join(($p_relativeXliffPathAndBaseFileName, 
                                           $g_XLIFF_FILE_EXTENSION), 
                                           '')"/>
        
        </xsl:otherwise>
    </xsl:choose>
    </xsl:variable>
    
    <xsl:variable name="languageXliffFileUrl" 
                  select="concat($p_baseXliffPath, $localizedXliffFileName)"/>

    <!-- Get localized XLIFF file if it exists. Otherwise, return nothing -->
   <xsl:if test="FileUtils:doesUrlResourceExist($languageXliffFileUrl) = true()">
      <xsl:sequence select="document($languageXliffFileUrl)"/>
    </xsl:if>
  </xsl:function>

  <!-- *********************************************************************************************
      This function determines if the given p_attribute belongs to a adfmf
      preference element.
      
      param:  p_attribute a xml attribute 
      return: true if the given attribute is associated with a preference.
              false otherwise.
   -->
   <xsl:function name="adfmf:isPreferenceAttribute" as="xsl:boolean">
      <xsl:param name="p_attribute" 
                 as="attribute()" 
                 required="yes"/>
        <xsl:choose>
          <xsl:when test="$p_attribute/ancestor-or-self::adfmf:preferenceGroup">
            <xsl:sequence select="true()"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:sequence select="false()"/>
          </xsl:otherwise>
       </xsl:choose>
   </xsl:function>
   
  <!-- *********************************************************************************************
      This function returns the value of the 'var' attribute of the 'loadBundle'
      element contained in adfmf-application.xml or adfmf-feature.xml
      param:  p_appOrFeatureElement either a adfmf:application or adfmf:features element
      return: the 'var' value of the 'loadBundle' element if it exists.  Empty
              string otherwise.
   -->
   <xsl:function name="adfmf:getLoadBundleVariableName" as="xsl:string">
     <xsl:param name="p_appOrFeatureElement" 
                as="element(adfmf:application) or element(adfmf:features)"/>
                
      <xsl:variable name="loadBundle" select="$p_appOrFeatureElement/adfmf:loadBundle"/>
      <xsl:variable name="loadBundleVarValue">
          <xsl:choose>
            <xsl:when test="$loadBundle">
              <xsl:value-of select="$loadBundle/@var"/>
            </xsl:when>

            <xsl:otherwise>
              <xsl:sequence select="''"/>
              </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:sequence select="$loadBundleVarValue"/>
    </xsl:function>
    
  <!-- *********************************************************************************************
    This function creates a resource string prefix of the form:
    feature.<featureId>.<bundleVarName> if the given p_featureId is non-empty.
    Otherwise, the returned string is of the form: application.<bundleVarName>
  -->
  <xsl:function name="adfmf:createResourceStringPrefix" as="xsl:string">
    <xsl:param name="p_adfmfBundleVariableName" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_featureId" 
               as="xsl:string" 
               required="yes"/>
   
      <xsl:variable name="resourceStringPrefix">
        <xsl:choose>
        <xsl:when test="$p_featureId and string-length ($p_featureId) > 0">
              <xsl:value-of select="string-join(($g_FEATURE_PREFIX, 
                                                 $p_featureId, 
                                                 $p_adfmfBundleVariableName), 
                                                 $g_STRING_RESOURCE_DELIMITOR_MIDDLE)"/>
          </xsl:when>
        <xsl:otherwise>
              <xsl:value-of select="concat($g_APPLICATION_PREFIX, 
                                           $g_STRING_RESOURCE_DELIMITOR_MIDDLE, 
                                           $p_adfmfBundleVariableName)"/>
          </xsl:otherwise>
      </xsl:choose>
      </xsl:variable>
    <xsl:value-of select="$resourceStringPrefix"/>
  </xsl:function>
  <!-- *********************************************************************************************
      This function determines if the given p_AppOrFeatureElem contains one or more 
      adfmf:preferences/adfmf:preferenceGroup elements.
      
      param:  p_AppOrFeatureElem a adfmf:application or adfmf:feature element
      return: true() if the given element contains one or more adfmf:preferences/
              adfmf:preferenceGroup elements. false() otherwise.
   -->
  <xsl:function name="adfmf:hasPreferenceElement" as="xsl:boolean">
    <xsl:param name="p_AppOrFeatureElem" 
               as="element (adfmf:application) or element (adfmf:feature)" 
               required="yes"/>
    <xsl:choose>
      <xsl:when test="$p_AppOrFeatureElem/adfmf:preferences/adfmf:preferenceGroup != ''">
        <xsl:sequence select="true()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:sequence select="false()"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
  
  <!-- *********************************************************************************************
      This function determines if the given p_applicationElement contains at least one 
      adfmf:preferences/adfmf:preferenceGroup element in the application or any of the 
      features imported into the application.
      
      param:  p_applicationElement a adfmf:application element
      param:  p_featureArchiveUrlNodeSet a document-node containing the URLs of the FARs imported to
              the application.
      return: true() if at least one adfmf:preferences/adfmf:preferenceGroup element
              was found.  false() otherwise.
   -->
  <xsl:function name="adfmf:applicationHasPreferenceElements" as="xsl:boolean">
    <xsl:param name="p_applicationElement" 
               as="element (adfmf:application)" 
               required="yes"/>
               
    <xsl:param name="p_featureArchiveUrlNodeSet" 
               as="document-node()" 
               required="yes"/>
    <!-- 
         Set to true if and only if a adfmf:preferenceGroup element is found.
    -->
    <xsl:variable name="hasPreferenceElement" as="xsl:boolean">
      <xsl:choose>
        <xsl:when test="adfmf:hasPreferenceElement ($p_applicationElement) = true()">
          <!-- adfmf-application.xml has preference elements -->
          <xsl:value-of select="true()"/>
        </xsl:when>
        <xsl:otherwise>
          <!-- Examine all the features in the FARs and build a variable 
               consisting of true or false strings (true if there is a 
               preferences element, false otherwise).  When done, if this 
               variable has 'true' in it, then some feature had a preference
               element.  This is a fairly simple XSLT solution to determine if
               any of the FARs have preferences.  A alternative, but more 
               complex solution, would be to use recursion instead of the 
               looping.
           -->
          <xsl:variable name="featurePreferenceResults">

            <!-- Loop through the FAR JAR URLs and load each adfmf-feature.xml document -->
            <xsl:for-each select="$p_featureArchiveUrlNodeSet">
              <xsl:variable name="featureArchiveUrl" select="." as="xsl:string"/>      
              <xsl:variable name="featureDoc" 
                            select="adfmf:loadFeatureDocument ($featureArchiveUrl)"/>

              <!-- Loop through the list of featureReference's contained in 
              adfmf-application.xml -->
              <xsl:for-each select="$gFeatureReferenceSet">
                
                <!-- featureId from featureReference element -->
                <xsl:variable name="featureId" select="@id"/>
                
                <!-- Loop through each of the features in the current feature document and,
                     if the feature document has the featureId, check to see if it contains
                     a preference element.
                -->
                <xsl:for-each select="$featureDoc/adfmf:features">
                  <xsl:variable name="featureElem" select="key('featureKey', $featureId)"/>
                  <xsl:choose>
                    <xsl:when test="$featureElem and adfmf:hasPreferenceElement 
                                    ($featureElem) = true()">
                      true
                    </xsl:when>
                    <xsl:otherwise>
                      false
                    </xsl:otherwise>
                  </xsl:choose>
                </xsl:for-each>
              </xsl:for-each>
            </xsl:for-each>
          </xsl:variable>
          <!-- If featurePreferenceResults contains 'true', then some feature had
               preferences.
          -->
          <xsl:value-of select="contains ($featurePreferenceResults, 'true')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    
    <xsl:sequence select="$hasPreferenceElement"/>
  </xsl:function>

  <!-- *********************************************************************************************
    This template recursively replaces all occurrences of p_searchString with p_replaceString 
    in the given p_sourceString.  This template is used because at the time of writing
    Oracle's XSLT processor:
    1) does not support XSLT2.0 replace() string function
    2) done not support function recursion
    This template can also be used when the XSLT translate() function does not
    meet the string replacement requirements - for example, translate() can only
    be used when replacing a single character with another single character or
    empty character.
    
    Note that this template is taken from book, "XSLT Cookbook", 2nd edition, 
    by Sal Mangano. Chapter 2.7, "Replacing Text" on page 41.

    If bug 14045664 (xdkj-xslt20:replace() function is not implemented correctly) is fixed, then 
    this template can be removed and clients of the template can be updated to use the Oracle XSLT 
    method.
  -->
  <xsl:template name="adfmf:string-replace-all">
    <xsl:param name="p_sourceString" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_searchString" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_replaceString" 
               as="xsl:string" 
               required="yes"/>
    <xsl:choose>
      <!-- See if the source string has the search string in it -->
      <xsl:when test="$p_sourceString and contains ($p_sourceString, $p_searchString) = true ()">
          <!-- 
            If so, output (order is important):
               1) the value of the source string up to the search string
               2) the replacement string
               3) the result of recursively calling this template with the
                  remaining string that after the search string.
          -->
          <xsl:value-of select="substring-before ($p_sourceString, $p_searchString)"/>
          
          <xsl:value-of select="$p_replaceString"/>
          
          <xsl:call-template name="adfmf:string-replace-all">
            <xsl:with-param name="p_sourceString" 
                            select="substring-after ($p_sourceString, $p_searchString)"/>
            <xsl:with-param name="p_searchString" select="$p_searchString"/>
            <xsl:with-param name="p_replaceString" select="$p_replaceString"/>
          </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <!-- No more occurrences of the search string found in the source string so
             output the current source string.  
        -->
        <xsl:value-of select="$p_sourceString"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  <!-- Copied from iOS XSL(s) -->
<!-- ***********************************************************************************************
  Get the content of the specified Expression Language string resource reference.
  This is accomplished by removing the surrounding delimiters #{} .
  Thus, the EL expression having pattern #{<BundleVarName><DECORATED_STRING_ID>}
  will return the following content:
  
     <BundleVarName><DECORATED_STRING_ID>
  
  See function "adfmf:getStringResourceId" for more information on
  the above string ID reference.
  
  TODO: This function is copied (without change) from iOS XSL file "PreferenceTransform.xsl"
        and also is duplicated in iOS XSL file "TransformStringResources.xsl".  So the iOS
        should be refactored to use this method if/when it is updated to use this 
        TransformUtilities.xsl file.
  -->
  <xsl:function name="adfmf:getExpressionLanguageContent" as="xsl:string">
    <!-- This is an ADFMF string resource identifier that
         must satisfy pattern "#{<BundleVarName><DECORATED_STRING_ID>}"
    -->
    <xsl:param name="adfmfStringResourceReference"
               as="xsl:string"
               required="yes"/>
    
    <!-- Remove prefix delimiter "#{" from the string resource reference -->
    <xsl:variable
      name="noPrefix"
      select="substring-after($adfmfStringResourceReference, $g_STRING_RESOURCE_DELIMITER_START)"
      as="xsl:string"
    />
    
    <!-- Remove suffix delimiter "}" from the string resource ID pattern -->
    <xsl:variable
      name="getExpressionLanguageContent"
      select="substring($noPrefix, 
                        1, 
                        string-length($noPrefix) - 
                         (string-length($g_STRING_RESOURCE_DELIMITER_END)))"
      as="xsl:string"
    />

    <xsl:value-of select="$getExpressionLanguageContent"/>
  </xsl:function>

  <!-- *********************************************************************************************
  Gets the value of attribute adfmf:loadBundle/@var directly from the EL expression
  of the specified attribute.  The EL expression must specify a string resource reference
  with the following format,
  
      #{<BundleVarName><DECORATED_STRING_ID>}
      
  where DECORATED_STRING_ID has one of the following patterns:
  
      .STRING_ID
      ['STRING_ID']

  TODO: This function is duplicated in file "TransformStringResources.xsl".
        So they should be refactored to use a single common method.
  -->
  <xsl:function name="adfmf:getLoadBundleVar" as="xsl:string">
    <!-- The value of this attribute must represent a string resource identifier that
         satisfies the following pattern that is described in the function comments.
         
            #{<BundleVarName><DECORATED_STRING_ID>}
    -->
    <xsl:param name="p_adfmfStringResourceReference"
               as="attribute(adfmf:*)"
               required="yes"/>

    <!-- Extract content of the form <BundleVarName><DECORATED_STRING_ID> by
         stripping outter prefix "${" and suffix "}".
    -->
    <xsl:variable
      name="expressionLanguageContent"
      select="adfmf:getExpressionLanguageContent($p_adfmfStringResourceReference)"
      as="xsl:string"/>

    <!-- Get the value of loadBundle/@var value from the EL expression content -->
    <xsl:choose>
      <xsl:when test="contains($expressionLanguageContent, 
                               $g_PREFIX_TO_STRING_ID_THAT_CONTAINS_DOTS)">
        <!--
        Now we know that the expression language content has the following pattern:
        
            bundle['STRING_ID']
              
        where STRING_ID contains dots.
        -->
        <xsl:sequence select="substring-before($expressionLanguageContent,
                                               $g_PREFIX_TO_STRING_ID_THAT_CONTAINS_DOTS)"/>
      </xsl:when>
      <xsl:otherwise>
        <!--
        Now we know that the expression language content has the following pattern:
        
            bundle.STRING_ID
              
        where STRING_ID does not contain dots.
        -->
      <xsl:value-of select="substring-before($expressionLanguageContent,
                                             $g_TOKEN_DELIMITOR_DOT)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>
 

  <!-- *********************************************************************************************
  Gets the naked resource string identifier (i.e. STRING_ID) of the XLIFF string resource
  identifier that is embedded in the EL expression of the specified attribute value.
  
  The specified attribute must satisfy the following pattern:
         
    #{<BundleVarName><DECORATED_STRING_ID>}
         
  <BundleVarName> is the value of attribute adfmf:loadBundle/@var from the same file
                  as the specified attribute.
         
  <DECORATED_STRING_ID> is one of the following:
         
      .STRING_ID   - where STRING_ID is an XLIFF resource string identifier that is
                     prefixed by a dot (i.e. ".") token delimeter, but doesn't
                     contain dot in the string identifier itself.
   or
      ['STRING_ID'] - where STRING_ID is an XLIFF resource string identifier that is
                     prefixed by a dot token delimeter, and does contains one or
                     more dots within the string identifier
                     Example: ['oracle.mobile.name']
  
  Given the above pattern, this function returns a string with value STRING_ID
  as defined above.

  TODO: This function is copied (without change) from iOS XSL file "PreferenceTransform.xsl"
        and also is duplicated in iOS XSL file "TransformStringResources.xsl".  So the iOS
        should be refactored to use this method if/when it is updated to use this 
        TransformUtilities.xsl file.
  -->
  <xsl:function name="adfmf:getStringResourceId" as="xsl:string">
    <!-- The value of this attribute must represent a string resource identifier that
         satisfies the following pattern as described in the function comments.
         
            #{<BundleVarName><DECORATED_STRING_ID>}
    -->
    <xsl:param name="p_adfmfStringResourceReference"
               as="attribute(adfmf:*)"
               required="yes"/>

    <xsl:variable name="loadBundleVar"
                  select="adfmf:getLoadBundleVar($p_adfmfStringResourceReference)"
                  as="xsl:string"/>
    
    <!-- Extract content of the form <BundleVarName><DECORATED_STRING_ID> by
         stripping outter prefix "${" and suffix "}".
    -->
    <xsl:variable
      name="expressionLanguageContent"
      select="adfmf:getExpressionLanguageContent($p_adfmfStringResourceReference)"
      as="xsl:string"/>

    <!--
    Remove the string bundle file prefix. The following table shows two source
    patterns and the corresponding result we want after removing the bundle prefix.
    
                          Result String
    Original String       (documented earlier as <DECORATED_STRING_ID>)
    *****************     **********************************************
    bundle.STRING_ID      .STRING_ID
    bundle['STRING_ID']   ['STRING_ID']
    -->
    <xsl:variable
      name="decoratedXliffStringId"
      select="substring-after($expressionLanguageContent, $loadBundleVar)"
      as="xsl:string"
    />

    <!--
    Parse to get the naked XLIFF string identifier. This is the
    same ID as what appears in an XLIFF file.  The following table
    shows two source string and the corresponding result we want.
         
    Original String       Result String
    *****************     **********************************************         
    .STRING_ID            STRING_ID
    ['STRING_ID']         STRING_ID
    -->
    <xsl:variable name="nakedStringIdentifier" as="xsl:string">
      <xsl:choose>
        <xsl:when test="starts-with($decoratedXliffStringId, $g_TOKEN_DELIMITOR_DOT)">
          <!--
          A dot delimiter prefixes the XLIFF string identifier token. So we have
          the following pattern.
          
             .STRING_ID
          
          So remove the dot prefix to get the XLIFF string identifier by itself.
          As a corollary, this pattern tells us that the XLIFF string identifier does
          not contain embedded dots, although that does not make a difference for our use-case.
          -->
          <xsl:sequence select="substring-after($decoratedXliffStringId, $g_TOKEN_DELIMITOR_DOT)"/>
        </xsl:when>
        <xsl:otherwise>
          <!--
          We have the following pattern.
          
             ['STRING_ID']

          We need to extract the STRING_ID from within the surrounding
          brackets and single-quotes.
            
          This pattern tells us that STRING_ID represents a string ID
          that contains dots, although that does not make a difference
          for our use-case.
          
          For example, suppose we have the following decorated XLIFF string ID:
          
            ['city.employee.middle.name']
            
          then the STRING_ID component is:
          
            city.employee.middle.name
          -->

          <!-- Remove prefix [' -->
          <xsl:variable name="stringIdWithSuffix"
                        select="substring-after($decoratedXliffStringId,
                                                $g_PREFIX_TO_STRING_ID_THAT_CONTAINS_DOTS)"
                        as="xsl:string"/>

          <!-- Remove suffix '] to finally get the XLIFF resource string identifier.  -->
          <xsl:sequence select="substring-before($stringIdWithSuffix,
                                                 $g_SUFFIX_TO_STRING_ID_THAT_CONTAINS_DOTS)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:value-of select="$nakedStringIdentifier"/>
  </xsl:function>  
  
<!-- *********************************************************************************************
  Returns the resource text that corresponds to the specified string resource identifier
  in the  specified XLIFF document.  For the specified string resource identifier, this
  method returns the value of element <xlf:target> if it exists and specifies a non-empty
  string. If <xlf:target> does not exist or is empty then the value of element <xlf:source>
  is returned if it exists and specifies a non-empty string. If the string identifier is not
  found or both elements <xlf:target> and <xlf:source> are not found or are empty then this
  function does not return a string.
  
  If the specified XLIFF document is a localized XLIFF document then the string may or may not be 
  found.  However, if the specified XLIFF document is a base (non-localized) XLIFF document, then 
  the string should always be found because a base XLIFF document is supposed to always have all 
  resource strings whether or not they are localized.
  -->
  <xsl:function name="adfmf:getResourceTextFromTargetOrSource" as="xsl:string">
    <!-- Either a base XLIff document or a localized XLIFF document. -->
    <xsl:param name="p_xliffDocument"
               required="yes"
               as="document-node()"/>

    <!-- String resource identifier that will be searched for in the specified XLIFF document. -->
    <xsl:param name="p_stringResourceId"
               as="xsl:string"
               required="yes"/>

    <xsl:for-each select="$p_xliffDocument/xlf:xliff/xlf:file/xlf:body">

      <!-- This loop only enters once.  The above for loop is only needed to
           change context node for key(). No actual looping is needed. -->
      <xsl:variable name="transUnitElement" select="key('xliffStringKey', $p_stringResourceId)"/>
      
      <!-- Element <xlf:source> of the specified string resource. -->
      <xsl:variable name="sourceElement" select="$transUnitElement/xlf:source"/>
      
      <!-- Element <xlf:target> of the specified string resource. -->
      <xsl:variable name="targetElement" select="$transUnitElement/xlf:target"/>
      
      <xsl:choose>
        <xsl:when test="$targetElement and (string-length($targetElement) &gt; 0)">
          <!-- Control gets here iff element <xlf:target> exists with a non-empty string -->
          <xsl:value-of select="$targetElement"/>
        </xsl:when>
        <xsl:when test="$sourceElement and (string-length($sourceElement) &gt; 0)">
          <!-- Control gets here iff element <xlf:source> exists with a non-empty string -->
          <xsl:value-of select="$sourceElement"/>
        </xsl:when>
      </xsl:choose>      
    </xsl:for-each>
  </xsl:function>  
  <!-- *********************************************************************************************
  Returns the resource text that corresponds to the specified string resource identifier.
  First try to get the resource string from the localized XLIFF document.  If the localized XLIFF
  file does not exist or doesn't contain the resource string, then retrieve the resource
  string from the base XLIFF document.
  -->
  <xsl:function name="adfmf:getResourceText" as="xsl:string">
    <!-- Specifies a string resource identifier. -->
    <xsl:param name="p_stringResourceId"
               required="yes"
               as="xsl:string"/>

    <!-- The base (non-localized) XLIFF file must always exist. -->
    <xsl:param name="p_baseXliffDocument"
               required="yes"
               as="document-node()"/>

    <!-- The localized XLIFF file may or may not exist. -->
    <xsl:param name="p_localizedXliffFileDocument"
               required="no"
               as="document-node()"/>

    <!-- Get resource text from localized XLIFF document if possible. -->
    <xsl:variable name="localizedResourceString" as="xsl:string">
      <xsl:if test="exists($p_localizedXliffFileDocument/*)">
        <xsl:value-of select="adfmf:getResourceTextFromTargetOrSource($p_localizedXliffFileDocument, 
                                                                      $p_stringResourceId)"/>
      </xsl:if>
    </xsl:variable>

    <xsl:choose>
      <xsl:when test="(string-length($localizedResourceString) &gt; 0)">
        <!-- Return localized string -->
        <xsl:value-of select="$localizedResourceString"/>
      </xsl:when>
      <xsl:otherwise>
        <!-- Get resource text from base XLIFF document. -->
        <xsl:value-of select="adfmf:getResourceTextFromTargetOrSource($p_baseXliffDocument, 
                                                                      $p_stringResourceId)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:function>  

  <!-- *********************************************************************************************
       This function loads a XLIFF file, if it exists.  If the file doesn't exist then an 
       empty sequence is returned.
       
       param: p_appOrFeatureElement - a adfmf:application or adfmf:feature element
       param: p_baseXliffPath  - a path to the base location of the XLIFF files.
       param: p_attributeValue - a value of a attribute containing a EL-expression reference to a
                                 string in a XLIFF file.
       param: p_oracleLocale   - the XLIFF file locale suffix, which may be empty, or may specify 
                                 just a language code alone, or a language code followed by 
                                 underscore and a country code. For the definition of a valid Oracle 
                                 locale, see Java method "isValidOracleLocale" in class 
                                 "oracle.adfmf.common.OracleLocale".
       return:  
       
        if p_oracleLocale is not null/empty:  a XLIFF document read from the location 
          $p_baseXliffPath/<relativeXLIFFPath>_$p_oracleLocale.xlif
          
        if p_oracleLocale is null/empty:  a XLIFF document read from the location 
          $p_baseXliffPath/<relativeXLIFFPath>.xlif
          
        where <relativeXLIFFPath> is determined by using the adfmf:loadBundle/@basename attribute 
        whose adfmf:loadBundle/@var attribute value matches the value contained in the given 
        $p_attributeValue.  For example, if:
          1) p_oracleLocale is null/empty 
          2) $p_appOrFeatureElement has the following loadBundle elements:
             <adfmf:loadBundle basename='mobile.foo' var='vcBundle1'/>
             <adfmf:loadBundle basename='mobile.bar' var='vcBundle2'/>
          3) $p_attributeValue was '#{vcBundle2.SOME_ID_FROM_VIEW_CONTROLLER_BUNDLE2}',
          
        then the <relativeXLIFFPath> would be generated using the second <adfmf:loadBundle> and
        would be "mobile/bar" and the .xlf file loaded would be $p_baseXliffPath/mobile/bar.xlf
   -->
  <xsl:function name="adfmf:getXliffDocumentFromAttribute" as="document-node()">
    
    <xsl:param name="p_appOrFeatureElement" 
               as="element (adfmf:application) or element (adfmf:feature)" 
               required="yes"/>
               
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string"
               required="yes"/>
               
    <xsl:param name="p_attributeValue" 
               as="xsl:string"
               required="yes" />
               
    <xsl:param name="p_oracleLocale" 
               as="xsl:string" 
               required="no" />

    <!-- Get the loadBundle/@var value directly from the attribute value -->
    <xsl:variable name="loadBundleVar" select="adfmf:getLoadBundleVar ($p_attributeValue)"/>
    
    <!-- Get the basename attribute from the adfmf:loadBundle element whose var attribute value
         matches the value obtained from the attribute -->
    <xsl:variable name="baseNameValue">

      <xsl:choose>
        
        <xsl:when test="$p_appOrFeatureElement/ancestor-or-self::adfmf:application">
          <!-- adfmf:loadBundle elements are child elements of adfmf:application -->
          <xsl:value-of 
                   select="$p_appOrFeatureElement/adfmf:loadBundle[@var=$loadBundleVar]/@basename"/>
        </xsl:when>
        
        <xsl:otherwise>
           <!-- adfmf:loadBundle elements are child elements of adfmf:features element. Since
                p_appOrFeatureElement is a adfmf:feature, locate the adfmf:loadBundle using the 
                parent -->
          <xsl:value-of 
                select="$p_appOrFeatureElement/../adfmf:loadBundle[@var=$loadBundleVar]/@basename"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    
    <!-- Convert the basename to a relative path by replacing all '.'s with '/'s -->
    <xsl:variable name="relativeXliffPath" 
                  select="translate ($baseNameValue,
                                     $g_STRING_RESOURCE_DELIMITOR_MIDDLE, 
                                     $g_FILE_PATH_SEPARATOR)"/>
    
    <!-- Load the .xlf file -->
    <xsl:sequence select="adfmf:getXliffDocument ($p_oracleLocale, 
                                                  $p_baseXliffPath, 
                                                  $relativeXliffPath)"/>
  </xsl:function>  
  
  <!-- *********************************************************************************************
    Returns the resource string whose XLIFF id is contained in the given p_xliffStringReferenceValue
    
    If the resource string is contained in a localized XLIFF file, it is returned.  If the 
    localized XLIFF file does not exist or doesn't contain the resource string, then the resource
    string from the base XLIFF document is returned.

    param:  p_appOrFeatureElement a adfmf:application or adfmf:feature element 
    param:  p_baseXliffPath a relative path to the location of a .xlf file
    param:  p_xliffStringReferenceValue a full EL-expression reference to a string in a .xlf file 
    param:  p_oracleLocale a oracle locale.  May be empty/null
    
    return: A string from a .xlf file
   -->
  <xsl:function name="adfmf:getStringFromXliffFile" as="xsl:string">
    <xsl:param name="p_appOrFeatureElement" 
               as="element (adfmf:application) or element (adfmf:feature)" 
               required="yes"/>
               
    <xsl:param name="p_baseXliffPath" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_xliffStringReferenceValue" 
               as="xsl:string" 
               required="yes"/>
               
    <xsl:param name="p_oracleLocale" 
               as="xsl:string" 
               required="no"/>

    <xsl:variable name="baseDoc" 
                  as="document-node()"
                  select="adfmf:getXliffDocumentFromAttribute ($p_appOrFeatureElement,
                                                               $p_baseXliffPath,
                                                               $p_xliffStringReferenceValue,
                                                               '')"/>
                                                                                        
    <xsl:variable name="localizedDoc" 
                  as="document-node()"    
                  select="adfmf:getXliffDocumentFromAttribute ($p_appOrFeatureElement,
                                                               $p_baseXliffPath,
                                                               $p_xliffStringReferenceValue,
                                                               $p_oracleLocale)"/>

    <xsl:variable name="xliffId" 
                  as="xsl:string" 
                  select="adfmf:getStringResourceId($p_xliffStringReferenceValue)"/>
                                                                                  
    <xsl:variable name="xliffResource" 
                  select="adfmf:getResourceText ($xliffId, 
                                                 $baseDoc, 
                                                 $localizedDoc)"/>
    <xsl:value-of select="$xliffResource"/>
    
  </xsl:function>
  

</xsl:stylesheet>
