package oracle.adf.model.datacontrols.device;

import javax.el.MethodExpression;

import oracle.adfmf.framework.EmbeddedFeatureContext;
import oracle.adf.model.datacontrols.device.Location;
import oracle.adfmf.framework.exception.AdfException;
import oracle.adfmf.framework.model.AdfELContext;
import oracle.adfmf.resource.CDCErrorBundle;
import oracle.adfmf.util.ResourceBundleHelper;
import oracle.adfmf.util.Utility;

/**
 * Master device data control class. Services such as Device Properties,
 * Geolocation, and Camera are encapsulated in this class. This is also
 * used directly by JDeveloper to expose these services at design time
 * in the DataControl palette. 
 */
public class DeviceDataControl
{
  private static long locationWatchId              = 1;

  private static String DDCM_PREFIX                = "DDCM-";
  // Auto-generated watchIDs for watchPosition requests initiated by device data control
  private static String DDCM_LOCATION_WATCH_PREFIX = DDCM_PREFIX + "LocationWatch-";
  
  /**
   * Register a callback to receive regular updates on the device's position
   * 
   * @param enableHighAccuracy
   *          - provides a hint that the application would like to receive the
   *          best possible results (boolean)
   * @param updateInterval
   *          - defines how often to retrieve the position in milliseconds.
   * @param locationListener
   *          - The name of a bean method that will be invoked in response to a
   *          location update, e.g. "MyBean.myMethod". Do not use EL syntax,
   *          unless you truly want the result of evaluating that expression to
   *          become the name of the locationListener
   * 
   */
  public void startLocationMonitor(boolean enableHighAccuracy, int updateInterval, String locationListener)
  {
    String watchId = DDCM_LOCATION_WATCH_PREFIX + locationWatchId++;
    DeviceManagerFactory.getDeviceManager().startUpdatingPosition(updateInterval, enableHighAccuracy, watchId,
      new MethodExpressionResolvingGeolocationCallback(locationListener));
  }

  /**
   * Save a previously-created Contact to the device's address book. Note that
   * you can create a Contact by simply instantiating one and setting its
   * properties, but in order to persist it, you must call createContact.
   * 
   * See PhoneGap documentation for more details:
   * http://docs.phonegap.com/en/1.0.0/phonegap_contacts_contacts.md.html#Contacts
   * 
   * @param aContact
   *          - a previously-constructed Contact object
   * @return a Contact object. Note that you should always use the returned
   *         Contact object in subsequent operations, because this object will
   *         have the requisite primary key
   *         
   */
  public Contact createContact(Contact aContact)
  {
    return DeviceManagerFactory.getDeviceManager().createContact(aContact);
  }
  
  /**
   * Update an existing Contact object. Note that you can modify a Contact via
   * its various properties and methods, but in order to persist the changes,
   * you must call updateContact
   * 
   * See PhoneGap documentation for more details:
   * http://docs.phonegap.com/en/1.0.0/phonegap_contacts_contacts.md.html#Contacts
   * 
   * @param aContact
   *          - the modified Contact object to persist in the device's address
   *          book.
   * @return - the modified Contact object
   */
  public Contact updateContact(Contact aContact)
  {
    return DeviceManagerFactory.getDeviceManager().updateContact(aContact);
  }
  
  /**
   * Remove a contact from the address book
   * 
   * See PhoneGap documentation for more details:
   * http://docs.phonegap.com/en/1.0.0/phonegap_contacts_contacts.md.html#Contacts
   * 
   * @param aContact - the Contact object to remove
   */
  public void removeContact(Contact aContact)
  {
    DeviceManager ddcm = DeviceManagerFactory.getDeviceManager();

    if (ddcm == null)
    {
      throw new AdfException(AdfException.ERROR, 
        ResourceBundleHelper.CDC_ERROR_BUNDLE, CDCErrorBundle.ERR_DDC_MANAGER_NOT_FOUND, null);
    }

    ddcm.removeContact(aContact);
  }
  
  /**
   * Enables retrieval of one or more Contact objects matching the specified
   * criteria.
   * 
   * See PhoneGap documentation for more details:
   * http://docs.phonegap.com/en/1.0.0/phonegap_contacts_contacts.md.html#Contacts
   * 
   * @param contactFields
   *          - comma-delimited list of field names to match on, e.g.
   *          "name,address"
   * @param filter
   *          - search criteria. Note that the search criteria is matched
   *          against all fields specified in contactFields, so if you specify
   *          "John", a Contact object will be matched if any of the fields
   *          specified in contactFields (e.g. name OR address OR phonenumber)
   *          contains the filter string
   * 
   * @return an array of Contact objects. Note that each Contact object will
   *         ONLY contain the fields you specified in the contactFields
   *         parameter; all others will be empty/null.
   * 
   */
  public Contact[] findContacts(String contactFields, String filter, boolean multiple)
  {
    DeviceManager ddcm = DeviceManagerFactory.getDeviceManager();
    Contact[] foundContacts = null;

    if (ddcm == null)
    {
      throw new AdfException(AdfException.ERROR, 
        ResourceBundleHelper.CDC_ERROR_BUNDLE, CDCErrorBundle.ERR_DDC_MANAGER_NOT_FOUND, null);
    }

    foundContacts = ddcm.findContacts(contactFields, filter, multiple);

    return foundContacts;
  }
  
  /**
   * Display the operating system's default UI for sending a text message, and optionally pre-populate certain fields
   * 
   * @param to
   *          - recipient list (comma-separated)
   * @param body
   *          - text of message
   */
  public void sendSMS(String to, String body)
  {
    DeviceManagerFactory.getDeviceManager().sendSMS(to, body);
  }

  /**
   * Display the operating system's default UI for sending an e-mail, and optionally pre-populate certain fields.
   * 
   * @param to
   *          - recipient list (comma-separated)
   * @param cc
   *          - carbon copy recipient list (comma-separated)
   * @param subject
   *          - subject line
   * @param body
   *          - text of e-mail
   * @param bcc
   *          - blind carbon copy recipient list (comma-separated)
   * @param attachments
   *          - list of files to attach (comma-separated). Must be specified as
   *          absolute filenames; use AdfmfJavaUtilities.getDirectoryPathRoot()
   *          to help determine absolute filename.
   * @param mimeTypes
   *          - list of MIME types, specified in the same order as the list of
   *          attachments. Specify null to let the system automatically
   *          determine MIME types for all attachments, or leave blanks in the
   *          comma-separated list to indicate the system should determine the
   *          MIME type only for those selected attachments. Example:
   *          attachments = "www/foo.txt,www/bar.png", mimetypes = "text/plain,"
   *          means the first attachment is of type text/plain, but the system
   *          should automatically determine the MIME type for the second one.
   */
  public void sendEmail(String to, String cc, String subject, String body, String bcc, String attachments, String mimeTypes)
  {
    DeviceManagerFactory.getDeviceManager().sendEmail(to, cc, subject, body, bcc, attachments, mimeTypes);
  }

  /**
   * Provides access to the device's default camera application, which enables
   * taking a picture or retrieving a previously-saved image
   * 
   * See PhoneGap documentation for more details: 
   * http://docs.phonegap.com/en/1.0.0/phonegap_camera_camera.md.html#camera.getPicture
   * 
   * @param quality
   *          - Quality of saved image. Range is [0, 100]
   * @param destinationType
   *          - Choose the format of the return value. 
   *            0 = DESTINATIONTYPE_DATA_URL = Return image as base64 encoded string
   *            1 = DESTINATIONTYPE_FILE_URI = Return image as a filename URI, like "file:///absolute/path/to/image.png"
   *            DESTINATIONTYPE_FILE_URI is recommended to avoid exhausting device resources.
   * @param sourceType
   *          - Where should the picture come from? 
   *            0 = SOURCETYPE_PHOTOLIBRARY = Device's photo library
   *            1 = SOURCETYPE_CAMERA = Take a picture with the device's camera
   *            2 = SOURCETYPE_SAVEDPHOTOALBUM = Device's saved photo album
   * @param allowEdit
   *          - Allow simple editing of image before selection
   * @param encodingType
   *          - Choose the encoding of the returned image file.
   *            0 = ENCODING_TYPE_JPEG = JPEG image
   *            1 = ENCODING_TYPE_PNG = PNG image
   * @param targetWidth
   *          - Width in pixels to scale image. Must be used with targetHeight.
   *          Aspect ratio is maintained
   * @param targetHeight
   *          - Height in pixels to scale image. Must be used with targetWidth.
   *          Aspect ratio is maintained
   *          
   *  
   * @return a String representing either base64 image data or a file URI, depending on the value of destinationType
   */
  public String getPicture(int quality, int destinationType, int sourceType, boolean allowEdit, int encodingType,
    int targetWidth, int targetHeight)
  {
    return DeviceManagerFactory.getDeviceManager().getPicture(quality, destinationType, sourceType, allowEdit, encodingType,
      targetWidth, targetHeight);
  }
  
  public void displayFile(String fileURL, String headerText){
    DeviceManagerFactory.getDeviceManager().displayFile(fileURL, headerText);
  }

  /**
   * This class is used internally to indirectly invoke application developer's
   * callback by evaluating the provided MethodExpression. This
   * MethodExpression will be evaluated each time the device receives a location
   * update.
   * 
   */
  private class MethodExpressionResolvingGeolocationCallback
    implements GeolocationCallback
  {
    private String elString;

    private MethodExpressionResolvingGeolocationCallback(String elString)
    {
      if (elString != null)
      {
        if (!elString.startsWith("#{"))
          elString = "#{" + elString;
        if (!elString.endsWith("}"))
          elString = elString + "}";
      }
      this.elString = elString;
    }

    public void locationUpdated(Location position)
    {
      // This is valid; it just means "listen for updates, but I don't want to
      // do anything beyond simply
      // updating the corresponding variable in the DeviceScope context"
      if (Utility.isEmpty(elString))
        return;

      AdfELContext elContext = EmbeddedFeatureContext.getInstance().getAdfELContext();
      MethodExpression methodExpression = elContext.getExpressionFactory().createMethodExpression(elContext, elString,
        void.class, new Class[] { Location.class });
      methodExpression.invoke(elContext, new Object[] { position });
    }
  }
}
