/************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * Use is subject to license terms.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0. You can also
 * obtain a copy of the License at http://odftoolkit.org/docs/license.txt
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * 
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ************************************************************************/

package org.odftoolkit.odfdom.dom.type;

import org.odftoolkit.odfdom.dom.type.OdfMeasure;
import org.odftoolkit.odfdom.dom.util.OdfUnitConverter;

/**
 * DOM implementation of OpenDocument data type 'lengthOrPercent'
 *  This value type stores either a measure value in 1/100th mm as int or a
 *  percent value as double.
 */
public class OdfLengthOrPercent
{
    private Object mValue;
    
    private String lengtheOrPercent;
    
    /**
     * Construct OdfLengthOrPercent without the initialized value
     */
    public OdfLengthOrPercent()
    {        
        mValue = null;
    }
   
    /**
     * Copy Construct OdfLengthOrPercent with the same value of val
     * 
     * @param val the copied OdfLengthOrPercent object
     */
    public OdfLengthOrPercent(OdfLengthOrPercent val)
    {        
        if( val == null )
            throw new IllegalArgumentException( "Illegal length or percent value");
        
        if( val.isPercent() )
        	lengtheOrPercent = String.valueOf( val.getPercent() ) + "%";
        else
        	lengtheOrPercent = OdfUnitConverter.getMeasureString(val.getLength());
    }
    
    /**
     * set Length value for OdfLengthOrPercent Object 
     * 
     * @param length the length value which is set to OdfLengthOrPercent Object
     */
    public void setLength( int length )
    {
        mValue = new Integer(length);
    }
    
    /**
     * set Percent value for OdfLengthOrPercent Object
     * 
     * @param percent the percent value which is set to OdfLengthOrPercent Object
     */
    public void setPercent( double percent )
    {
        mValue = new Double(percent);
    }
    
    /**
     * get the internal value type of OdfLengthOrPercent Object
     * 
     * @return true if the internal value type is Percent
     * 		   false if the internal value type is Length
     */
    public boolean isPercent()
    {
        return (mValue != null) && mValue instanceof Double;
    }
    /**
     * get the Length value of this OdfLengthOrPercent Object
     * 
     * @return the Length value of this OdfLengthOrPercent Object
     */
    public int getLength()
    {
        if( isPercent() )
        {
            throw new IllegalArgumentException("I do not have a length value");
        }
        else if( mValue != null )
        {
            return ((Integer)mValue).intValue();
        }
        else
        {
            return 0;
        }
    }
    
    /**
     * get the Percent value of this OdfLengthOrPercent Object
     * 
     * @return the Percent value of this OdfLengthOrPercent Object
     */
    public double getPercent()
    {
        if( isPercent() )
        {
            return ((Double)mValue).doubleValue();
        }
        else
        {
            throw new IllegalArgumentException("I do not have a percent value");
        }
    }
    
    /**
	 * Returns a String Object representing this OdfLengthOrPercent's value
	 * 
	 * @return return a string representation of the value of this OdfLengthOrPercent object
	 */
    public String toString( )
    {
    	return lengtheOrPercent;
    }

    /**
     * Returns an OdfLengthOrPercent instance representing the specified String value
     * 
     * @param stringValue a String value
     * @return return an OdfLengthOrPercent instance representing stringValue
     */
    public static OdfLengthOrPercent valueOf( String stringValue )
    {
        OdfLengthOrPercent ret = new OdfLengthOrPercent();
        int n = stringValue.indexOf( "%" );
	if( n != -1 )
            ret.setPercent( OdfPercent.valueOf(stringValue).doubleValue());
        else
            ret.setLength(OdfMeasure.valueOf(stringValue).intValue());
        return ret;
    }
}
